/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam.reader;

import org.broad.igv.sam.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.util.AsciiLineReader;
import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;

/**
 * A wrapper for SamTextReader that supports query by interval.
 *
 * @author jrobinso
 */
public class GeraldQueryReader implements AlignmentQueryReader {

    private static Logger log = Logger.getLogger(GeraldQueryReader.class);
    static int MAX_READ_LENGTH = 100;
    static int maxTileCount = 20;
    File alignmentFile;
    SamIndex samIndex;
    FileInputStream is;
    AlignmentParser parser;

    public GeraldQueryReader(File alignmentFile) {
        this.alignmentFile = alignmentFile;
        parser = getParserFor(alignmentFile);
        samIndex = SamUtils.getIndexFor(alignmentFile);
        try {
            is = new FileInputStream(alignmentFile);
        } catch (FileNotFoundException fileNotFoundException) {
            fileNotFoundException.printStackTrace();
        }
    }

    public GeraldQueryReader(File samFile, File indexFile) {
        this.alignmentFile = samFile;
        if (indexFile.exists()) {
            samIndex = new SamIndex(indexFile);
        }
    }

    private static AlignmentParser getParserFor(File file) {
        if (file.getName().endsWith(".aligned")) {
            return new DotAlignedParser();
        } else {
            return new GeraldParser();
        }
    }

    public SAMFileHeader getHeader() {
        return null;
    }

    public CloseableIterator<Alignment> query(final String sequence, final int start, final int end, final boolean contained) {

        if (samIndex == null) {
            throw new java.lang.UnsupportedOperationException("SAM files must be indexed to support query methods");
        }
        if (!samIndex.containsChromosome(sequence))  {
            return EmptyAlignmentIterator.getInstance();
        }

        // If contained == false (include overlaps) we need to adjust the start to
        // ensure we get features that extend into this segment.
        int startAdjustment = contained ? 0 : samIndex.getLongestFeature(sequence);
        return new GeraldQueryIterator(sequence, start - startAdjustment, end, contained);

    }

    public void close() throws IOException {
        if (is != null) {
            is.close();
        }
    }

    /**
     *
     */
    class GeraldQueryIterator implements CloseableIterator<Alignment> {

        String chr;
        int start;
        int end;
        boolean contained;
        Alignment currentRecord;
        AsciiLineReader reader;

        public GeraldQueryIterator(String sequence, int start, int end, boolean contained) {
            this.chr = sequence;
            this.start = start;
            this.end = end;
            this.contained = contained;
            seekToStart();
            reader = new AsciiLineReader(is);
            advanceToFirstRecord();
        }

        private Alignment readNextRecord() {
            currentRecord = parser.readNextRecord(reader);
            return currentRecord;
        }

        private void advanceToFirstRecord() {
            while ((readNextRecord()) != null) {
                if (!currentRecord.getChromosome().equals(chr)) {
                    break;
                } else if ((contained && currentRecord.getStart() >= start) ||
                        (!contained && currentRecord.getEnd() >= start)) {
                    break;
                }
            }
        }

        public void close() {
            try {
                is.close();
            } catch (IOException ex) {
                log.error("Error closing alignment file", ex);
            }
        }

        public boolean hasNext() {
            if (currentRecord == null || !chr.equals(currentRecord.getChromosome())) {
                return false;
            } else {
                return contained ? currentRecord.getEnd() <= end
                        : currentRecord.getStart() <= end;
            }
        }

        public Alignment next() {
            Alignment ret = currentRecord;
            readNextRecord();
            return ret;

        }

        public void remove() {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        private void seekToStart() {

            if (samIndex == null) {
                throw new java.lang.UnsupportedOperationException("SAM files must be indexed to support query methods");
            }

            // If contained == false (include overlaps) we need to adjust the start to
            // ensure we get features that extend into this segment.
            int startAdjustment = contained ? 0 : MAX_READ_LENGTH; //samIndex.getLongestFeature(chr);
            int startTileNumber = Math.max(0, (start - startAdjustment)) / samIndex.getTileWidth();

            SamIndex.TileDef seekPos = samIndex.getTileDef(chr, startTileNumber);
            long startPosition = seekPos == null ? 0 : seekPos.getStartPosition();

            try {
                // Skip to the start of the chromosome (approximate)
                is = new FileInputStream(alignmentFile);
                is.getChannel().position(startPosition);

            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }
    }
}
