/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam.reader;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.sam.*;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.util.CloseableIterator;

import org.broad.igv.util.LRUCache;

//~--- JDK imports ------------------------------------------------------------

import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.ui.IGVMainFrame;

/**
 * A wrapper for SamTextReader that supports query by interval.
 *
 * @author jrobinso
 */
public class CachingQueryReader implements AlignmentQueryReader {

    private static Logger log = Logger.getLogger(CachingQueryReader.class);

    String cachedChr = "";
    static int maxTileCount = 30;
    static int tileSize = 8000;
    AlignmentQueryReader reader;
    LRUCache<Integer, Tile> cache;

    public CachingQueryReader(AlignmentQueryReader reader) {
        this.reader = reader;
        cache = new LRUCache(maxTileCount);
    }

    public void close() throws IOException {
        reader.close();
    }

    public SAMFileHeader getHeader() {
        return reader.getHeader();
    }


    public CloseableIterator<Alignment> query(String sequence, int start, int end, boolean contained) {

        int startTile = (start + 1) / tileSize;
        int endTile = end / tileSize;    // <= inclusive
        List<Tile> tiles = getTiles(sequence, startTile, endTile);

        if (tiles.size() == 0) {
            return EmptyAlignmentIterator.getInstance();
        }

        // Count total # of records
        int recordCount = tiles.get(0).getOverlappingRecords().size();
        for (Tile t : tiles) {
            recordCount += t.getContainedRecords().size();
        }

        List<Alignment> alignments = new ArrayList(recordCount);
        alignments.addAll(tiles.get(0).getOverlappingRecords());
        for (Tile t : tiles) {
            alignments.addAll(t.getContainedRecords());
        }
        return new TiledIterator(start, end, alignments);
    }

    private List<Tile> getTiles(String seq, int startTile, int endTile) {

        if(!seq.equals(cachedChr)) {
            cache  = new LRUCache(maxTileCount);
            cachedChr = seq;
        }


        List<Tile> tiles = new ArrayList(endTile - startTile + 1);
        List<Tile> tilesToLoad = new ArrayList(endTile - startTile + 1);

        for (int t = startTile; t <= endTile; t++) {
            Tile tile = cache.get(t);

            if (tile == null) {
                int start = t * tileSize;
                int end = start + tileSize;

                tile = new Tile(t, start, end);
                cache.put(t, tile);
            }

            tiles.add(tile);

            // The current tile is loaded,  load any preceding tiles we have pending
            if (tile.isLoaded()) {
                if (tilesToLoad.size() > 0) {
                    loadTiles(seq, tilesToLoad);
                }
                tilesToLoad.clear();
            } else {
                tilesToLoad.add(tile);
            }
        }

        if (tilesToLoad.size() > 0) {
            loadTiles(seq, tilesToLoad);
        }

        return tiles;
    }

    private void loadTiles(String seq, List<Tile> tiles) {
  
        int start = tiles.get(0).start;
        int end = tiles.get(tiles.size() - 1).end;
        CloseableIterator<Alignment> iter = null;

        //log.debug("Loading : " + start + " - " + end);
        int alignmentCount = 0;
        //long t0 = System.currentTimeMillis();
        try {
            iter = reader.query(seq, start, end, false);

            while (iter.hasNext()) {
                Alignment record = iter.next();

                // Range of tile indeces that this alignment contributes to.
                int aStart = record.getAlignmentStart();
                int aEnd = record.getEnd();
                int idx0 = Math.max(0, (aStart - start) / tileSize);
                int idx1 = Math.min(tiles.size() - 1, (record.getEnd() - start) / tileSize);

                // Loop over tiles this read overlaps
                for (int i = idx0; i <= idx1; i++) {
                    Tile t = tiles.get(i);

                    if ((aStart >= t.start) && (aStart < t.end)) {
                        t.containedRecords.add(record);
                    } else if ((aEnd >= t.start) && (aStart < t.start)) {
                        t.overlappingRecords.add(record);
                    }
                }

                alignmentCount++;
                if (alignmentCount % 5000 == 0) {
                    IGVMainFrame.getInstance().setStatusBarMessage("Reads loaded: " + alignmentCount);
               }
            }

            for (Tile t : tiles) {
                t.setLoaded(true);
            }

            //double dt = (System.currentTimeMillis() - t0) / 1000.0;
            //log.debug("Loaded " + alignmentCount + " in " + dt);

        } finally {
            if (iter != null) {
                iter.close();
            }
            IGVMainFrame.getInstance().resetStatusMessage();
        }
    }

    static class Tile {

        private boolean loaded = false;
        private List<Alignment> containedRecords;
        private int end;
        private List<Alignment> overlappingRecords;
        private int start;
        private int tileNumber;

        Tile(int tileNumber, int start, int end) {
            this.tileNumber = tileNumber;
            this.start = start;
            this.end = end;
            containedRecords = new ArrayList(2 * tileSize);
            overlappingRecords = new ArrayList();
        }

        /**
         * @return the tileNumber
         */
        public int getTileNumber() {
            return tileNumber;
        }

        /**
         * @param tileNumber the tileNumber to set
         */
        public void setTileNumber(int tileNumber) {
            this.tileNumber = tileNumber;
        }

        /**
         * @return the start
         */
        public int getStart() {
            return start;
        }

        /**
         * @param start the start to set
         */
        public void setStart(int start) {
            this.start = start;
        }

        /**
         * @return the containedRecords
         */
        public List<Alignment> getContainedRecords() {
            return containedRecords;
        }

        /**
         * @param containedRecords the containedRecords to set
         */
        public void setContainedRecords(List<Alignment> containedRecords) {
            this.containedRecords = containedRecords;
        }

        /**
         * @return the overlappingRecords
         */
        public List<Alignment> getOverlappingRecords() {
            return overlappingRecords;
        }

        /**
         * @param overlappingRecords the overlappingRecords to set
         */
        public void setOverlappingRecords(List<Alignment> overlappingRecords) {
            this.overlappingRecords = overlappingRecords;
        }

        /**
         * @return the loaded
         */
        public boolean isLoaded() {
            return loaded;
        }

        /**
         * @param loaded the loaded to set
         */
        public void setLoaded(boolean loaded) {
            this.loaded = loaded;
        }
    }

    public class TiledIterator implements CloseableIterator<Alignment> {

        int tileIdx = 0;
        Iterator<Alignment> currentSamIterator;
        int end;
        Alignment nextRecord;
        int start;
        List<Alignment> alignments;

        TiledIterator(int start, int end, List<Alignment> alignments) {
            this.alignments = alignments;
            this.start = start;
            this.end = end;
            currentSamIterator = alignments.iterator();
            advanceToFirstRecord();
        }

        public void close() {
            // No-op
        }

        public boolean hasNext() {
            return nextRecord != null;
        }

        public Alignment next() {
            Alignment ret = nextRecord;

            advanceToNextRecord();

            return ret;
        }

        public void remove() {

            // ignored
        }

        private void advanceToFirstRecord() {
            advanceToNextRecord();
        }

        private void advanceToNextRecord() {
            advance();

            while ((nextRecord != null) && (nextRecord.getEnd() < start)) {
                advance();
            }
        }

        private void advance() {
            if (currentSamIterator.hasNext()) {
                nextRecord = currentSamIterator.next();
                if (nextRecord.getAlignmentStart() > end) {
                    nextRecord = null;
                }
            } else {
                nextRecord = null;
            }
        }
    }
}


//~ Formatted by Jindent --- http://www.jindent.com
