/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam.reader;

import org.broad.igv.sam.*;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import net.sf.samtools.BAMRecordCodec;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMTextHeaderCodec;
import net.sf.samtools.util.AsciiLineReader;
import net.sf.samtools.util.CloseableIterator;
import net.sf.samtools.util.LineReader;
import org.apache.log4j.Logger;
import org.broad.igv.util.ResourceLocator;

/**
 *
 * @author jrobinso
 */
public class BAMRemoteQueryReader implements AlignmentQueryReader {

    Logger log = Logger.getLogger(BAMRemoteQueryReader.class);
    String serverURL;
    String file;
    SAMFileHeader header;
    String chr;
    int start;
    int end;
    boolean contained;

    public BAMRemoteQueryReader(ResourceLocator locator) {
        this.serverURL = locator.getServerURL();
        this.file = locator.getPath();
        loadHeader();
    }

    public void close() throws IOException {
        // Nothing to do
    }

    public CloseableIterator<Alignment> query(String chr, int start, int end, boolean contained) {
        this.chr = chr;
        this.start = start;
        this.end = end;
        this.contained = contained;

        try {
            URL url = new URL(serverURL + "?method=bamQuery&samFile=" + file + "&chr=" +
                    chr + "&start=" + start + "&end=" + end);
            URLConnection connection = url.openConnection();
            connection.setRequestProperty("Connection", "close");

            //connection.getInputStream().read();
            return new RemoteQueryIterator(connection.getInputStream());

        } catch (IOException ex) {
            log.error("Error opening file", ex);
            throw new RuntimeException(ex);
        }
    }

    public SAMFileHeader getHeader() {
        if (header == null) {
            loadHeader();
        }
        return header;
    }

    private void loadHeader() {
        InputStream is = null;
        try {
            URL url = new URL(serverURL + "?method=samHeader&samFile=" + file);
            URLConnection connection = url.openConnection();
            connection.setRequestProperty("Connection", "close");
            is = new BufferedInputStream(connection.getInputStream(), 8192);

            LineReader reader = new AsciiLineReader(is);
            SAMTextHeaderCodec code = new SAMTextHeaderCodec();
            header = code.decode(reader, null);

        } catch (IOException ex) {
            log.error("Error opening file", ex);
            throw new RuntimeException(ex);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException ex) {
                    log.error("Error closing url stream", ex);
                }
            }
        }
    }

    class RemoteQueryIterator implements CloseableIterator<Alignment> {

        InputStream inputStream;
        SAMRecord nextRecord;
        BAMRecordCodec codec;
        int mReferenceIndex;

        public RemoteQueryIterator(InputStream is) throws IOException {
            this.inputStream = is;
            codec = new BAMRecordCodec(header);
            codec.setInputStream(is);
            mReferenceIndex = getHeader().getSequenceIndex(chr);
            advanceToFirstRecord();
        }

        private void advanceToFirstRecord() throws IOException {
            do {
                advance();
            } while (nextRecord != null &&
                    (contained && nextRecord.getAlignmentStart() < start ||
                    (nextRecord.getAlignmentEnd() < start)));
        }

        private void advance() throws IOException {
            nextRecord = getNextRecord();
        }

        // TODO -- account for sequence (chr)
        SAMRecord getNextRecord() throws IOException {
            while (true) {

                // Pull next record from stream
                final SAMRecord record = codec.decode();
                if (record == null) {
                    return null;
                }
                // If beyond the end of this reference sequence, end iteration
                final int referenceIndex = record.getReferenceIndex();
                if (referenceIndex != mReferenceIndex) {
                    if (referenceIndex < 0 ||
                            referenceIndex > mReferenceIndex) {

                        return null;
                    }
                    // If before this reference sequence, continue
                    continue;
                }

                if (start == 0 && end == Integer.MAX_VALUE) {
                    // Quick exit to avoid expensive alignment end calculation
                    return record;
                }
                final int alignmentStart = record.getAlignmentStart();
                final int alignmentEnd = record.getAlignmentEnd();
                if (alignmentStart > end) {
                    // If scanned beyond target region, end iteration
                    return null;
                }

                // Filter for overlap with region
                if (contained) {
                    if (alignmentStart >= start && alignmentEnd <= end) {
                        return record;
                    }
                } else {
                    if (alignmentEnd >= start && alignmentStart <= end) {
                        return record;
                    }
                }
            }
        }

        public void close() {
            if (inputStream != null) {
                try {
                    inputStream.close();
                    inputStream = null;
                } catch (IOException ex) {
                    log.error("Error closing sam record stream", ex);
                }
            }
        }

        public boolean hasNext() {
            return nextRecord != null;
        }

        public SamAlignment next() {
            try {
                SamAlignment ret = new SamAlignment(nextRecord);
                advance();
                return ret;
            } catch (IOException ex) {
                log.error("Error advancing sam iterator", ex);
                return null;
            }
        }

        public void remove() {
            // ignored
        }

        // Just in case caller forgot to close the iterator
        @Override
        protected void finalize() throws Throwable {
            super.finalize();
            if (inputStream != null) {
                inputStream.close();
            }
        }
    }
}
