/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam.reader;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import javax.swing.JProgressBar;
import org.broad.igv.feature.ParsingUtils;
import org.broad.igv.sam.reader.SamIndexCreatorDialog.IndexWorker;
import org.broad.igv.ui.GuiUtilities;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public abstract class AlignmentIndexer {

    static int DEFAULT_TILEWIDTH = 16000;
    static int MILLISECONDS_IN_MINUTE = 60 * 1000;
    /** Optional progress bar.
     * TODO -- this should be done through event/listeners
     */
    SamIndexCreatorDialog.IndexWorker worker;
    JProgressBar progressBar;
    File samFile;

    public static AlignmentIndexer getInstance(File samFile,
            JProgressBar progressBar,
            SamIndexCreatorDialog.IndexWorker worker) {

        if (samFile.getName().endsWith("sorted.txt")) {
            return new GeraldIndexer(samFile, progressBar, worker);
        } else if (samFile.getName().endsWith(".aligned")) {
            return new DotAlignedIndexer(samFile, progressBar, worker);
        }
        return new SamIndexer(samFile, progressBar, worker);
    }

    AlignmentIndexer(File samFile,
            JProgressBar progressBar,
            SamIndexCreatorDialog.IndexWorker worker) {
        this.samFile = samFile;
        this.progressBar = progressBar;
        this.worker = worker;
    }

    public SamIndex createSamIndex() throws
            IOException,
            FileNotFoundException {
        File idxFile = new File(samFile.getParent(), samFile.getName() + ".sai");
        return createSamIndex(idxFile);
    }

    public SamIndex createSamIndex(File idxFile) throws
            IOException,
            FileNotFoundException {
        return createSamIndex(idxFile, DEFAULT_TILEWIDTH);
    }

    public SamIndex createSamIndex(
            File idxFile, int tileWidth) throws
            IOException,
            FileNotFoundException {

        FileInputStream is = new FileInputStream(samFile);
        InputStream bis = new BufferedInputStream(is);
        AsciiLineReader reader = new AsciiLineReader(bis);

        long fileLength = samFile.length();
        long progressIncrement = fileLength / 100;

        long lastFilePosition = 0;


        String lastChr = null;
        int lastAlignmentStart = 0;

        String[] fields = new String[100];

        SamIndex samIndex = new SamIndex(tileWidth);
        int recordCount = 0;
        long filePosition = 0;
        int lastTileRecorded = 0;
        int longestFeature = 1000;

        long startTime = System.currentTimeMillis();
        int progressCounter = 1; // progress in %

        String nextLine = "";
        while ((nextLine = reader.readLine(true)) != null) {

            if (worker != null && worker.isCancelled()) {
                return null;
            }

            int nBytes = nextLine.length();
            nextLine = nextLine.trim();
            int nFields = ParsingUtils.split(nextLine, fields, '\t');
            if (!nextLine.startsWith("@") && nFields > 3) {
                String chr = getChromosome(fields);

                if (lastChr == null || !chr.equals(lastChr)) {
                    // New chromosome.  Start a new section
                    if (lastChr != null) {
                        samIndex.add(lastChr, filePosition, recordCount, longestFeature);
                        filePosition = lastFilePosition;
                        lastTileRecorded = 0;
                        recordCount = 0;
                        lastAlignmentStart = 0;
                    }

                    lastChr = chr;
                } else {
                    // Get alignmetn start and verify file is sorted.
                    int alignmentStart = getAlignmentStart(fields);
                    if (alignmentStart < lastAlignmentStart) {
                        throw new UnsortedFileException("SAM file must be sorted by start position. " +
                                "Sort test failed at: " + nextLine);
                    }

                    lastAlignmentStart = alignmentStart;

                    int tileNumber = alignmentStart / tileWidth;
                    if (tileNumber > lastTileRecorded) {

                        // We have crossed a tile boundary.  Record index and counts for previous tile
                        samIndex.add(lastChr, filePosition, recordCount, longestFeature);

                        // If tiles were skipped record zero counts for these.
                        for (int cnt = 0; cnt < (tileNumber - lastTileRecorded - 1); cnt++) {
                            samIndex.add(lastChr, filePosition, 0, longestFeature);
                        }

                        filePosition = lastFilePosition;
                        lastTileRecorded = tileNumber;
                        recordCount = 0;
                    }

                    recordCount++;
                }

            }

            // Record last partial tile
            samIndex.add(lastChr, filePosition, recordCount, longestFeature);


            lastFilePosition += nBytes;

            if (lastFilePosition > (progressCounter * progressIncrement)) {
                updateProgress(progressCounter, startTime);
                progressCounter++;

            }
        }

        is.close();

        if (idxFile != null) {
            samIndex.store(idxFile);
        }

        if (progressBar != null) {
            progressBar.setValue(100);
        }

        return samIndex;

    }

    abstract int getAlignmentStart(String[] fields) throws NumberFormatException;

    abstract String getChromosome(String[] fields);

    private void updateProgress(int progressCounter, long startTime) {
        final long timeToComplete = ((100 - progressCounter) *
                (System.currentTimeMillis() - startTime)) / progressCounter;
        final int p = progressCounter;
        if (progressBar != null) {
            GuiUtilities.invokeOnEventThread(new Runnable() {

                public void run() {
                    progressBar.setValue(p);
                    if (worker != null) {
                        worker.setTimeRemaining(timeToComplete);
                    }

                }
            });
        } else {
            System.out.println(
                    "Progress: " + progressCounter + "%");
        }





    }

    static public class SamIndexer extends AlignmentIndexer {

        public SamIndexer(File samFile, JProgressBar progressBar, IndexWorker worker) {
            super(samFile, progressBar, worker);
        }

        int getAlignmentStart(String[] fields) throws NumberFormatException {
            // Get alignmetn start and verify file is sorted.
            int alignmentStart = Integer.parseInt(fields[3].trim()) - 1;
            return alignmentStart;
        }

        String getChromosome(String[] fields) {
            String chr = fields[2];
            return chr;
        }
    }

    /*
     *         String readChr = mapChromosome(fields[CHROMOSOME_COLUMN]);
    int alignmentStart = Integer.parseInt(fields[ALIGNMENT_START_COLUMN].trim()) - 1;
     */
    static public class GeraldIndexer extends AlignmentIndexer {

        public GeraldIndexer(File samFile, JProgressBar progressBar, IndexWorker worker) {
            super(samFile, progressBar, worker);
        }

        int getAlignmentStart(String[] fields) throws NumberFormatException {
            // Get alignmetn start and verify file is sorted.
            return Integer.parseInt(fields[GeraldParser.ALIGNMENT_START_COLUMN].trim()) - 1;
        }

        String getChromosome(String[] fields) {
            return GeraldParser.mapChromosome(fields[GeraldParser.CHROMOSOME_COLUMN]);
        }
    }

    static public class DotAlignedIndexer extends AlignmentIndexer {

        public DotAlignedIndexer(File samFile, JProgressBar progressBar, IndexWorker worker) {
            super(samFile, progressBar, worker);
        }

        int getAlignmentStart(String[] fields) throws NumberFormatException {
            // Get alignmetn start and verify file is sorted.
            return Integer.parseInt(fields[1]) - 1;
        }

        String getChromosome(String[] fields) {
            return fields[0];
        }
    }
}

