/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.feature.LocusScore;

//~--- JDK imports ------------------------------------------------------------


import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import net.sf.samtools.SAMRecord;
import org.broad.igv.feature.ParsingUtils;

/**
 *
 * @author jrobinso
 */
public class SamAlignment extends AbstractAlignment implements Alignment {

    public static final char DELETE_CHAR = '-';
    public static final char SKIP_CHAR = '=';
    public static final char MATCH = 'M';
    public static final char INSERTION = 'I';
    public static final char DELETION = 'D';
    public static final char SKIPPED_REGION = 'N';
    public static final char SOFT_CLIP = 'S';
    public static final char HARD_CLIP = 'H';
    public static final char PADDING = 'P';
    private int alignmentStart;
    private int alignmentEnd;
    boolean negativeStrand;
    boolean readNegativeStrandFlag;
    boolean duplicateReadFlag;
    boolean readUnmappedFlag;
    boolean readPairedFlag;
    boolean properPairFlag;
    SAMRecord record;
    String cigarString;
    String readSequence;
    private boolean softClippedStart = false;
    private boolean softClippedEnd = false;

    /**
     * Constructs ...
     *
     *
     * @param record
     */
    public SamAlignment(SAMRecord record) {
        if (record.getReadName().trim().equals("IL7_360:3:77:695:562")) {
            System.out.println();
        }

        this.record = record;
        this.chr = ParsingUtils.convertChrString("", record.getReferenceName());


        // SAMRecord is 1 based inclusive.  IGV is 0 based exclusive.
        this.alignmentStart = record.getAlignmentStart() - 1;
        this.alignmentEnd = record.getAlignmentEnd();
        this.negativeStrand = record.getReadNegativeStrandFlag();
        this.cigarString = record.getCigarString();
        this.setMappingQuality(record.getMappingQuality());
        this.readName = record.getReadName().trim();
        this.readNegativeStrandFlag = record.getReadNegativeStrandFlag();
        this.duplicateReadFlag = record.getDuplicateReadFlag();
        this.readUnmappedFlag = record.getReadUnmappedFlag();
        this.readPairedFlag = record.getReadPairedFlag();
        this.setInferredInsertSize(record.getInferredInsertSize());
        this.readSequence = record.getReadString();

        if (record.getReadPairedFlag()) {
            String mateChr = ParsingUtils.convertChrString("", record.getMateReferenceName());
            this.properPairFlag = record.getProperPairFlag();
            this.setMate(new ReadMate(mateChr,
                    record.getMateAlignmentStart(),
                    record.getMateNegativeStrandFlag(),
                    record.getMateUnmappedFlag()));
        }


        createAlignmentBlocks(record.getCigarString(), record.getReadBases(),
                record.getBaseQualities());


    }

    /**
     * Create the alignment blocks from the read bases and alignment information in the CIGAR
     * string.  The CIGAR string encodes insertions, deletions, skipped regions, and padding.
     *
     * @param cigarString
     * @param readBases
     * @param readBaseQualities
     */
    /**
     * Create the alignment blocks from the read bases and alignment information in the CIGAR
     * string.  The CIGAR string encodes insertions, deletions, skipped regions, and padding.
     *
     * @param cigarString
     * @param readBases
     * @param readBaseQualities
     */
    void createAlignmentBlocks(String cigarString, byte[] readBases, byte[] readBaseQualities) {

        int nInsertions = 0;
        int nBlocks = 0;

        List<CigarOperator> operators = new ArrayList();
        StringBuffer buffer = new StringBuffer(4);

        if (cigarString.equals("*")) {
            alignmentBlocks = new AlignmentBlock[1];
            alignmentBlocks[0] = new AlignmentBlock(getStart(), readBases, readBaseQualities);
            return;
        }

        boolean firstOperator = true;
        int softClippedBaseCount = 0;
        for (int i = 0; i < cigarString.length(); i++) {
            char next = cigarString.charAt(i);
            if (Character.isDigit(next)) {
                buffer.append(next);
            } else {
                int nBases = Integer.parseInt(buffer.toString());
                char op = next;
                if (op == MATCH) {
                    nBlocks++;
                } else if (op == SOFT_CLIP && firstOperator) {
                    softClippedBaseCount += nBases;
                } else if (op == DELETION || op == SKIPPED_REGION) {
                } else if (op == INSERTION) {
                    nInsertions++;
                }
                operators.add(new CigarOperator(nBases, op));
                buffer = new StringBuffer(4);

                firstOperator = false;
            }

        }

        alignmentBlocks = new AlignmentBlock[nBlocks];
        insertions = new AlignmentBlock[nInsertions];

        // fill new arrays
        int fromIdx = softClippedBaseCount;
        int blockStart = getStart();
        int blockIdx = 0;
        int insertionIdx = 0;
        for (CigarOperator op : operators) {
            if (op.operator == MATCH) {
                byte[] blockBases = new byte[op.nBases];
                byte[] blockQualities = new byte[op.nBases];

                // TODO -- represent missing sequence ("*") explicitly rather.
                // This is inefficient with respect to space
                if (readBases == null || readBases.length == 0) {
                    Arrays.fill(blockBases, (byte) '=');
                    Arrays.fill(readBaseQualities, (byte) 255);
                } else {

                    System.arraycopy(readBases, fromIdx, blockBases, 0, op.nBases);
                    System.arraycopy(readBaseQualities, fromIdx, blockQualities, 0, op.nBases);
                }

                alignmentBlocks[blockIdx++] = new AlignmentBlock(blockStart, blockBases,
                        blockQualities);

                fromIdx += op.nBases;
                blockStart += op.nBases;
            } else if (op.operator == DELETION  || op.operator == SKIPPED_REGION) {
                blockStart += op.nBases;
            } else if (op.operator == INSERTION) {
                byte[] blockBases = new byte[op.nBases];
                byte[] blockQualities = new byte[op.nBases];

                // TODO -- represent missing sequence ("*") explicitly rather.
                // This is inefficient with respect to space
                if (readBases == null || readBases.length == 0) {
                    Arrays.fill(blockBases, (byte) '=');
                    Arrays.fill(readBaseQualities, (byte) 255);
                } else {
                    System.arraycopy(readBases, fromIdx, blockBases, 0, op.nBases);
                    System.arraycopy(readBaseQualities, fromIdx, blockQualities, 0, op.nBases);
                }

                insertions[insertionIdx++] = new AlignmentBlock(blockStart, blockBases,
                        blockQualities);

                fromIdx += op.nBases;

            }
        }

    }


    // Default constructor to support unit tests
    SamAlignment() {
    }

    public boolean isNegativeStrand() {
        return readNegativeStrandFlag;
    }

    public boolean isDuplicate() {
        return duplicateReadFlag;
    }

    public boolean isMapped() {
        return !readUnmappedFlag;
    }

    public boolean isPaired() {
        return readPairedFlag;
    }

    public boolean isProperPair() {
        return properPairFlag;
    }

    /**
     * TODO
     *
     *
     * @return
     */
    public LocusScore copy() {
        return new SamAlignment(this);
    }

    /**
     * A copy constructor
     *
     */
    private SamAlignment(SamAlignment alignment) {
        this.chr = alignment.chr;
        this.alignmentStart = alignment.alignmentStart;
        this.alignmentEnd = alignment.alignmentEnd;
        this.negativeStrand = alignment.negativeStrand;
        this.mate = alignment.mate;
        this.alignmentBlocks = alignment.alignmentBlocks;
        this.insertions = alignment.insertions;
        this.cigarString = alignment.cigarString;
        this.mappingQuality = alignment.mappingQuality;
        this.readName = alignment.readName;
        this.readNegativeStrandFlag = alignment.readNegativeStrandFlag;
        this.duplicateReadFlag = alignment.duplicateReadFlag;
        this.readUnmappedFlag = alignment.readUnmappedFlag;
        this.readPairedFlag = alignment.readPairedFlag;
        this.inferredInsertSize = alignment.inferredInsertSize;
        this.properPairFlag = alignment.properPairFlag;
    }

    /**
     * @return the unclippedStart
     */
    public int getAlignmentStart() {
        return alignmentStart;
    }

    /**
     * @param unclippedStart the unclippedStart to set
     */
    public void setUnclippedStart(int unclippedStart) {
        this.alignmentStart = unclippedStart;
    }

    public String getCigarString() {
        return cigarString;
    }

    public String getReadSequence() {
        return readSequence;
    }

    /**
     * @return the alignmentEnd
     */
    public int getAlignmentEnd() {
        return alignmentEnd;
    }

    public int getStart() {
        return alignmentStart;
    }

    public void setStart(int start) {
        this.alignmentStart = start;
    }

    public int getEnd() {
        return alignmentEnd;
    }

    public void setEnd(int end) {
        this.alignmentEnd = alignmentEnd;
    }

    /**
     * @return the softClippedStart
     */
    public boolean isSoftClippedStart() {
        return softClippedStart;
    }

    /**
     * @return the softClippedEnd
     */
    public boolean isSoftClippedEnd() {
        return softClippedEnd;
    }

    static class CigarOperator {

        int nBases;
        char operator;

        /**
         * Constructs ...
         *
         *
         * @param nBases
         * @param operator
         */
        public CigarOperator(int nBases, char operator) {
            this.nBases = nBases;
            this.operator = operator;
        }
    }
}
