/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.sam.reader.SamQueryReaderFactory;
import org.broad.igv.sam.reader.AlignmentQueryReader;
import com.jidesoft.swing.JidePopupMenu;
import java.awt.event.MouseEvent;
import org.apache.log4j.Logger;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.renderer.Renderer;
import org.broad.igv.track.RegionScoreType;
import org.broad.igv.track.RenderContext;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import net.sf.samtools.SAMFileHeader;
import org.broad.igv.renderer.GraphicUtils;
import org.broad.igv.track.AbstractTrack;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import net.sf.samtools.util.CloseableIterator;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.track.SequenceTrack;
import org.broad.igv.track.TrackMenuUtils;
import org.broad.igv.ui.GuiUtilities;
import org.broad.igv.ui.LongRunningTask;
import org.broad.igv.ui.ViewContext;
import org.broad.igv.ui.panel.DragEvent;
import org.broad.igv.ui.panel.DragEventManager;
import org.broad.igv.ui.panel.DragListener;

/**
 *
 * @author jrobinso
 */
public class AlignmentTrack extends AbstractTrack implements DragListener {

    public static final int MIN_ALIGNMENT_SPACING = 10;
    SequenceTrack sequenceTrack;
    boolean filterZeroQuality = true;
    boolean fileterDuplicates = true;
    private static Logger log = Logger.getLogger(AlignmentTrack.class);
    private static int EXPANDED_HEIGHT = 14;
    private static int COLLAPSED_HEIGHT = 4;
    AlignmentQueryReader reader;
    int height;
    AlignmentRenderer renderer;
    String lastChromosomeName = null;
    int bamIntervalWidth = 16000;
    List<Row> alignmentRows;
    double minVisibleScale = 25;
    Interval loadedInterval = null;
    private boolean shortChrNameConventions;
    Rectangle renderedRect;
    //HDFDataSource coverageData;

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param name
     * @param alignments
     */
    public AlignmentTrack(ResourceLocator locator, String name) {
        super(locator, name);

        reader = SamQueryReaderFactory.getReader(locator);
        //reader = new BAMCachingQueryReader(locator);

        float maxRange = PreferenceManager.getInstance().getSAMPreferences().getMaxVisibleRange();
        minVisibleScale = (maxRange * 1000) / 700;

        renderer = new AlignmentRenderer();
        this.setExpanded(true);

        sequenceTrack = new SequenceTrack("Reference");
        sequenceTrack.setHeight(14);


        SAMFileHeader header = reader.getHeader();
        if (header != null &&
                header.getSequenceDictionary().getSequence("1") != null &&
                header.getSequenceDictionary().getSequence("chr1") == null) {
            shortChrNameConventions = true;
        }

        //HDFDataManager dm = new HDFDataManager(new ResourceLocator(
        //        "/Users/jrobinso/IGV/NA12878.chrom1.SRP000032.2009_02.counts.h5"));
        //coverageData = new HDFDataSource(dm, "", 0);

        DragEventManager.getInstance().addDragListener(this);

    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        reader.close();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public int getHeight() {
        int h = Math.max(100, getNLevels() * (getRowHeight()) + 20);
        return h;
    }

    private int getRowHeight() {
        return isExpanded() ? EXPANDED_HEIGHT : COLLAPSED_HEIGHT;
    }

    private int getNLevels() {


        return (alignmentRows == null ? 1 : alignmentRows.size());
    }

    @Override
    public int getPreferredHeight() {
        return Math.max(100, getHeight());
    }

    @Override
    public void renderName(Graphics2D graphics, Rectangle trackRectangle, Rectangle visibleRect) {
        Rectangle rect = null;
        if (visibleRect != null) {
            Rectangle intersectedRect = trackRectangle.intersection(visibleRect);
            if (intersectedRect.height > 15) {
                rect = intersectedRect;
            } else {
                rect = new Rectangle(trackRectangle);
            }
        }

        super.renderName(graphics, rect, visibleRect);
    }

    public void render(RenderContext context, Rectangle rect) {

        // Split rects
        int seqHeight = sequenceTrack.getHeight();
        if (seqHeight > 0) {
            Rectangle seqRect = new Rectangle(rect);
            seqRect.height = seqHeight;
            sequenceTrack.render(context, seqRect);
        }

        rect.y += seqHeight;
        rect.height -= seqHeight;
        renderedRect = new Rectangle(rect);

        // Alignments are rendered for scales > ~ 1 Mb
        //  scale * viewWidth = 1 Mb => scale ~ 1000000 / 700  => 1430
        //                      10 Kb =>  10000/700 =>14
        if (context.getScale() > minVisibleScale) {
            //double origin = context.getOrigin();
            //double locScale = context.getScale();
            //double end = origin + rect.getWidth() * locScale;
            //int zoom = context.getZoom();
            //List<LocusScore> scores = coverageData.getSummaryScoresForRange(context.getChr(),
            //        (int) origin, (int) end, zoom);

            //BarChartRenderer re = new BarChartRenderer();
            //re.renderScores(this, scores, context, rect);

            Graphics2D g = context.getGraphic2DForColor(Color.black);
            GraphicUtils.drawCenteredText("Zoom in to see alignments.", rect, g);
            return;
        }

        renderFeatures(context, rect);
    }

    private void renderFeatures(RenderContext context, Rectangle inputRect) {
        try {

            if (alignmentRows == null) {
                return;
            }

            PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
            float maxRange = prefs.getMaxVisibleRange();
            int maxLevels = prefs.getMaxLevels();

            Rectangle visibleRect = context.getVisibleRect();

            // Divide rectangle into equal height levels
            double y = inputRect.getY();
            double h = isExpanded() ? EXPANDED_HEIGHT : COLLAPSED_HEIGHT;
            int levelNumber = 0;
            // levelList is copied to prevent concurrent modification exception
            List<Row> tmp = new ArrayList(alignmentRows);
            for (Row row : tmp) {

                if ((visibleRect != null && y > visibleRect.getMaxY()) ||
                        levelNumber > maxLevels) {
                    return;
                }

                if (y + h > visibleRect.getY()) {
                    Rectangle rect = new Rectangle(inputRect.x, (int) y, inputRect.width, (int) h);
                    renderer.renderAlignments(row.alignments, context, rect);
                }

                y += h;
                levelNumber++;
            }
        } catch (Exception ex) {
            log.error("Error rendering track", ex);
            throw new RuntimeException("Error rendering track ", ex);

        }

    }

    public void reloadData() {
        loadedInterval = null;
        alignmentRows = null;
        preloadData();

    }

    public void preloadData() {
        ViewContext vc = IGVModel.getInstance().getViewContext();
        preloadData(vc.getChrName(), (int) vc.getOrigin(), (int) vc.getEnd(), vc.getZoom());
    }
    public boolean isLoading = false;

    @Override
    public synchronized void preloadData(final String chr,
            final int start, final int end, int zoom) {

        final PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
        final float maxRange = prefs.getMaxVisibleRange();
        final int maxLevels = prefs.getMaxLevels();
        final int qualityThreshold = prefs.getQualityThreshold();
        minVisibleScale = (maxRange * 1000) / 700;

        if (IGVModel.getInstance().getViewContext().getScale() > minVisibleScale) {
            return;
        }

// If the requested interval is outside the currently loaded range load
        if (loadedInterval == null || !loadedInterval.contains(chr, start, end) && !isLoading) {
            isLoading = true;

            Runnable runnable = new Runnable() {

                public void run() {
                    alignmentRows = new ArrayList(maxLevels);
                    // Expand start and end to facilitate panning, but by no more than
                    // 1 screen or 8kb, whichever is less
                    int expandLength = Math.min(8000, end - start) / 2;
                    int intervalStart = Math.max(0, start - expandLength);
                    int intervalEnd = end + expandLength;
                    CloseableIterator<Alignment> iter = null;
                    try {
                        String sequence = chr;
                        // TODO -- put here to handle 1 vs chr1 problems.
                        if (shortChrNameConventions) {
                            sequence = chr.replace("chr", "");
                        }
                        iter = reader.query(sequence, intervalStart, intervalEnd, false);
                        alignmentRows = AlignmentPacker.packAlignments(iter, prefs.isShowDuplicates(), qualityThreshold, maxLevels, intervalEnd);
                        loadedInterval = new Interval(chr, intervalStart, intervalEnd);
                    } catch (Exception exception) {
                        log.error("Error loading alignments", exception);
                        JOptionPane.showMessageDialog(IGVMainFrame.getInstance(), "There was an error reading: " + getSourceFile() + exception.getMessage());
                    } finally {
                        isLoading = false;
                        if (iter != null) {
                            iter.close();
                        }
                    }
                }
            };

            // Don't block the swing dispatch thread
            if (SwingUtilities.isEventDispatchThread()) {
                try {

                    LongRunningTask.submit(runnable).get();   // <= This will force a wait until the task is complete
                } catch (Exception ex) {
                    log.error("Error loading alignments", ex);
                }
            } else {
                runnable.run();
            }



            //calculateLevels(features);
            IGVMainFrame.getInstance().doResizeTrackPanels();
        }

    }

    /**
     * Sort alignment rows such that alignments that intersect from the
     * center appear left to right by start position
     */
    public void sortRows() {
        if (alignmentRows == null) {
            return;
        }

        double center = IGVModel.getInstance().getViewContext().getCenter();
        for (Row row : alignmentRows) {
            row.updateScore(center);
        }

        Collections.sort(alignmentRows, new Comparator<Row>() {

            public int compare(Row arg0, Row arg1) {
                return (int) (arg0.score - arg1.score);
            }
        });
    }

    /**
     * Copy the contents of the popup text to the system clipboard.
     */
    public void copyToClipboard(final MouseEvent e) {
        double location = getViewContext().getChromosomePosition(e.getX());
        double displayLocation = location + 1;
        Alignment alignment = this.getAlignmentAt(displayLocation, e.getY());

        if (alignment != null) {
            StringBuffer buf = new StringBuffer();
            buf.append(alignment.getValueString(location, null).replace("<br>", "\n"));
            buf.append("\n");
            buf.append("Alignment start position = " + alignment.getChromosome() + ":" + (alignment.getAlignmentStart() + 1));
            buf.append("\n");
            buf.append(alignment.getReadSequence());
            StringSelection stringSelection = new StringSelection(buf.toString());
            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            clipboard.setContents(stringSelection, null);
        }

    }

    /**
     * Copy the contents of the popup text to the system clipboard.
     */
    public void gotoMate(final MouseEvent e) {
        double location = getViewContext().getChromosomePosition(e.getX());
        double displayLocation = location + 1;
        Alignment alignment = this.getAlignmentAt(displayLocation, e.getY());

        if (alignment != null) {
            ReadMate mate = alignment.getMate();
            if (mate != null && mate.isMapped()) {
                String chr = mate.mateChr;
                int start = mate.mateStart - 1;
                IGVModel.getInstance().getViewContext().centerOnLocation(chr, start);
            }
        }

    }

    public void setStatType(WindowFunction type) {
        // ignored
    }

    public WindowFunction getWindowFunction() {
        return null;
    }

    public void setRendererClass(Class rc) {
        // ignored
    }

    // SamTracks use a custom renderer, not derived from Renderer
    public Renderer getRenderer() {
        return null;
    }

    public boolean isLogNormalized() {
        return false;
    }

    public float getRegionScore(String chr, int start, int end, int zoom, RegionScoreType type) {
        return 0.0f;
    }

    public String getValueStringAt(
            String chr, double position, int y) {

        Alignment feature = getAlignmentAt(position, y);

        // TODO -- highlight mate

        String tmp = (feature == null) ? null : feature.getValueString(position, getWindowFunction());

        return tmp;
    }

    private Alignment getAlignmentAt(double position, int y) {
        if (alignmentRows == null || alignmentRows.isEmpty()) {
            return null;
        }

        int h = isExpanded() ? EXPANDED_HEIGHT : COLLAPSED_HEIGHT;
        int levelNumber = (y - renderedRect.y) / h;
        if (levelNumber < 0 || levelNumber >= alignmentRows.size()) {
            return null;
        }

        Row row = alignmentRows.get(levelNumber);
        List<Alignment> features = row.alignments;

        // give a 2 pixel window, otherwise very narrow features will be missed.
        double bpPerPixel = IGVModel.getInstance().getViewContext().getScale();
        double minWidth = 2 * bpPerPixel;    /* * */
        return (Alignment) FeatureUtils.getFeatureAt(position, minWidth, features);

    }

    public void dragStopped(DragEvent evt) {
        // if sort option == true
        if (PreferenceManager.getInstance().getSAMPreferences().isAutosort() &&
                IGVModel.getInstance().getViewContext().getScale() < 1) {
            sortRows();
            IGVMainFrame.getInstance().repaintDataPanels();
        }











    }

    class Interval {

        String chr;
        int start;
        int end;

        public Interval(String chr, int start, int end) {
            this.chr = chr;
            this.start = start;
            this.end = end;
        }

        boolean contains(String chr, int start, int end) {
            return this.chr.equals(chr) && this.start <= start && this.end >= end;
        }
    }

    public static class Row {

        int idx;
        double score = 0;
        List<Alignment> alignments;
        int start;
        int lastEnd;

        public Row(int idx) {
            this.idx = idx;
            this.alignments = new ArrayList(100);

        }

        public void addAlignment(Alignment alignment) {
            if (alignments.isEmpty()) {
                this.start = alignment.getStart();
            }
            alignments.add(alignment);
            lastEnd = alignment.getEnd();

        //System.out.println("Row: " + idx + " Added alignment: \t" + alignment.getAlignmentStart() +
        //        "\t" + alignment.getEnd());
        }

        public void updateScore(double center) {
            int adjustedCenter = (int) center;

            Alignment centerAlignment = getFeatureContaining(alignments, adjustedCenter);
            score = centerAlignment == null ? Double.MAX_VALUE : centerAlignment.getStart();

        }
    }

    private static Alignment getFeatureContaining(
            List<Alignment> features, int right) {

        int leftBounds = 0;
        int rightBounds = features.size() - 1;
        int idx = features.size() / 2;
        int lastIdx = -1;

        while (idx != lastIdx) {
            lastIdx = idx;
            Alignment f = features.get(idx);
            if (f.contains(right)) {
                return f;
            }

            if (f.getStart() > right) {
                rightBounds = idx;
                idx =
                        (leftBounds + idx) / 2;
            } else {
                leftBounds = idx;
                idx =
                        (rightBounds + idx) / 2;

            }

        }
        // Check the extremes
        if (features.get(0).contains(right)) {
            return features.get(0);
        }

        if (features.get(rightBounds).contains(right)) {
            return features.get(rightBounds);
        }

        return null;
    }

    @Override
    public boolean handleClick(MouseEvent e) {
        if (e.isPopupTrigger()) {
            getPopupMenu(e).show(e.getComponent(), e.getX(), e.getY());
            //sortRows();
            //IGVMainFrame.getInstance().repaintDataPanels();
            return true;
        } else {
            return super.handleClick(e);
        }

    }

    public JPopupMenu getPopupMenu(
            final MouseEvent evt) {

        JPopupMenu popupMenu = new JidePopupMenu();

        JLabel popupTitle = new JLabel("  " + getDisplayName(), JLabel.CENTER);

        Font newFont = popupMenu.getFont().deriveFont(Font.BOLD, 12);
        popupTitle.setFont(newFont);
        if (popupTitle != null) {
            popupMenu.add(popupTitle);
        }

        addSortMenuItem(popupMenu);
        addShadeBaseMenuItem(popupMenu);
        addCopyToClipboardItem(popupMenu, evt);
        addGoToMate(popupMenu, evt);
        popupMenu.addSeparator();


        JLabel trackSettingsHeading = new JLabel("  Track Settings",
                JLabel.LEFT);
        trackSettingsHeading.setFont(newFont);

        popupMenu.add(trackSettingsHeading);

        TrackMenuUtils.addTrackRenameItem(popupMenu);

        TrackMenuUtils.addExpandCollapseItem(popupMenu);

        addRemoveMenuItem(popupMenu);

        return popupMenu;
    }

    private void addRemoveMenuItem(JPopupMenu menu) {
        JMenuItem item = new JMenuItem("Remove Tracks");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        TrackMenuUtils.removeTracks();
                    }
                });
            }
        });
        menu.add(item);

    }

    public void addSortMenuItem(JPopupMenu menu) {
        // Change track height by attribute
        JMenuItem item = new JMenuItem("Sort");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        sortRows();
                        IGVMainFrame.getInstance().repaintDataPanels();
                    }
                });
            }
        });
        if (IGVModel.getInstance().getViewContext().getScale() >= MIN_ALIGNMENT_SPACING) {
            item.setEnabled(false);
        }

        menu.add(item);
    }

    public void addCopyToClipboardItem(JPopupMenu menu, final MouseEvent me) {
        // Change track height by attribute
        JMenuItem item = new JMenuItem("Copy read details to clipboard");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        copyToClipboard(me);
                    }
                });
            }
        });
        if (IGVModel.getInstance().getViewContext().getScale() >= MIN_ALIGNMENT_SPACING) {
            item.setEnabled(false);
        }

        menu.add(item);
    }

    public void addGoToMate(JPopupMenu menu, final MouseEvent me) {
        // Change track height by attribute
        JMenuItem item = new JMenuItem("Go to mate region");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        gotoMate(me);
                    }
                });
            }
        });

        menu.add(item);
    }

    public void addShadeBaseMenuItem(JPopupMenu menu) {
        // Change track height by attribute
        final JMenuItem item = new JCheckBoxMenuItem("Shade base by quality");
        item.setSelected(PreferenceManager.getInstance().getSAMPreferences().isShadeBaseQuality());
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        PreferenceManager.getInstance().put(
                                PreferenceManager.SAMPreferences.SHADE_BASE_QUALITY,
                                String.valueOf(item.isSelected()));
                        PreferenceManager.getInstance().getSAMPreferences().setShadeBaseQuality(item.isSelected());
                        IGVMainFrame.getInstance().repaintDataPanels();
                    }
                });
            }
        });

        menu.add(item);
    }
}
