/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam;

import cern.colt.map.OpenIntObjectHashMap;
import org.broad.igv.PreferenceManager.SAMPreferences;
import org.broad.igv.feature.SequenceManager;
import org.broad.igv.renderer.*;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.Track;
import org.broad.igv.ui.FontManager;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.ViewContext;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.PreferenceManager;
import org.broad.igv.util.ChromosomeColors;

/**
 *
 * @author jrobinso
 */
public class AlignmentRenderer {

    static Map<Character, Color> nucleotideColors = new HashMap();
    static Color purple = new Color(118, 24, 220);
    static float[] rbgBuffer = new float[3];
    static float[] colorComps = new float[3];
    static float[] alignComps = new float[3];
    static float[] whiteComponents = Color.white.getRGBColorComponents(null);
    static Color grey2 = new Color(165, 165, 165);
    static Color grey1 = new Color(200, 200, 200);


    static {
        nucleotideColors.put('A', Color.GREEN);
        nucleotideColors.put('a', Color.GREEN);
        nucleotideColors.put('C', Color.BLUE);
        nucleotideColors.put('c', Color.BLUE);
        nucleotideColors.put('T', Color.RED);
        nucleotideColors.put('t', Color.RED);
        nucleotideColors.put('G', new Color(242, 182, 65));
        nucleotideColors.put('g', new Color(242, 182, 65));
        nucleotideColors.put('N', Color.gray);
        nucleotideColors.put('n', Color.gray);
    }
    static Font font = FontManager.getScalableFont(10);
    private ViewContext viewContext;
    private SequenceRenderer seqRenderer;

    /**
     * Constructs ...
     *
     */
    public AlignmentRenderer() {
        this.viewContext = IGVModel.getInstance().getViewContext();
        this.seqRenderer = new SequenceRenderer();

    }

    /**
     * Render a list of alignments in the given rectangle.
     *
     * @param featureList
     * @param context
     * @param trackRectangle
     * @param track
     */
    public void renderAlignments(List<Alignment> alignments, RenderContext context, Rectangle rect) {

        double origin = context.getOrigin();
        double locScale = context.getScale();


        if ((alignments != null) && (alignments.size() > 0)) {

            final SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
            boolean shadeCenter = prefs.isShadeCenterFlag();
            int insertSizeThreshold = prefs.getInsertSizeThreshold();

            for (Alignment alignment : alignments) {

                // Compute the start and dend of the alignment in pixels
                double pixelStart = ((alignment.getStart() - origin) / locScale);
                double pixelEnd = ((alignment.getEnd() - origin) / locScale);

                // If the any part of the feature fits in the track rectangle draw  it
                if ((pixelEnd >= rect.getX()) && (pixelStart <= rect.getMaxX())) {
                    Color alignmentColor = getAlignmentColor(alignment, insertSizeThreshold, locScale, viewContext.getCenter(), shadeCenter);

                    Graphics2D g = context.getGraphic2DForColor(alignmentColor);
                    g.setFont(font);

                    // If the alignment is 3 pixels or less,  draw alignment as a single block,
                    // further detail would not be seen and just add to drawing overhead
                    if (pixelEnd - pixelStart < 4) {
                        int w = Math.max(1, (int) (pixelEnd - pixelStart));
                        int h = (int) Math.max(1, rect.getHeight() - 2);
                        int y = (int) (rect.getY() + (rect.getHeight() - h) / 2);
                        g.fillRect((int) pixelStart, y, w, h);
                    } else {
                        drawAlignment(alignment, rect, g, context, alignmentColor, prefs.isFlagUnmappedPair());
                    }

                }
            }

            // Draw a border around the center base
            if (locScale < 5) {
                // Calculate center lines
                double center = (int) (viewContext.getCenter() - origin);
                int centerLeftP = (int) (center / locScale);
                int centerRightP = (int) ((center + 1) / locScale);
                float transparency = Math.max(0.5f, (float) Math.round(10 * (1 - .75 * locScale)) / 10);
                Graphics2D gBlack = context.getGraphic2DForColor(new Color(0, 0, 0, transparency));
                GraphicUtils.drawDashedLine(gBlack, centerLeftP, rect.y, centerLeftP,
                        rect.y + rect.height);
                if ((centerRightP - centerLeftP > 2)) {
                    GraphicUtils.drawDashedLine(gBlack, centerRightP, rect.y, centerRightP,
                            rect.y + rect.height);
                }
            }
        }
    }

    /**
     * Method for drawing alignments without "blocks" (e.g. DotAlignedAlignment)
     * @param prefs
     * @param rect
     * @param alignment
     * @param g
     * @param origin
     * @param locScale
     * @param context
     */
    private void drawSimpleAlignment(Alignment alignment, Rectangle rect,
            Graphics2D g, RenderContext context, boolean flagUnmappedPair) {
        double origin = context.getOrigin();
        double locScale = context.getScale();
        int x = (int) ((alignment.getStart() - origin) / locScale);
        int length = alignment.getEnd() - alignment.getStart();
        int w = (int) Math.ceil(length / locScale);
        int h = (int) Math.max(1, rect.getHeight() - 2);
        int y = (int) (rect.getY() + (rect.getHeight() - h) / 2);
        int arrowLength = Math.min(5, w / 6);
        int[] xPoly = null;
        int[] yPoly = {y, y, y + h / 2, y + h, y + h};
        if (alignment.isNegativeStrand()) {
            xPoly = new int[]{x + w, x, x - arrowLength, x, x + w};
        } else {
            xPoly = new int[]{x, x + w, x + w + arrowLength, x + w, x};
        }
        g.fillPolygon(xPoly, yPoly, xPoly.length);

        if (flagUnmappedPair && alignment.isPaired() && !alignment.getMate().isMapped()) {
            Graphics2D cRed = context.getGraphic2DForColor(Color.red);
            cRed.drawPolygon(xPoly, yPoly, xPoly.length);
        }
    }

    /**
     * Draw a single alignment
     *
     * @param alignment
     * @param rect  -- the bounding rectangle.  Used to bound the height of the alignment
     * @param g  -- the GraphicContext on which to draw
     * @param context -- rendering context, contains misc parameters
     * @param alignmentColor  -- color of the alignment,  used for computing base color with alpha
     * @param flagUnmappedPair 
     */
    private void drawAlignment(Alignment alignment, Rectangle rect, Graphics2D g, RenderContext context,
            Color alignmentColor, boolean flagUnmappedPair) {

        double origin = context.getOrigin();
        double locScale = context.getScale();
        AlignmentBlock[] blocks = alignment.getAlignmentBlocks();

        if (blocks == null) {
            drawSimpleAlignment(alignment, rect, g, context, flagUnmappedPair);
            return;
        }

        AlignmentBlock terminalBlock = alignment.isNegativeStrand() ? blocks[0] : blocks[blocks.length - 1];

        Graphics2D greyGraphics = context.getGraphic2DForColor(new Color(185, 185, 185));

        int lastBlockEnd = Integer.MIN_VALUE;

        for (AlignmentBlock aBlock : alignment.getAlignmentBlocks()) {
            int x = (int) ((aBlock.getStart() - origin) / locScale);
            int w = (int) Math.ceil(aBlock.getBases().length / locScale);
            int h = (int) Math.max(1, rect.getHeight() - 2);
            int y = (int) (rect.getY() + (rect.getHeight() - h) / 2);

            // Create polygon to represent the alignment. 
            boolean isZeroQuality = alignment.getMappingQuality() == 0;
            if (w <= 10 || h <= 10 || aBlock != terminalBlock) {
                g.fillRect(x, y, w, h);
                if (isZeroQuality) {
                    greyGraphics.drawRect(x, y, w - 1, h);
                }

                if (flagUnmappedPair && alignment.isPaired() && !alignment.getMate().isMapped()) {
                    Graphics2D cRed = context.getGraphic2DForColor(Color.red);
                    cRed.drawRect(x, y, w, h);
                }
            } else {
                int arrowLength = Math.min(5, w / 6);
                int[] xPoly = null;
                int[] yPoly = {y, y, y + h / 2, y + h, y + h};
                if (alignment.isNegativeStrand()) {
                    xPoly = new int[]{x + w, x, x - arrowLength, x, x + w};
                } else {
                    xPoly = new int[]{x, x + w, x + w + arrowLength, x + w, x};
                }
                g.fillPolygon(xPoly, yPoly, xPoly.length);
                if (isZeroQuality) {
                    greyGraphics.drawPolygon(xPoly, yPoly, xPoly.length);
                }

                if (flagUnmappedPair && alignment.isPaired() && !alignment.getMate().isMapped()) {
                    Graphics2D cRed = context.getGraphic2DForColor(Color.red);
                    cRed.drawPolygon(xPoly, yPoly, xPoly.length);
                }
            }

            if (lastBlockEnd > Integer.MIN_VALUE) {
                Graphics2D gGrey = context.getGraphic2DForColor(Color.GRAY);
                gGrey.drawLine(lastBlockEnd, y + h / 2, x, y + h / 2);

            }
            lastBlockEnd = x + w;

            if (locScale < 5) {
                drawBases(context, rect, aBlock, alignmentColor);
            }
        }

        // Render insertions if locScale ~ 0.25 (base level)
        if (locScale < 0.25) {
            drawInsertions(origin, rect, locScale, alignment, context);
        }
    }

    /**
     * Draw the bases for an alignment block.
     * @param context
     * @param rect
     * @param alignment
     */
    private void drawBases(RenderContext context, Rectangle rect, AlignmentBlock block, Color alignmentColor) {

        alignmentColor.getRGBColorComponents(alignComps);

        double locScale = context.getScale();
        double origin = context.getOrigin();
        String chr = context.getChr();
        String genome = IGVModel.getInstance().getViewContext().getGenomeId();

        byte[] read = block.getBases();
        if ((read != null) && (read.length > 0)) {

            // Compute bounds, get a graphics to use,  and compute a font
            int pY = (int) rect.getY();
            int dY = (int) rect.getHeight();
            int dX = (int) Math.max(1, (1.0 / locScale));
            Graphics2D g = (Graphics2D) context.getGraphics().create();
            if (dX >= 8) {
                Font f = FontManager.getScalableFont(Font.BOLD, Math.min(dX, 12));
                g.setFont(f);
            }

            // Get the base qualities, start/end,  and reference sequence
            byte[] qualities = block.getQualities();
            int start = block.getStart();
            int end = start + read.length;
            byte[] reference = SequenceManager.readSequence(genome, chr, start, end);


            // Loop through base pair coordinates
            for (int loc = start; loc < end; loc++) {

                // Index into read array,  just the genomic location offset by
                // the start of this block
                int idx = loc - start;

                // Is this base a mismatch?  Note '=' means indicates a match by definition
                boolean misMatch =
                        (read[idx] != '=') &&
                        (reference != null && ((idx >= reference.length) || (reference[idx] != read[idx])));

                if (misMatch) {
                    char c = (char) read[loc - start];
                    Color color = nucleotideColors.get(c);
                    PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
                    if (prefs.isShadeBaseQuality()) {
                        float alpha = 0;
                        byte qual = qualities[loc - start];
                        int minQ = prefs.getBaseQualityMin();
                        if (qual < minQ) {
                            alpha = 0.1f;
                        } else {
                            int maxQ = prefs.getBaseQualityMax();
                            alpha = Math.max(0.1f, Math.min(1.0f, 0.1f + 0.9f * (qual - minQ) / (maxQ - minQ)));
                            color.getRGBColorComponents(colorComps);
                        }

                        // Round alpha to nearest 0.1, for effeciency;
                        alpha = ((int) (alpha * 10 + 0.5f)) / 10.0f;
                        color = getCompositeColor(alignComps, colorComps, alpha);
                    }


                    // If there is room for text draw the character, otherwise
                    // just draw a rectangle to represent the 
                    int pX0 = (int) ((loc - origin) / locScale);
                    if ((dX >= 8) && (dY >= 12)) {
                        if (color == null) {
                            color = Color.black;
                        }
                        g.setColor(color);
                        drawCenteredText(g, new char[]{c}, pX0, pY + 2, dX, dY - 2);
                    } else {

                        if (dX > 4) {
                            dX--;
                        }

                        if (color != null) {
                            g.setColor(color);
                            if (dY < 6) {
                                g.fillRect(pX0, pY + 1, dX, dY - 2);
                            } else {
                                g.fillRect(pX0, pY + 2, dX, dY - 4);
                            }
                        }
                    }
                }

            }
        }
    }

    private void drawCenteredText(Graphics2D g, char[] chars, int x, int y, int w, int h) {

        // Get measures needed to center the message
        FontMetrics fm = g.getFontMetrics();

        // How many pixels wide is the string
        int msg_width = fm.charsWidth(chars, 0, 1);

        // How far above the baseline can the font go?
        int ascent = fm.getMaxAscent();

        // How far below the baseline?
        int descent = fm.getMaxDescent();

        // Use the string width to find the starting point
        int msgX = x + w / 2 - msg_width / 2;

        // Use the vertical height of this font to find
        // the vertical starting coordinate
        int msgY = y + h / 2 - descent / 2 + ascent / 2;

        g.drawChars(chars, 0, 1, msgX, msgY);

    }

    private void drawInsertions(double origin, Rectangle rect, double locScale, Alignment alignment, RenderContext context) {

        Graphics2D gInsertion = context.getGraphic2DForColor(purple);
        AlignmentBlock[] insertions = alignment.getInsertions();
        if (insertions != null) {
            for (AlignmentBlock aBlock : insertions) {
                int x = (int) ((aBlock.getStart() - origin) / locScale);
                int h = (int) Math.max(1, rect.getHeight() - 2);
                int y = (int) (rect.getY() + (rect.getHeight() - h) / 2);

                gInsertion.fillRect(x - 2, y, 4, 2);
                gInsertion.fillRect(x - 1, y, 2, h);
                gInsertion.fillRect(x - 2, y + h - 2, 4, 2);
            }
        }
    }

    private Color getAlignmentColor(Alignment alignment, int insertSizeThreshold, double locScale,
            double center, boolean shadeCenter) {

        // Set color used to draw the feature.  Highlight features that intersect the
        // center line.  Also update row "score" if alignment intersects center line

        Color c = grey1;
        if (shadeCenter && center >= alignment.getStart() && center <= alignment.getEnd()) {
            if (locScale < 1) {
                c = grey2;
            }
        }

        if (alignment.getMappingQuality() > 0) {
            if (alignment.isPaired() && alignment.getMate().isMapped()) {
                boolean sameChr = alignment.getMate().mateChr.equals(alignment.getChromosome());
                if (sameChr) {
                    int readDistance = Math.abs(alignment.getInferredInsertSize());
                    if (readDistance > insertSizeThreshold) {
                        c = ChromosomeColors.getColor(alignment.getMate().mateChr);
                    //c = getDistanceColor(readDistance);
                    }
                } else {
                    c = ChromosomeColors.getColor(alignment.getMate().mateChr);
                    if (c == null) {
                        c = Color.black;
                    }
                }
            }
        } else {
            // Maping Q = 0
            float alpha = 0.15f;
            c.getColorComponents(rbgBuffer);
            // Assuming white background TODO -- this should probably be passed in
            return getCompositeColor(whiteComponents, rbgBuffer, alpha);
        }

        return c;
    }
    /**
     * Compute a composite color using alpha transparency rules
     */
    // TODO -- move this method and cache to a utility class
    private static OpenIntObjectHashMap colorMap = new OpenIntObjectHashMap(1000);

    private static Color getCompositeColor(float[] dest, float[] source, float alpha) {
        int r = (int) ((alpha * source[0] + (1 - alpha) * dest[0]) * 255 + 0.5);
        int g = (int) ((alpha * source[1] + (1 - alpha) * dest[1]) * 255 + 0.5);
        int b = (int) ((alpha * source[2] + (1 - alpha) * dest[2]) * 255 + 0.5);
        int a = 255;
        int value = ((a & 0xFF) << 24) |
                ((r & 0xFF) << 16) |
                ((g & 0xFF) << 8) |
                ((b & 0xFF) << 0);

        Color c = (Color) colorMap.get(value);
        if (c == null) {
            c = new Color(value);
            colorMap.put(value, c);
        }
        return c;
    }
}
