/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.TreeSet;
import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;
import org.broad.igv.sam.AlignmentTrack.Row;

/**
 * A utility class to experiment with alignment packing methods. 
 * 
 * Numbers:  
 * 
 * packAlignments1  (original)
 *   Packed  19075 out of 19075 in 59 rows in:     0.046 seconds
 *   Packed  111748 out of 431053 in 1000 rows in: 14.313 seconds
 * 
 * packAlignments2  (row by row)
 *    Packed 17027 out of 17027 in 58 rows:        0.035 seconds
 *    Packed 113061 out of 431053 in 1000 rows in  8.276 seconds
 *:
 * packAlignments2b  (row by row with hash)
 *    Packed 15274 out of 15274 in 58 rows in:     0.011 seconds
 *    Packed 101595 out of 423736 in 1000 rows in: 0.177 seconds
 * 
 * packAlignments3  (priority queue)
 *    Packed 19075 out of 19075 in 63 rows in:      0.044 seconds
 *    Packed 104251 out of 430716 in 1000 rows in:  0.108 seconds
 * @author jrobinso
 */
public class AlignmentPacker {

    private static Logger log = Logger.getLogger(AlignmentPacker.class);
    /** Minimum gap between the end of one alignment and start of another.  */
    public static final int MIN_ALIGNMENT_SPACING = 5;

    /**
     * Allocates each alignment to the rows such that there is no overlap.
     *
     * @param iter  Iterator wrapping the collection of alignments
     * @param showDuplicates  indicates alignments marked "duplicate" are shown
     * @param qualityThreshold  alignment mapQ threshold.  Alignments with lower values are not shown
     * @param maxLevels the maximum number of levels (rows) to create
     */
    public static List<AlignmentTrack.Row> packAlignments(
            CloseableIterator<Alignment> iter,
            boolean showDuplicates,
            int qualityThreshold,
            int maxLevels,
            int end) {

        //return packAlignments1(iter, showDuplicates, qualityThreshold, maxLevels);
        //return packAlignments2(iter, showDuplicates, qualityThreshold, maxLevels);
        return packAlignments2a(iter, showDuplicates, qualityThreshold, maxLevels, end);
    }

    /**
     * Allocate a collection or alignments to levels such that there are no
     * overlaps.  This is the "original" method.
     */
    public static List<AlignmentTrack.Row> packAlignments1(CloseableIterator<Alignment> iter, boolean showDuplicates, int qualityThreshold, int maxLevels) {

        List<AlignmentTrack.Row> alignmentRows = new ArrayList(maxLevels);
        alignmentRows.add(new Row(0));

        // Loop through all alignments
        int totalCount = 0;
        int allocatedCount = 0;
        while (iter.hasNext()) {
            Alignment alignment = iter.next();

            // Filter alignments that will not be displayed.
            if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {
                // Find first row in which this alignment will fit.
                boolean alignmentAllocated = false;
                for (Row row : alignmentRows) {
                    if (alignment.getStart() > (row.lastEnd + MIN_ALIGNMENT_SPACING)) {
                        row.addAlignment(alignment);
                        alignmentAllocated = true;
                        allocatedCount++;
                        break;

                    }
                }
                if (!alignmentAllocated && alignmentRows.size() < maxLevels) {
                    Row newRow = new Row(alignmentRows.size());
                    newRow.addAlignment(alignment);
                    alignmentRows.add(newRow);
                    allocatedCount++;
                }
                totalCount++;

            }

        }
        return alignmentRows;
    }
    /**
     * This method fills each row before proceeding to the next one.
     *
     */

    // Create this array once ("static") and reusue
    static Alignment[] buffer = new Alignment[1000000];

    public static List<AlignmentTrack.Row> packAlignments2(CloseableIterator<Alignment> iter, boolean showDuplicates, int qualityThreshold, int maxLevels) {

        List<AlignmentTrack.Row> alignmentRows = new ArrayList(maxLevels);
        if (!iter.hasNext()) {
            return alignmentRows;
        }

        int maxSize = buffer.length;
        int sz = 0;
        while (iter.hasNext() && sz < maxSize) {
            Alignment alignment = iter.next();
            if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {
                buffer[sz++] = alignment;
            }
        }

        // Initialize variables
        int totalCount = sz;
        int allocatedCount = 0;
        int currentRowEnd = 0;
        Row currentRow = new Row(0);
        alignmentRows.add(currentRow);

        // Make multiple passes through the alignment buffer (array) until all
        // alignments have been allocated.  For each alignment, if it will fit
        // in the current row add it,  otherwise return it to the buffer for
        // the next pass.
        while (sz > 0 && alignmentRows.size() < maxLevels) {

            int newSz = 0;
            for (int i = 0; i < sz; i++) {
                Alignment alignment = buffer[i];
                if (alignment.getStart() >= currentRowEnd) {
                    currentRow.addAlignment(alignment);
                    currentRowEnd = currentRow.lastEnd + MIN_ALIGNMENT_SPACING;
                    allocatedCount++;
                } else {
                    // Skip to the next alignment
                    buffer[newSz++] = alignment;
                }
            }

            if (newSz == 0) {
                // we're done
                break;
            } else {
                // Start new row for unallocated alignments
                sz = newSz;
                currentRow = new Row(alignmentRows.size());
                alignmentRows.add(currentRow);
                currentRowEnd = 0;
            }
        }
        return alignmentRows;

    }

    /**
     * Allocates each alignment to the row with the leftmost current
     * end position.  No preference is given to rows with lower indeces.  This
     * is the fastest algorithm,  but distributes alignments more or less randomly
     * across all levels.
     *
     */
    public static List<AlignmentTrack.Row> packAlignments3(CloseableIterator<Alignment> iter, boolean showDuplicates, int qualityThreshold, int maxLevels) {

        // Create list to hold rows.   
        List<AlignmentTrack.Row> alignmentRows = new ArrayList(maxLevels);


        // Create priority queue based on end position of row.  The row with
        // the "leftmost" end position will be at the top.  If 2 rows have the
        // same end position the one with the lowest row index is given priority.
        PriorityQueue<Row> rowQueue = new PriorityQueue<Row>(maxLevels,
                new Comparator<Row>() {

                    public int compare(Row row1, Row row2) {
                        int endDiff = row1.lastEnd - row2.lastEnd;
                        if (Math.abs(endDiff) > 5) {
                            return endDiff;
                        } else {
                            return row1.idx - row2.idx;
                        }

                    }
                });

        // Initialize queue and row list
        Row firstRow = new Row(0);
        rowQueue.add(firstRow);
        alignmentRows.add(firstRow);

        // Loop through all alignments
        int totalCount = 0;
        int allocatedCount = 0;
        while (iter.hasNext()) {
            Alignment alignment = iter.next();

            // Filter alignments that will not be displayed.
            if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {

                Row row = rowQueue.peek();
                if (alignment.getAlignmentStart() > (row.lastEnd + MIN_ALIGNMENT_SPACING)) {
                    // Alignment "fits".  
                    row.addAlignment(alignment);
                    rowQueue.poll();
                    rowQueue.add(row);
                    allocatedCount++;

                } else if (alignmentRows.size() < maxLevels) {
                    // Won't fit in any existing row, create a new row
                    //System.out.println("new row: " + alignment.getStart());

                    Row newRow = new Row(alignmentRows.size());
                    newRow.addAlignment(alignment);
                    alignmentRows.add(newRow);
                    rowQueue.add(newRow);
                    allocatedCount++;
                }
                totalCount++;
            }

        }
        return alignmentRows;
    }

    /**
     * Basically the same as packAlignments2 but with the buffer replaced by
     * a "hash" to facilitate a direct jump to the next alignment that will
     * fit.
     */
    public static List<AlignmentTrack.Row> packAlignments2a(
            CloseableIterator<Alignment> iter,
            boolean showDuplicates,
            int qualityThreshold,
            int maxLevels,
            int end) {

        List<AlignmentTrack.Row> alignmentRows = new ArrayList(maxLevels);
        if (!iter.hasNext()) {
            return alignmentRows;
        }


        // Compares 2 alignments by length.
        Comparator lengthComparator = new Comparator<Alignment>() {

            public int compare(Alignment row1, Alignment row2) {
                return (row2.getEnd() - row2.getAlignmentStart()) -
                        (row1.getEnd() - row2.getAlignmentStart());

            }
        };


        // Strictly speaking we should loop discarding dupes, etc
        Alignment firstAlignment = iter.next();
        int start = firstAlignment.getAlignmentStart();
        int hashSize = end - start + 1;

        PriorityQueue[] hash = new PriorityQueue[hashSize];
        PriorityQueue firstBucket = new PriorityQueue(5, lengthComparator);
        hash[0] = firstBucket;
        firstBucket.add(firstAlignment);
        int totalCount = 1;

        while (iter.hasNext()) {
            Alignment alignment = iter.next();
            if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {

                int hashCode = alignment.getAlignmentStart() - start;
                /*if(hashCode > hashSize) {
                // grow array
                PriorityQueue[] tmp = new PriorityQueue[hashSize + 1000];
                System.arraycopy(hash, 0, tmp, 0, hashSize);
                hash = tmp;
                hashSize = hash.length;
                }*/
                if (hashCode < hashSize) {
                    PriorityQueue bucket = hash[hashCode];
                    if (bucket == null) {
                        bucket = new PriorityQueue<Alignment>(5, lengthComparator);
                        hash[hashCode] = bucket;
                    }
                    bucket.add(alignment);
                    totalCount++;
                } else {
                    log.debug("Alignment out of bounds: " + alignment.getAlignmentStart() + " (> " + end);
                }
            }
        }

        int allocatedCount = 0;
        int nextStart = start;
        Row currentRow = new Row(0);


        while (allocatedCount < totalCount && alignmentRows.size() < maxLevels) {

            while (nextStart <= end) {
                PriorityQueue<Alignment> bucket = null;
                while (bucket == null && nextStart <= end) {
                    int hashCode = nextStart - start;
                    bucket = hash[hashCode];
                    if (bucket == null) {
                        nextStart++;
                    }
                }

                if (bucket != null) {
                    Alignment alignment = bucket.remove();
                    if (bucket.isEmpty()) {
                        hash[nextStart - start] = null;
                    }
                    currentRow.addAlignment(alignment);
                    nextStart = currentRow.lastEnd + MIN_ALIGNMENT_SPACING;
                    allocatedCount++;

                }
            }

            if (currentRow.alignments.size() > 0) {
                alignmentRows.add(currentRow);
            }
            currentRow = new Row(alignmentRows.size());
            nextStart = start;
        }
        return alignmentRows;

    }

    /**
     * Basically the same as packAlignments2a, but with some details encapsulate
     * in class "PositionHash". 
     */
    public static List<AlignmentTrack.Row> packAlignments2b(
            CloseableIterator<Alignment> iter,
            boolean showDuplicates,
            int qualityThreshold,
            int maxLevels,
            int end) {

        List<AlignmentTrack.Row> alignmentRows = new ArrayList(maxLevels);
        if (!iter.hasNext()) {
            return alignmentRows;
        }


        PositionHash hash = new PositionHash(iter, showDuplicates, qualityThreshold, maxLevels, end);

        int allocatedCount = 0;
        int nextStart = hash.start;
        Row currentRow = new Row(0);

        // Make multiple passes through the alignment buffer (array) until all
        // alignments have been allocated. 
        while (!hash.isEmpty() && alignmentRows.size() < maxLevels) {

            while (nextStart <= end) {
                Alignment alignment = hash.next(nextStart);
                if (alignment == null) {
                    break;
                } else {
                    currentRow.addAlignment(alignment);
                    nextStart = currentRow.lastEnd + MIN_ALIGNMENT_SPACING;
                    allocatedCount++;
                }
            }

            if (currentRow.alignments.size() > 0) {
                alignmentRows.add(currentRow);
            }
            currentRow = new Row(alignmentRows.size());
            nextStart = hash.start;
        }
        return alignmentRows;

    }

    static class PositionHash {

        int totalCount;
        int start;
        int end;
        int bucketCount;
        Queue[] hash;
        Comparator alignmentComparator;

        PositionHash(CloseableIterator<Alignment> iter,
                boolean showDuplicates,
                int qualityThreshold,
                int maxLevels,
                int end) {
            init(iter, end, showDuplicates, qualityThreshold);

        }

        Alignment next(int nextStart) {

            Queue<Alignment> bucket = null;
            while (bucket == null && nextStart <= end) {
                int hashCode = nextStart - start;
                bucket = hash[hashCode];
                if (bucket == null) {
                    nextStart++;
                }
            }

            if (bucket != null) {
                Alignment alignment = bucket.remove();
                if (bucket.isEmpty()) {
                    hash[nextStart - start] = null;
                    bucketCount--;
                }
                return alignment;
            }
            return null;
        }

        Alignment next() {
            return next(start);
        }

        public boolean isEmpty() {
            return bucketCount <= 0;
        }

        private PriorityQueue createBucket() {
            bucketCount++;
            return new PriorityQueue(5, alignmentComparator);
        }

        private void init(CloseableIterator<Alignment> iter, int end, boolean showDuplicates, int qualityThreshold) {
            this.end = end;

            // Compares 2 alignments by length;
            alignmentComparator = new Comparator<Alignment>() {

                public int compare(Alignment row1, Alignment row2) {
                    return (row2.getEnd() - row2.getAlignmentStart()) - (row1.getEnd() - row2.getAlignmentStart());
                }
            };

            totalCount = 0;
            bucketCount = 0;

            // Find the first alignment
            Alignment firstAlignment = null;
            while (firstAlignment == null && iter.hasNext()) {
                Alignment alignment = iter.next();
                if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {
                    firstAlignment = alignment;
                    totalCount++;
                }
            }

            if (firstAlignment == null) {
                return;
            }

            start = firstAlignment.getAlignmentStart();
            int hashSize = end - start + 1;
            hash = new Queue[hashSize];

            Queue<Alignment> firstBucket = createBucket();
            bucketCount++;
            hash[0] = firstBucket;
            firstBucket.add(firstAlignment);

            while (iter.hasNext()) {
                Alignment alignment = iter.next();
                if ((showDuplicates || !alignment.isDuplicate()) && alignment.getMappingQuality() >= qualityThreshold && alignment.isMapped()) {
                    int hashCode = alignment.getAlignmentStart() - start;
                    if (hashCode < hashSize) {
                        Queue<Alignment> bucket = hash[hashCode];
                        if (bucket == null) {
                            bucket = createBucket();
                            hash[hashCode] = bucket;
                        }
                        bucket.add(alignment);
                        totalCount++;
                    }
                }
            }
        }
    }
}
