/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam;

import org.broad.igv.track.WindowFunction;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractAlignment implements Alignment {

    String chr;
    int inferredInsertSize;
    int mappingQuality = 255;  // 255 by default
    ReadMate mate;
    String readName;
    AlignmentBlock[] alignmentBlocks;
    AlignmentBlock[] insertions;
    private boolean negativeStrand;

    public AbstractAlignment() {
    }

    public AbstractAlignment(AbstractAlignment alignment) {
        this.chr = alignment.chr;
        this.inferredInsertSize = alignment.inferredInsertSize;
        this.mappingQuality = alignment.mappingQuality;
        this.mate = alignment.mate;
        this.readName = alignment.readName;
        this.negativeStrand = alignment.negativeStrand;
        this.alignmentBlocks = alignment.alignmentBlocks;
        this.insertions = alignment.insertions;
    }

    public String getChromosome() {
        return chr;
    }

    public String getDescription() {
        return getReadName();
    }


    public ReadMate getMate() {
        return mate;
    }

    public String getReadName() {
        return readName;
    }


    public int getMappingQuality() {
        return mappingQuality;
    }

    public int getInferredInsertSize() {
        return inferredInsertSize;
    }

    public AlignmentBlock[] getAlignmentBlocks() {
        return alignmentBlocks;
    }

    public AlignmentBlock[] getInsertions() {
        return insertions;
    }

    /**
     * @return the negativeStrand
     */
    public boolean isNegativeStrand() {
        return negativeStrand;
    }

    /**
     * @param negativeStrand the negativeStrand to set
     */
    public void setNegativeStrand(boolean negativeStrand) {
        this.negativeStrand = negativeStrand;
    }

    public boolean contains(double location) {
        return location >= getStart() && location < getEnd();
    }

    public String getValueString(double position, WindowFunction windowFunction) {

        // First check insertions.  Position is zero based, block coords 1 based
        for (AlignmentBlock block : this.insertions) {
            double insertionLeft = block.getStart() + .75;
            double insertionRight = block.getStart() + 1.25;
            if (position > insertionLeft && position < insertionRight) {
                return "Insertion: " + new String(block.getBases());
            }
        }

        int basePosition = (int) position - 1;
        StringBuffer buf = new StringBuffer();
        buf.append("Read name = " + getReadName() + "<br>");
        buf.append("Alignment start = " + (getAlignmentStart() + 1) + "<br>");
        buf.append("Cigar = " + getCigarString() + "<br>");
        buf.append("Mapped = " + (isMapped() ? "yes" : "no") + "<br>");
        buf.append("Mapping quality = " + getMappingQuality() + "<br>");
        buf.append("----------------------" + "<br>");

        for (AlignmentBlock block : this.alignmentBlocks) {
            if (block.contains((int) basePosition)) {
                int offset = (int) basePosition - block.getStart();
                byte base = block.getBases()[offset];
                byte quality = block.getQualities()[offset];
                buf.append("Base = " + (char) base + "<br>");
                buf.append("Base phred quality = " + quality + "<br>");
            }
        }

        if (this.isPaired()) {
            buf.append("----------------------" + "<br>");
            buf.append("Pair = " + getMate().toString() + "<br>");
            buf.append("Pair is mapped = " + (getMate().isMapped() ? "yes" : "no") + "<br>");
            //buf.append("Pair is proper = " + (getProperPairFlag() ? "yes" : "no") + "<br>");
            if (this.getChromosome().equals(getMate().mateChr)) {
                buf.append("Insert size = " + getInferredInsertSize() + "<br>");
            }
        }
        buf.append("----------------------");
        return buf.toString();
    }

    public abstract String getCigarString();

    public abstract boolean isMapped();

    public abstract boolean isPaired();

    public abstract boolean isProperPair();

    public float getConfidence() {
        return ((float) getMappingQuality()) / 255;
    }

    public void setConfidence(float c) {
        // required by LocusScore interface, ignored
    }

    public float getScore() {
        return getMappingQuality();
    }

    /**
     * @param mappingQuality the mappingQuality to set
     */
    public void setMappingQuality(int mappingQuality) {
        this.mappingQuality = mappingQuality;
    }

    /**
     * @param inferredInsertSize the inferredInsertSize to set
     */
    public void setInferredInsertSize(int inferredInsertSize) {
        this.inferredInsertSize = inferredInsertSize;
    }

    /**
     * @param mate the mate to set
     */
    public void setMate( // 255 by default
            ReadMate mate) {
        this.mate = mate;
    }
}
