/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/

package org.broad.igv.roc;

//~--- non-JDK imports --------------------------------------------------------

import hep.aida.bin.StaticBin1D;

import org.broad.igv.feature.Feature;
import org.broad.igv.feature.GeneManager;
import org.broad.igv.track.RegionScoreType;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackManager;
import org.broad.igv.track.TrackType;

//~--- JDK imports ------------------------------------------------------------

import java.io.*;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;

import java.util.List;
import java.util.Map;


/**
 *
 * @author jrobinso
 */
public class ROCUtils {

    static int MAX_GENES = 5000;

    String attribute;
    String chr;
    int start;
    int end;
    int zoom;

    List<String> samples;

    Map<String, List<Track>> expressionTracks = new HashMap();


    /**
     * Constructs ...
     *
     *
     * @param attribute
     * @param chr
     * @param start
     * @param end
     * @param zoom
     */
    public ROCUtils(String attribute, String chr, int start, int end, int zoom) {

        {
            this.attribute = attribute;
            this.chr = chr;
            this.start = start;
            this.end = end;
            this.zoom = zoom;
        }
    }


    /**
     * Method description
     *
     *
     * @param threshold
     *
     * @return
     */
    public List<ROCScore> computeScores(float threshold) {

        List<ROCScore> scores = new ArrayList(25000);

        int[] classVector = initCopyNumberMap(threshold);

        Collection<String> chromosomes = GeneManager.getGeneManager("hg18").getChromosomes();
        for (String c : chromosomes)
        {
            if (scores.size() > MAX_GENES)
            {
                break;
            }
            for (Feature gene : GeneManager.getGeneManager("hg18").getGenesForChromosome(c))
            {

                // Feature gene = GeneManager.getGeneManager("hg18").getGene("AASDH");
                int[] cv = new int[classVector.length];
                System.arraycopy(classVector, 0, cv, 0, classVector.length);
                int start = gene.getStart() - 10;
                int end = gene.getEnd() + 10;
                float[] values = computeExpressionValues(gene.getChromosome(), start, end);
                ROC roc = new ROC(cv, values);
                double auc = roc.computeAUC();
                scores.add(new ROCScore(auc, gene.getName()));

                if (scores.size() > MAX_GENES)
                {
                    break;
                }
            }
        }

        Collections.sort(scores);

        return scores;
    }


    // Find copyNumbers that have both cn and expression tracks
    int[] initCopyNumberMap(float threshold) {

        Map<String, StaticBin1D> copyNumbers = new HashMap();
        samples = new ArrayList();

        // Collect set of sample names with at least one gene expression track
        for (Track track : TrackManager.getInstance().getAllTracks(false))
        {
            if ((track.getTrackType() == TrackType.GENE_EXPRESSION) && track.isVisible())
            {
                String sample = track.getAttributeValue(attribute);
                List<Track> eTracks = expressionTracks.get(sample);
                if (eTracks == null)
                {
                    eTracks = new ArrayList();
                    expressionTracks.put(sample, eTracks);
                }
                eTracks.add(track);
            }

        }

        // Initial value bins, one for each sample that (1) is visible, and (2) contains
        // both copy # and expression values.
        for (Track track : TrackManager.getInstance().getAllTracks(false))
        {
            if ((track.getTrackType() == TrackType.COPY_NUMBER) && track.isVisible())
            {
                String sample = track.getAttributeValue(attribute);
                if (expressionTracks.containsKey(sample))
                {
                    StaticBin1D bin = copyNumbers.get(sample);
                    if (bin == null)
                    {
                        bin = new StaticBin1D();
                        copyNumbers.put(sample, bin);
                        samples.add(sample);
                    }
                    bin.add(track.getRegionScore(chr, start, end, zoom,
                                                 RegionScoreType.AMPLIFICATION));
                }
            }

        }


        int[] classVector = new int[samples.size()];
        int n = 0;
        for (String sample : samples)
        {
            double cn = copyNumbers.get(sample).max();
            classVector[n] = (cn > threshold) ? 1 : 0;
            n++;
        }


        return classVector;
    }

    /**
     *
     * @return
     */
    float[] computeExpressionValues(String chr, int start, int end) {

        Map<String, Float> expressionValues = new HashMap(samples.size() * 2);

        for (String sample : samples)
        {
            for (Track track : expressionTracks.get(sample))
            {

                Float val = track.getRegionScore(chr, start, end, zoom, RegionScoreType.EXPRESSION);
                if (expressionValues.containsKey(sample))
                {
                    expressionValues.put(sample, Math.max(val, expressionValues.get(sample)));
                }
                else
                {
                    expressionValues.put(sample, val);
                }

            }
        }

        int n = 0;
        float[] values = new float[samples.size()];
        for (String sample : samples)
        {
            values[n] = expressionValues.get(sample).floatValue();
            n++;
        }


        return values;
    }


}
