/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */



/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.renderer;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.IGVConstants;

import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.LocusScore;

import org.broad.igv.track.DataTrack;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.Track;
import org.broad.igv.ui.FontManager;
import org.broad.igv.ui.MiscStuff;

//~--- JDK imports ------------------------------------------------------------

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Stroke;

import java.text.DecimalFormat;

import java.util.List;

/**
 *
 * @author jrobinso
 */
public abstract class XYPlotRenderer extends DataRenderer {

    protected void drawDataPoint(Color graphColor, int dx, int pX, int baseY, int pY,
                                 RenderContext context) {
        context.getGraphic2DForColor(graphColor).fillRect(pX, pY, dx, 2);

    }

    /**
     * Render the track in the given rectangle.
     *
     * @param track
     * @param locusScores
     * @param context
     * @param arect
     */
    public void renderScores(Track track, List<LocusScore> locusScores, RenderContext context,
                             Rectangle arect) {


        Graphics2D noDataGraphics = context.getGraphic2DForColor(IGVConstants.NO_DATA_COLOR);


        Rectangle adjustedRect = calculateDrawingRect(arect);


        double origin = context.getOrigin();
        double locScale = context.getScale();

        Color posColor = track.getColor();
        Color negColor = track.getAltColor();
        if (negColor == null)
        {
            negColor = posColor;
        }

        // Get the Y axis definition, consisting of minimum, maximum, and base value.  Often
        // the base value is == min value which is == 0.
        DataRange axisDefinition = track.getAxisDefinition();
        float maxValue = axisDefinition.getMaximum();
        float baseValue = axisDefinition.getBaseline();
        float minValue = axisDefinition.getMinimum();

        // Calculate the Y scale factor.
        double yScaleFactor = adjustedRect.getHeight() / (maxValue - minValue);

        // Calculate the Y position in pixels of the base value.
        int baseY = (int) (adjustedRect.getY() + (maxValue - baseValue) * yScaleFactor);

        int lastPx = 0;
        for (LocusScore score : locusScores)
        {

            // Note -- don't cast these to an int until the range is checked.
            // could get an overflow.
            double pX = ((score.getStart() - origin) / locScale);
            double dx = Math.ceil((Math.max(1, score.getEnd() - score.getStart())) / locScale) + 1;
            if ((pX + dx < 0))
            {
                continue;
            }
            else if (pX > adjustedRect.getMaxX())
            {
                break;
            }

            float dataY = score.getScore();

            if (!Float.isNaN(dataY) && (score.getConfidence() > confThreshold))
            {

                // Compute the pixel y location.  Clip to bounds of rectangle.
                int pY = (int) Math.max(adjustedRect.getMinY(),
                                        Math.min(adjustedRect.getMaxY(),
                                                 adjustedRect.getY()
                                                 + (maxValue - dataY) * yScaleFactor));

                Color color = (dataY >= 0) ? posColor : negColor;
                drawDataPoint(color, (int) dx, (int) pX, baseY, pY, context);

            }
            if (MiscStuff.isShowMissingDataEnabled())
            {

                // Draw from lastPx + 1  to pX - 1;
                int w = (int) pX - lastPx - 4;
                if (w > 0)
                {
                    noDataGraphics.fillRect(lastPx + 2, (int) arect.getY(), w,
                                            (int) arect.getHeight());
                }
            }
            if (!Float.isNaN(dataY))
            {

                lastPx = (int) pX + (int) dx;

            }
        }
        if (MiscStuff.isShowMissingDataEnabled())
        {
            int w = (int) arect.getMaxX() - lastPx - 4;
            if (w > 0)
            {
                noDataGraphics.fillRect(lastPx + 2, (int) arect.getY(), w, (int) arect.getHeight());
            }
        }

    }

    static DecimalFormat formatter = new DecimalFormat();

    /**
     * Method description
     *
     *
     * @param track
     * @param context
     * @param arect
     */
    @Override
    public void renderAxis(Track track, RenderContext context, Rectangle arect) {

        // For now disable axes for all chromosome view
        if (context.getChr().equals(IGVConstants.CHR_ALL))
        {
            return;
        }

        super.renderAxis(track, context, arect);

        Rectangle drawingRect = calculateDrawingRect(arect);


        PreferenceManager.ChartPreferences prefs =
            PreferenceManager.getInstance().getChartPreferences();

        Color labelColor = prefs.isColorTrackName() ? track.getColor() : Color.black;
        Graphics2D labelGraphics = context.getGraphic2DForColor(labelColor);

        labelGraphics.setFont(FontManager.getScalableFont(8));

        if (prefs.isDrawTrackName())
        {

            // Only attempt if track height is > 25 pixels
            if (arect.getHeight() > 25)
            {
                Rectangle labelRect = new Rectangle(arect.x, arect.y + 10, arect.width, 10);
                labelGraphics.setFont(FontManager.getScalableFont(10));
                GraphicUtils.drawCenteredText(track.getDisplayName(), labelRect, labelGraphics);
            }
        }

        if (prefs.isDrawAxis())
        {

            Rectangle axisRect = new Rectangle(arect.x, arect.y + 1, AXIS_AREA_WIDTH, arect.height);

            DataRange axisDefinition = track.getAxisDefinition();
            float maxValue = axisDefinition.getMaximum();
            float baseValue = axisDefinition.getBaseline();
            float minValue = axisDefinition.getMinimum();

            // Bottom (minimum tick mark)
            int pY = computeYPixelValue(drawingRect, axisDefinition, minValue);

            labelGraphics.drawLine(axisRect.x + AXIS_AREA_WIDTH - 10, pY,
                                   axisRect.x + AXIS_AREA_WIDTH - 5, pY);
            GraphicUtils.drawRightJustifiedText(formatter.format(minValue),
                    axisRect.x + AXIS_AREA_WIDTH - 15, pY, labelGraphics);

            // Top (maximum tick mark)
            int topPY = computeYPixelValue(drawingRect, axisDefinition, maxValue);

            labelGraphics.drawLine(axisRect.x + AXIS_AREA_WIDTH - 10, topPY,
                                   axisRect.x + AXIS_AREA_WIDTH - 5, topPY);
            GraphicUtils.drawRightJustifiedText(formatter.format(maxValue),
                    axisRect.x + AXIS_AREA_WIDTH - 15, topPY + 4, labelGraphics);

            // Connect top and bottom
            labelGraphics.drawLine(axisRect.x + AXIS_AREA_WIDTH - 10, topPY,
                                   axisRect.x + AXIS_AREA_WIDTH - 10, pY);

            // Middle tick mark.  Draw only if room
            int midPY = computeYPixelValue(drawingRect, axisDefinition, baseValue);

            if ((midPY < pY - 15) && (midPY > topPY + 15))
            {
                labelGraphics.drawLine(axisRect.x + AXIS_AREA_WIDTH - 10, midPY,
                                       axisRect.x + AXIS_AREA_WIDTH - 5, midPY);
                GraphicUtils.drawRightJustifiedText(formatter.format(baseValue),
                        axisRect.x + AXIS_AREA_WIDTH - 15, midPY + 4, labelGraphics);
            }

        }
    }

    /**
     * Method description
     *
     *
     * @param track
     * @param context
     * @param arect
     */
    @Override
    public void renderBorder(Track track, RenderContext context, Rectangle arect) {

        Rectangle adjustedRect = calculateDrawingRect(arect);

        // Draw boundaries if there is room
        if (adjustedRect.getHeight() >= 10)
        {

            // midline
            DataRange axisDefinition = track.getAxisDefinition();
            float maxValue = axisDefinition.getMaximum();
            float baseValue = axisDefinition.getBaseline();
            float minValue = axisDefinition.getMinimum();
            double maxX = adjustedRect.getMaxX();
            double x = adjustedRect.getX();
            double y = adjustedRect.getY();

            if ((baseValue > minValue) && (baseValue < maxValue))
            {
                int baseY = computeYPixelValue(adjustedRect, axisDefinition, baseValue);

                getBaselineGraphics(context).drawLine((int) x, baseY, (int) maxX, baseY);
            }

            PreferenceManager.ChartPreferences prefs =
                PreferenceManager.getInstance().getChartPreferences();

            Color borderColor = prefs.isColorBorders() ? track.getColor() : Color.black;
            Graphics2D borderGraphics = context.getGraphic2DForColor(borderColor);

            // Draw the baseline -- todo, this is a wig track option?
            float zeroValue = axisDefinition.getBaseline();
            int zeroX = computeYPixelValue(adjustedRect, axisDefinition, zeroValue);
            borderGraphics.drawLine(adjustedRect.x, zeroX, adjustedRect.x + adjustedRect.width,
                                    zeroX);

            // If the chart has + and - numbers draw both borders or none. This
            // needs documented somewhere.  
            boolean drawBorders = true;

            if (minValue * maxValue < 0)
            {
                drawBorders = prefs.isDrawBottomBorder() && prefs.isDrawTopBorder();
            }

            if (drawBorders && prefs.isDrawTopBorder())
            {
                borderGraphics.drawLine(adjustedRect.x, adjustedRect.y,
                                        adjustedRect.x + adjustedRect.width, adjustedRect.y);
            }

            if (drawBorders && prefs.isDrawBottomBorder())
            {
                borderGraphics.drawLine(adjustedRect.x, adjustedRect.y + adjustedRect.height,
                                        adjustedRect.x + adjustedRect.width,
                                        adjustedRect.y + adjustedRect.height);
            }
        }
    }

    /**
     * Get a grapphics object for the baseline.
     * TODO -- make the line style settable by the user
     *
     * @param context
     * @return
     */
    private static Graphics2D getBaselineGraphics(RenderContext context) {
        Graphics2D baselineGraphics;
        Stroke thindashed = new BasicStroke(1.0f,    // line width

        /* cap style */
        BasicStroke.CAP_BUTT,

        /* join style, miter limit */
        BasicStroke.JOIN_BEVEL, 1.0f,

        /* the dash pattern */
        new float[] { 8.0f, 3.0f, 2.0f, 3.0f },

        /* the dash phase */
        0.0f);    /* on 8, off 3, on 2, off 3 */

        baselineGraphics = (Graphics2D) context.getGraphic2DForColor(Color.lightGray).create();

        // baselineGraphics.setStroke(thindashed);
        return baselineGraphics;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayName() {
        return "Scatter Plot";
    }

    protected int computeYPixelValue(Rectangle drawingRect, DataRange axisDefinition, float dataY) {

        float maxValue = axisDefinition.getMaximum();
        float minValue = axisDefinition.getMinimum();
        boolean flipAxis = axisDefinition.isFlipAxis();

        double yScaleFactor = drawingRect.getHeight() / (maxValue - minValue);

        // Compute the pixel y location.  Clip to bounds of rectangle.
        // The distince in pixels frmo the data value to the axis maximum
        double delta = (maxValue - dataY) * yScaleFactor;
        double pY = flipAxis ? (drawingRect.getMaxY() - delta) : (drawingRect.getY() + delta);

        return (int) Math.max(drawingRect.getMinY(), Math.min(drawingRect.getMaxY(), pY));
    }

    protected Rectangle calculateDrawingRect(Rectangle arect) {

        double buffer = Math.min(arect.getHeight() * 0.2, 10);
        Rectangle adjustedRect = new Rectangle(arect);
        adjustedRect.y = (int) (arect.getY() + buffer);
        adjustedRect.height = (int) (arect.height - (adjustedRect.y - arect.getY()));


        return adjustedRect;
    }
}
