/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */






package org.broad.igv.renderer;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.data.rnai.RNAIGeneScore;
import org.broad.igv.data.rnai.RNAIHairpinValue;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.track.RenderContext;


import java.awt.Color;
import java.awt.Rectangle;


import java.util.List;
import org.broad.igv.track.Track;

/**
 *
 * @author jrobinso
 */
public class RNAiBarChartRenderer extends XYPlotRenderer {


    private static Color DEFAULT_COLOR = new Color(0, 0, 150);

    /**
     * Render the track in the given rectangle.
     *
     * @param track
     * @param locusScores
     * @param context
     * @param trackRect
     */
    @Override
    public void renderScores(Track track, List<LocusScore> locusScores, RenderContext context,
                             Rectangle trackRect) {


        // Experimental -- adjust the rectangle to maintain a buffer (space) between tracks.  The
        // settings (20% with 10 pixel maximum) should be adjustable.
        double buffer = Math.min(trackRect.getHeight() * 0.2, 10);
        Rectangle drawingRect = new Rectangle(trackRect);
        drawingRect.y = (int) (trackRect.getY() + buffer);
        drawingRect.height = (int) (trackRect.height - (drawingRect.y - trackRect.getY()));

        // The starting location (origin) in bp (base pairs) of the current view
        double origin = context.getOrigin();

        // The locaction scale  in bp / pixel of the current view
        double locScale = context.getScale();

        // The track color (user settable)
        DEFAULT_COLOR = new Color(94, 145, 201);

        
        // The axis definition defines the scale of the y axis (minimum, base, and maximum values)
        DataRange axisDefinition = track.getAxisDefinition();

        // Calculate the certical (Y) position in pixels of the base value.
        int baseY = computeYPixelValue(drawingRect, axisDefinition, axisDefinition.getBaseline());

        for (LocusScore score : locusScores)
        {

            // We'll have to case this score to an RNAiGeneScore here for now.  This is not
            // particularly safe and should be redesigned to eliminate the need to cache.
            RNAIGeneScore rnaiScore = (RNAIGeneScore) score;

            // Note -- don't cast these to an int until the range is checked.
            // could get an overflow.
            double pX = ((score.getStart() - origin) / locScale);
            double dx = Math.ceil((score.getEnd() - score.getStart()) / locScale) + 1;

            // Don't bother drawing if the data is not in range
            if ((pX + dx) >= 0 && (pX <= drawingRect.getMaxX()))
            {
                float dataY = score.getScore();

                // Draw the gene score
                if (!Float.isNaN(dataY))
                {
                    int pY = computeYPixelValue(drawingRect, axisDefinition, dataY);
                    drawDataPoint(new Color(57, 67, 201), (int) dx, (int) pX, baseY, pY,
                                  context);
                }

                // Draw the hairpin values, if any
                if (rnaiScore.getHairpinValues() != null)
                {        
                    // my blue color new Color(195, 211, 237)
                    for (RNAIHairpinValue hValue : rnaiScore.getHairpinValues())
                    {
                        int scoreY = computeYPixelValue(drawingRect,
                                                        axisDefinition,
                                                        hValue.getScoreMean());
                        context.getGraphic2DForColor(new Color(95, 120, 200).brighter()).drawLine(
                            (int) pX, scoreY, ((int)pX + (int)dx)-1, scoreY);
                    }
                }
            }
        }
    }

    @Override

    /**
     * Render a data point,  in this case as a bar.
     */
    protected void drawDataPoint(Color graphColor, int dx, int pX, int baseY, int pY,
                                 RenderContext context) {

        if (pY > baseY)
        {
            context.getGraphic2DForColor(graphColor).fillRect(pX, baseY, dx, pY - baseY);
        }
        else
        {
            context.getGraphic2DForColor(graphColor).fillRect(pX, pY, dx, baseY - pY);
        }

        // }
    }

    /**
     * Return the default color for this renderer.  Unless overriden by the user this color will
     * be used to render the bars.
     *
     * @return
     */
    public Color getDefaultColor() {
        return DEFAULT_COLOR;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayName() {
        return "RNAi BarChart";
    }
}
