/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

package org.broad.igv.renderer;

import org.broad.igv.feature.Feature;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.Mutation;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.Track;
import org.broad.igv.ui.FontManager;
import org.broad.igv.renderer.GraphicUtils;

public class MutationRenderer extends FeatureRenderer {

    private static Logger log = Logger.getLogger(MutationRenderer.class);
    
    public String getDisplayName() {
        return "Mutation";
    }
    static Font font = FontManager.getScalableFont(12);

    /**
     * Note:  assumption is that featureList is sorted by start position.
     * 
     * @param featureList
     * @param g2D
     * @param rect
     * @param originShift
     * @param locScale
     * 
     * 
     */
    public void renderFeatures(List<Feature> featureList, RenderContext context,
            Rectangle trackRectangle, Track track) {

        double origin = context.getOrigin();
        double locScale = context.getScale();

        if (featureList != null && featureList.size() > 0) {


            Rectangle lastRect = null;
            for (Feature feature : featureList) {


                // Note -- don't cast these to an int until the range is checked. 
                // could get an overflow.
                double pixelStart = ((feature.getStart() - 1 - origin) / locScale);
                double pixelEnd = ((feature.getEnd() - origin ) / locScale);

                // If the any part of the feature fits in the 
                // Track rectangle draw it
                if (pixelEnd >= trackRectangle.getX() &&
                        pixelStart <= trackRectangle.getMaxX()) {

                    // Set color used to draw the feature

                    Color color = feature.getColor();
                    Graphics2D g = context.getGraphic2DForColor(color);
                    g.setFont(font);


                    int w = (int) (pixelEnd - pixelStart);
                    if (w < 3) {
                        w = 3;
                        pixelStart--;
                    }

                    int mutHeight = (int) Math.max(1, trackRectangle.getHeight() - 2);
                    int mutY = (int) (trackRectangle.getY() + (trackRectangle.getHeight() - mutHeight) / 2);


                    Rectangle mutRect = new Rectangle((int) pixelStart, mutY, w, mutHeight);

                    if (getOverlayMode() == true) {
                        Graphics2D gRect = (PreferenceManager.getInstance().getColorOverlay() ? g : 
                            context.getGraphic2DForColor(Color.BLACK));
                        gRect.draw(mutRect);
                        mutRect.x--;
                        mutRect.width += 2;
                        gRect.draw(mutRect);

                    } else {
                        g.fill(mutRect);
                        if (lastRect != null && mutRect.intersects(lastRect)) {
                            // Indicate overlapping mutations 
                            Graphics2D g2 = context.getGraphic2DForColor(Color.BLACK);
                            g2.draw(mutRect);

                        }
                    }

                    // Try to draw the new base, if there's room.  If there are any exceptions
                    // fail silently (with a log message).   
                    if (mutRect.getHeight() > 10 && mutRect.getWidth() > 10) {
                        try {
                            String newBase = ((Mutation) feature).getNewBase();
                            if (newBase != null) {
                                Graphics2D gText = context.getGraphic2DForColor(Color.black);
                                font = FontManager.getScalableFont(10);
                                GraphicUtils.drawCenteredText(newBase, mutRect, gText);
                            }

                        } catch (Exception exception) {
                            log.error("Error drawing mutation base ", exception);
                        }
                    }

                    lastRect = mutRect;
                }
            }
        }
    }
}
