/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is suppliåed without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.renderer;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.data.rnai.RNAIGeneScore;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.TrackType;

//~--- JDK imports ------------------------------------------------------------

import java.awt.*;

import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.data.Segment;
import org.broad.igv.track.Track;

/**
 *
 * @author jrobinso
 */
public class HeatmapRenderer extends DataRenderer {

    static double log2 = Math.log(2);
    /**
     * A cache for "low confidence color", which is a function of zoom level.
     */
    static Map<Integer, Color> lowConfColorCache = new Hashtable();

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayName() {
        return "Heatmap";
    }

    /**
     * Return the color indicating a low confdence score as a function of zoom level.  Currently
     * this is only used with RNAi data.
     *
     * @param zoom
     * @return
     */
    private static Color getLowConfColor(int zoom) {
        Color lowConfColor = lowConfColorCache.get(zoom);
        if (lowConfColor == null) {

            int value = 225 - Math.min(70, zoom * 10);
            lowConfColor = new Color(value, value, value);
            lowConfColorCache.put(zoom, lowConfColor);
        }
        return lowConfColor;
    }

    /**
     * Render the data track as a heat map.
     *
     * This method has gotten quite complicated,  most of it from the option to join adjacent
     * copy number segments.
     *
     * @param track
     * @param scores
     * @param context
     * @param rect
     */
    public void renderScores(Track track, List<LocusScore> scores, RenderContext context,
            Rectangle rect) {

        ContinuousColorScale colorScale = track.getColorScale();


        double origin = context.getOrigin();
        double locScale = context.getScale();

        Color bgColor = colorScale.getNoDataColor();

        context.getGraphic2DForColor(bgColor).fill(rect);


        double maxX = rect.getMaxX();
        int maxY = (int) rect.getMaxY();
        int minY = (int) rect.getMinY();
        int height = (int) rect.getHeight();

        int lastPEnd = 0;
        int lastPStart = 0;
        Color lastColor = null;
        for (LocusScore score : scores) {
            if (lastPStart > maxX) {
                break;
            }

            // Note -- don't cast these to an int until the range is checked,
            // otherwise could get an overflow.
            double pStart = Math.round((score.getStart() - origin) / locScale);
            double dx = Math.ceil((score.getEnd() - score.getStart()) / locScale) + 1;
            float dataY = scaleData(track, score.getScore());
            Color graphColor = colorScale.getColor(dataY);
            int px = (int) pStart;
            int w = (int) dx;



            if ((pStart + dx) >= 0 && (lastPStart <= maxX)) {


                // TODo The instanceof test is very very ugly.
                // Refactor  to generalize "confidence" for all datasets
                if (!Float.isNaN(dataY) && ((score instanceof RNAIGeneScore) || (score.getConfidence() > confThreshold))) {
                    if (score instanceof RNAIGeneScore) {
                        RNAIGeneScore rnaiScore = (RNAIGeneScore) score;
                        if (rnaiScore.getConfidence() < 2) {
                            graphColor = getLowConfColor(context.getZoom());
                        }
                    }

                    Graphics2D g2D = context.getGraphic2DForColor(graphColor);
                    if (px < maxX) {
                        if (w <= 1) {
                            g2D.drawLine(px, minY, px, maxY);
                        } else {
                            g2D.fillRect(px, minY, w, height);
                        }
                    }

                    // Segmented copy numbers (score type "segment") can be optionally joined
                    if (score instanceof Segment &&
                            PreferenceManager.getInstance().isJoinAdjacentSegments() &&
                            lastColor != null && (px - lastPEnd) > 1) {

                        int midPoint = (px + lastPEnd) / 2;

                        context.getGraphic2DForColor(lastColor).fillRect(lastPEnd, minY,
                                midPoint - lastPEnd, height);
                        g2D.fillRect(midPoint, minY, px - midPoint, height);

                        // Cross hatch joined area  --- don't do for whole chromosome
                        if (!context.getChr().equals(IGVConstants.CHR_ALL)) {
                            if (px - lastPEnd > 4 && height > 2) {
                                Color c = new Color(0.4f, 0.4f, 0.4f, 0.5f);
                                Graphics2D gLine = context.getGraphic2DForColor(c);
                                int midpoint = minY + height / 2;
                                if (height > 4) {
                                    gLine.drawLine(lastPEnd, minY + 3, lastPEnd, minY + height - 3);
                                    gLine.drawLine(px, minY + 3, px, minY + height - 3);
                                }
                                gLine.drawLine(lastPEnd, minY + height / 2, px - 1, minY + height / 2);
                            }
                        }
                    }
                }

            }
            lastPStart = px;
            lastPEnd = px + w;
            lastColor = graphColor;


        }
    }

    //TODO -- move to graphic utils
    void crossHatch(Graphics2D g, Rectangle rect) {
        System.out.println(rect);
        g.fill(rect);
    //g.drawLine(rect.x, rect.y, rect.x, rect.y);
    //g.drawLine(rect.x + rect.width, rect.y, rect.x + rect.width, rect.y);
    }

    /**
     * Special normalization function for linear (non logged) copy number data
     * @param value
     * @param norm
     * @return
     */
    float getLogNormalizedValue(float value, double norm) {
        if (norm == 0) {
            return Float.NaN;
        } else {
            return (float) (Math.log(Math.max(Float.MIN_VALUE, value) / norm) / log2);
        }
    }

    private float scaleData(Track track, float dataY) {

        // Special case for copy # -- centers data around 2 copies (1 for allele
        // specific) and log normalizes
        if (((track.getTrackType() == TrackType.COPY_NUMBER) || (track.getTrackType() == TrackType.ALLELE_SPECIFIC_COPY_NUMBER)) && !track.isLogNormalized()) {
            double centerValue = (track.getTrackType() == TrackType.ALLELE_SPECIFIC_COPY_NUMBER)
                    ? 1.0 : 2.0;

            dataY = getLogNormalizedValue(dataY, centerValue);
        }


        return dataY;
    }
}
