/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.renderer;

//~--- JDK imports ------------------------------------------------------------
import java.awt.BasicStroke;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 *
 * @author jrobinso
 */
public class GraphicUtils {

    /**
     * Method description
     *
     *
     * @param g
     * @param chars
     * @param x
     * @param y
     * @param w
     * @param h
     */
    public static void drawCenteredChar(Graphics g, char[] chars, int x, int y,
            int w, int h) {

        // Get measures needed to center the message
        FontMetrics fm = g.getFontMetrics();

        // How many pixels wide is the string
        int msg_width = fm.charsWidth(chars, 0, 1);

        // How far above the baseline can the font go?
        int ascent = fm.getMaxAscent();

        // How far below the baseline?
        int descent = fm.getMaxDescent();

        // Use the string width to find the starting point
        int msgX = x + w / 2 - msg_width / 2;

        // Use the vertical height of this font to find
        // the vertical starting coordinate
        int msgY = y + h / 2 - descent / 2 + ascent / 2;

        g.drawChars(chars, 0, 1, msgX, msgY);

    }

    /**
     * Draw a block of text centered in or over the rectangle
     * @param text
     * @param rect
     * @param g
     */
    public static void drawCenteredText(String text, Rectangle rect, Graphics g) {
        drawCenteredText(text, rect.x, rect.y, rect.width, rect.height, g);

    }
    
        public static void drawCenteredText(String text, int x, int y, int w, int h, Graphics g) {
        FontMetrics fontMetrics = g.getFontMetrics();

        Rectangle2D textBounds = fontMetrics.getStringBounds(text, g);
        int xOffset = (int) ((w - textBounds.getWidth()) / 2);
        int yOffset = (int) ((h - textBounds.getHeight()) / 2);
        g.drawString(text, x + xOffset, y + h - yOffset - (int) (textBounds.getHeight() / 4));

    }

    /**
     * Draw a block of text centered verticallyin the rectangle
     * @param text
     * @param rect
     * @param g2D
     */
    public static void drawVerticallyCenteredText(String text, int margin, Rectangle rect,
            Graphics g2D, boolean rightJustify) {
        FontMetrics fontMetrics = g2D.getFontMetrics();
        Rectangle2D textBounds = fontMetrics.getStringBounds(text, g2D);
        int yOffset = (int) ((rect.getHeight() - textBounds.getHeight()) / 2);
        int yPos = (rect.y + rect.height) - yOffset - (int) (textBounds.getHeight() / 4);
        if (rightJustify) {
            drawRightJustifiedText(text, rect.x + rect.width - margin, yPos, g2D);
        } else {
            g2D.drawString(text, margin, yPos);
        }
    }
    /**
     * Draw a block of text centered verticallyin the rectangle
     * @param text
     * @param rect
     * @param g2D
     */
    static Pattern splitPattern = Pattern.compile("[,_\\s]+");

    public static void drawVerticallyCenteredWrappedText(String text, int margin, Rectangle rect,
            Graphics g2D) {
        
        FontMetrics fontMetrics = g2D.getFontMetrics();
        Rectangle2D textBounds = fontMetrics.getStringBounds(text, g2D);
        int yOffset = (int) ((rect.getHeight() - textBounds.getHeight()) / 2);
        int yPos = (rect.y + rect.height) - yOffset - (int) (textBounds.getHeight() / 4);

        if (textBounds.getWidth() > rect.width - margin) {
            List<String> strings = new ArrayList();
            String[] words = splitPattern.split(text);
            String nextString = words[0];
            for (int i = 1; i < words.length; i++) {
                Rectangle2D bounds = fontMetrics.getStringBounds(nextString + words[i], g2D);
                if (bounds.getWidth() > rect.width - margin) {
                    strings.add(nextString);
                    nextString = words[i];
                } else {
                    nextString += words[i];
                }
            }
            strings.add(nextString);

            int gap = 5;
            int nLevels = Math.min(strings.size(), (int) (rect.getHeight() / (textBounds.getHeight() + gap))) + 1;
            yPos -= (nLevels * (textBounds.getHeight() + gap) / 2);

            for(int i=0; i<nLevels; i++) {
                g2D.drawString(strings.get(i), margin, yPos);
                yPos += (int)(textBounds.getHeight() + gap);
            }

        } else {
            g2D.drawString(text, margin, yPos);
        }
    }

    /**
     * Draw a block of text right justified to the given location
     *
     * @param text
     * @param right
     * @param y
     * @param g
     */
    public static void drawRightJustifiedText(String text, int right, int y,
            Graphics g) {
        FontMetrics fontMetrics = g.getFontMetrics();

        Rectangle2D textBounds = fontMetrics.getStringBounds(text, g);
        int x = right - (int) textBounds.getWidth();
        g.drawString(text, x, y);

    }

    public static void drawDashedLine(Graphics2D g, int x1, int y1, int x2,
            int y2) {
        Stroke thindashed = new BasicStroke(1.0f, // line width
                BasicStroke.CAP_BUTT, // cap style
                BasicStroke.JOIN_BEVEL, 1.0f, // join style, miter limit
                 new float[]{8.0f, 3.0f, 2.0f, 3.0f}, // the dash pattern :  on 8, off 3, on 2, off 3
                0.0f);  // the dash phase
        drawDashedLine(g, thindashed, x1,
                y1, x2, y2);

    }

    /**
     * Method description
     *        Stroke thindashed = new BasicStroke(thickness, // line width
                BasicStroke.CAP_BUTT, // cap style
                BasicStroke.JOIN_BEVEL, 1.0f, // join style, miter limit
                dashPattern, // the dash pattern :  on 8, off 3, on 2, off 3
                phase);  // the dash phase

     *
     * @param g
     * @param thickness
     */
    public static void drawDashedLine(Graphics2D g, Stroke stroke,
            int x1, int y1, int x2, int y2) {


        Stroke currentStroke = g.getStroke();
        g.setStroke(stroke);
        g.drawLine(x1, y1, x2, y2);
        g.setStroke(currentStroke);

    }
}
