/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * CytobandRenderer.java
 *
 * Created on September 19, 2007, 5:36 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.renderer;

import org.broad.igv.feature.Cytoband;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.ui.FontManager;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.ViewContext;

/**
 *
 * @author jrobinso
 */
public class CytobandRenderer {

    boolean drawLabels = true;
    static final public int CYTOBAND_Y_OFFSET = 5;
    static final public int LABEL_OFFSET  = 25;
    private static Map<Integer, Color> stainColors =
            new HashMap<Integer, Color>();

    public void draw(List<Cytoband> data, Graphics g2D, Rectangle graphicRect) {

        if (data.size() > 0) {

            // If we are in the process of exporting to an image file
            // we need to write out the cytoband locus on the image
            if (IGVMainFrame.getInstance().isExportingSnapshot()) {
                String locus = IGVMainFrame.getInstance().getDisplayedLocusString();
                if(locus != null) {
                    Graphics g2 = g2D.create();
                    g2.setFont(FontManager.getScalableFont(Font.BOLD, 10));
                    g2.drawString(locus, 3, 11);
                }
            }
            
            // Draw Cytoband
            drawBands(data, g2D, graphicRect);

            // Draw Cytoband Labels
            if (drawLabels) {
                drawLabels(g2D, graphicRect);
            }
        }
    }

    public void drawBands(List<Cytoband> data, Graphics g2D, Rectangle graphicRect) {

        int[] xC = new int[3];
        int[] yC = new int[3];

        double scale =
                ((double) graphicRect.getWidth()) / getViewContext().getChromosomeLength();

        Cytoband firstBand = data.get(0);
        Cytoband lastBand = data.get(data.size() - 1);

        for (Cytoband cytoband : data) {

            int start = (int) (graphicRect.getX() + scale * cytoband.getStart());
            int end = (int) (graphicRect.getX() + scale * cytoband.getEnd());
            int y = (int) graphicRect.getY() + CYTOBAND_Y_OFFSET;
            int height = (int) graphicRect.getHeight();

            if (cytoband.getType() == 'c') { // centermere: "acen"

                int center = (int) (y + height / 2);
                if (cytoband.getName().startsWith("p")) {
                    xC[0] = (int) start;
                    yC[0] = (int) graphicRect.getMaxY() + CYTOBAND_Y_OFFSET;
                    xC[1] = (int) start;
                    yC[1] = y;
                    xC[2] = (int) end;
                    yC[2] = center;
                } else {
                    xC[0] = (int) end;
                    yC[0] = (int) graphicRect.getMaxY() + CYTOBAND_Y_OFFSET;
                    xC[1] = (int) end;
                    yC[1] = y;
                    xC[2] = (int) start;
                    yC[2] = center;
                }
                g2D.setColor(Color.RED.darker());
                g2D.fillPolygon(xC, yC, 3);
            } else {

                g2D.setColor(getCytobandColor(cytoband));
                //int arc = height;
                //if (cytoband == firstBand) {
                //    g2D.fillRoundRect(start, y, arc, (int) height, arc, arc);
                //    g2D.fillRect(start + arc / 2, y, (end - start - arc / 2), height);
                //    g2D.setColor(Color.BLACK);
                //    g2D.drawArc(start , y, 2*height, (int) height, 90, 180);
                //} else if (cytoband == lastBand) {
                //    g2D.fillRect(start, (int) y, (end - start - arc / 2), height);
                //    g2D.fillRoundRect(end - arc, y, arc, height, arc, arc);
                //    g2D.setColor(Color.BLACK);
                //    g2D.drawArc(end - 2*height , y, 2*height, (int) height, 180, 360);

                //} else {
                    g2D.fillRect(start, y, (end - start), height);
                    g2D.setColor(Color.BLACK);
                    g2D.drawRect(start, y, (end - start), height);
                    
                //}
            }
        }
    }

    private void drawLabels(final Graphics g, Rectangle graphicRect) {
        
        double width = graphicRect.getWidth();
        int y = (int) graphicRect.getY() + LABEL_OFFSET;
        
        // Draw labels
        g.setColor(Color.BLACK);
        FontMetrics fm = g.getFontMetrics();
        int minSpacing = 10;
        int prevEnd = 0;
        double sc = width / getViewContext().getChromosomeLength();
        int adjustedY = y ;
        for (Cytoband cytoband: getViewContext().getChromosome().getCytobands()) {
            int s =  (int) (sc * cytoband.getStart());
            int e = (int) (sc * cytoband.getEnd());
            int stringWidth = (int) fm.getStringBounds(cytoband.getName(), g).getWidth();
            int x = (int) (s + (e - s - stringWidth) / 2);
            if(x > (prevEnd + minSpacing)) {
                g.drawString(cytoband.getName(), x, adjustedY);
                prevEnd = x + stringWidth;
            }
        }
    }
    

    private void drawLabelVertical(final Graphics g, Rectangle graphicRect) {

        Graphics2D graphics2 = (Graphics2D) g.create();

        double width = graphicRect.getWidth();
        FontMetrics fontMetrics = graphics2.getFontMetrics();
        int y = (int) graphicRect.getY() +
                graphicRect.height +
                fontMetrics.getHeight() +
                fontMetrics.getMaxDescent();

        // Change the font
        FontManager.applyScalableTextFont(graphics2);


        // Draw labels
        FontMetrics fm = g.getFontMetrics();
        int minSpacing = 30;
        int prevEnd = 0;
        double sc = width / getViewContext().getChromosomeLength();
        int adjustedY = y - fm.getDescent();

        // Set text color
        g.setColor(Color.BLACK);

        for (Cytoband cytoband : getViewContext().getChromosome().getCytobands()) {

            Graphics2D graphics2D = (Graphics2D) g.create();
            Rectangle2D stringBounds =
                    fm.getStringBounds(cytoband.getName(), graphics2D);

            int stringWidth = (int) stringBounds.getWidth();

            int s = (int) (sc * cytoband.getStart());
            int e = (int) (sc * cytoband.getEnd());
            int x = (int) (s + (e - s - stringWidth) / 2);

            if (x > (prevEnd + minSpacing)) {

                // Rotate and Translate text
                Rectangle newOrigin = graphics2D.getClipBounds();
                applyTextTranslationAndRotation(
                        graphics2D, newOrigin.x + x, newOrigin.y + adjustedY);

                graphics2D.drawString(cytoband.getName(), 0, 0);
                prevEnd = x + stringWidth;
            }
            graphics2D.dispose();
        }
    }

    //Refactor -- remove reference to IGVMainFrame
    private ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }

    public void applyTextTranslationAndRotation(Graphics2D graphics2, int x, int y) {

        AffineTransform transform = new AffineTransform();
        transform.translate(x, y);
        transform.rotate(-Math.PI / 2);
        graphics2.transform(transform);
    }

    private static Color getCytobandColor(Cytoband data) {
        if (data.getType() == 'p') { // positive: "gpos100"
            int stain = data.getStain(); // + 4;

            int shade = (int) (255 - stain / 100.0 * 255);
            Color c = stainColors.get(shade);
            if (c == null) {
                c = new Color(shade, shade, shade);
                stainColors.put(shade, c);
            }

            return c;

        } else if (data.getType() == 'c') { // centermere: "acen"
            return Color.PINK;

        } else {
            return Color.WHITE;
        }
    }
}
