/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/



/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.renderer;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.IGVConstants;
import org.broad.igv.ui.util.UIUtilities;

import org.broad.igv.util.ColorUtilities;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;

/**
 *
 * @author jrobinso
 */
public class ContinuousColorScale extends AbstractColorScale {

    /**
     * String to use to identify this clas when serializing.  We could use
     * the class name, but that would invalidate serialized instances if the
     * name was ever changed.
     */
    public static String serializedClassName = "ContinuousColorScale";
    private boolean useDoubleGradient;
    private double negEnd;
    private double posEnd;
    private double negStart;
    private double posStart;
    private Color minColor;
    private Color midColor;
    private Color maxColor;
    private Color[] colors;
    private Color noDataColor = IGVConstants.NO_DATA_COLOR;


    /**
     * Constructs ...
     *
     *
     * @param string
     */
    public ContinuousColorScale(String string) {
        String[] tokens = string.split(";");
        if (tokens.length == 5)
        {
            this.negEnd = Double.parseDouble(tokens[1]);
            this.posEnd = Double.parseDouble(tokens[2]);
            this.minColor = ColorUtilities.convertRGBStringToColor(tokens[3]);
            this.maxColor = ColorUtilities.convertRGBStringToColor(tokens[4]);
            this.useDoubleGradient = false;
        }
        else if (tokens.length == 8)
        {
            this.negStart = Double.parseDouble(tokens[1]);
            this.negEnd = Double.parseDouble(tokens[2]);
            this.posStart = Double.parseDouble(tokens[3]);
            this.posEnd = Double.parseDouble(tokens[4]);
            this.minColor = ColorUtilities.convertRGBStringToColor(tokens[5]);
            this.midColor = ColorUtilities.convertRGBStringToColor(tokens[6]);
            this.maxColor = ColorUtilities.convertRGBStringToColor(tokens[7]);
            this.useDoubleGradient = true;
        }
        else
        {
            throw new RuntimeException("Illegal ColorScale: " + string);
        }
        initColors();


    }



    /**
     * Constructs ...
     *
     *
     * @param min
     * @param max
     * @param minColor
     * @param maxColor
     */
    public ContinuousColorScale(double min, double max, Color minColor, Color maxColor) {
        this.negEnd = min;
        this.posEnd = max;
        this.midColor = minColor;
        this.minColor = minColor;
        this.maxColor = maxColor;
        this.useDoubleGradient = false;
        initColors();
    }

    /**
     * Constructs ...
     *
     *
     * @param min
     * @param mid
     * @param max
     * @param minColor
     * @param midColor
     * @param maxColor
     */
    public ContinuousColorScale(double min, double mid, double max, Color minColor, Color midColor,
                                Color maxColor) {
        this.negEnd = min;
        this.posEnd = max;
        this.negStart = mid;
        this.posStart = mid;
        this.minColor = minColor;
        this.midColor = midColor;
        this.maxColor = maxColor;
        this.useDoubleGradient = true;
        initColors();
    }

    /**
     * Constructs ...
     *
     *
     * @param negStart
     * @param negEnd
     * @param posStart
     * @param posEnd
     * @param minColor
     * @param midColor
     * @param maxColor
     * @param useDoubleGradient
     */
    public ContinuousColorScale(double negStart, double negEnd, double posStart, double posEnd,
                                Color minColor, Color midColor, Color maxColor) {
        this.negEnd = negEnd;
        this.posEnd = posEnd;
        this.negStart = negStart;
        this.posStart = posStart;
        this.minColor = minColor;
        this.midColor = midColor;
        this.maxColor = maxColor;
        this.useDoubleGradient = true;
        initColors();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String asString() {
        StringBuffer buf = new StringBuffer();
        buf.append(serializedClassName + ";");
        if (useDoubleGradient)
        {
            buf.append(String.valueOf(negStart) + ";");
            buf.append(String.valueOf(negEnd) + ";");
            buf.append(String.valueOf(posStart) + ";");
            buf.append(String.valueOf(posEnd) + ";");
            buf.append(ColorUtilities.convertColorToRGBString(minColor) + ";");
            buf.append(ColorUtilities.convertColorToRGBString(midColor) + ";");
            buf.append(ColorUtilities.convertColorToRGBString(maxColor));
        }
        else
        {
            buf.append(String.valueOf(negEnd) + ";");
            buf.append(String.valueOf(posEnd) + ";");
            buf.append(ColorUtilities.convertColorToRGBString(minColor) + ";");
            buf.append(ColorUtilities.convertColorToRGBString(maxColor));
        }
        return buf.toString();
    }

    private void initColors() {

        colors = new Color[25];
        double delta = (posEnd - negEnd) / colors.length;
        if (isUseDoubleGradient())
        {
            ColorGradient csPos = new ColorGradient(posStart, posEnd, midColor, maxColor);
            ColorGradient csNeg = new ColorGradient(negEnd, negStart, minColor, midColor);

            for (int i = 0; i < colors.length; i++)
            {
                double x = getNegEnd() + i * delta;
                if ((x > negStart) && (x < posStart))
                {
                    colors[i] = midColor;
                }
                else if (x <= negStart)
                {
                    colors[i] = csNeg.getColor(x);
                }
                else
                {
                    colors[i] = csPos.getColor(x);
                }
            }

        }
        else
        {

            ColorGradient cs = new ColorGradient(negEnd, posEnd, minColor, maxColor);
            for (int i = 0; i < colors.length; i++)
            {
                double x = getNegEnd() + i * delta;
                colors[i] = cs.getColor(x);
            }
        }


    }

    /**
     * Method description
     *
     *
     * @param val
     *
     * @return
     */
    @Override
    public Color getColor(float val) {
        // See if we are in the midrange.  TO deal with floating point roundoffissues expand the range
        // by a small amount.
        if ((val >= 1.0001*negStart) && (val <= 1.0001*posStart))
        {
            return midColor;
        }
        else
        {
            double f = (val - getNegEnd()) / (getPosEnd() - getNegEnd());
            int index = (int) ((colors.length - 1) * f);
            index = Math.max(0, Math.min(index, colors.length - 1));
            return colors[index];
        }
    }

    /**
     * Method description
     *
     *
     * @param color
     */
    public void setNoDataColor(Color color) {
        this.noDataColor = color;

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getNoDataColor() {
        return noDataColor;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getNegStart() {
        return negStart;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getPosStart() {
        return posStart;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getNegEnd() {
        return negEnd;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getPosEnd() {
        return posEnd;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getMinColor() {
        return minColor;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getMidColor() {
        return midColor;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getMaxColor() {
        return maxColor;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isUseDoubleGradient() {
        return useDoubleGradient;
    }

    static class ColorGradient {

        boolean useDoubleGradient = true;
        double min, max, mid;
        BufferedImage posImage, negImage;

        /**
         * Constructs ...
         *
         *
         * @param min
         * @param mid
         * @param max
         * @param negColor
         * @param neutralColor
         * @param posColor
         */
        public ColorGradient(double min, double mid, double max, Color negColor,
                             Color neutralColor, Color posColor) {
            this.useDoubleGradient = true;
            this.min = min;
            this.max = max;
            this.mid = mid;
            this.posImage = createGradientImage(neutralColor, posColor);
            this.negImage = createGradientImage(negColor, neutralColor);
        }

        /**
         * Constructs ...
         *
         *
         * @param min
         * @param max
         * @param neutralColor
         * @param posColor
         */
        public ColorGradient(double min, double max, Color neutralColor, Color posColor) {
            this.useDoubleGradient = false;
            this.min = min;
            this.max = max;
            posImage = createGradientImage(neutralColor, posColor);
        }

        /**
         * Creates a gradient image given specified <CODE>Color</CODE>(s)
         *
         * @param color1
         *            <CODE>Color</CODE> to display at left side of gradient
         * @param color2
         *            <CODE>Color</CODE> to display at right side of gradient
         * @return returns a gradient image
         */
        private BufferedImage createGradientImage(Color color1, Color color2) {

            BufferedImage image =
                (BufferedImage) java.awt.GraphicsEnvironment.getLocalGraphicsEnvironment()
                    .getDefaultScreenDevice().getDefaultConfiguration()
                    .createCompatibleImage(256, 1);
            Graphics2D graphics = image.createGraphics();
            GradientPaint gp = new GradientPaint(0, 0, color1, 255, 0, color2);

            graphics.setPaint(gp);
            graphics.drawRect(0, 0, 255, 1);
            graphics.dispose();

            return image;
        }

        /**
         * Method description
         *
         *
         * @param value
         *
         * @return
         */
        public Color getColor(double value) {

            int rgb;

            if (useDoubleGradient)
            {

                double maximum = (value < mid) ? this.min : this.max;
                int colorIndex = (int) (255 * (value - mid) / (maximum - mid));

                colorIndex = (colorIndex > 255) ? 255 : colorIndex;
                rgb = (value < mid)
                      ? negImage.getRGB(255 - colorIndex, 0) : posImage.getRGB(colorIndex, 0);

            }
            else
            {

                double span = this.max - this.min;
                int colorIndex = 0;

                if (value <= min)
                {
                    colorIndex = 0;
                }
                else if (value >= max)
                {
                    colorIndex = 255;
                }
                else
                {
                    colorIndex = (int) (((value - this.min) / span) * 255);
                }

                rgb = posImage.getRGB(colorIndex, 0);
            }

            return new Color(rgb);
        }
    }
}
