/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.renderer;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.AminoAcid;
import org.broad.igv.feature.AminoAcidSequence;
import org.broad.igv.feature.Exon;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.Strand;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.Track;
import org.broad.igv.ui.FontManager;

//~--- JDK imports ------------------------------------------------------------

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.font.LineMetrics;
import java.awt.geom.Rectangle2D;

import java.util.List;
import org.broad.igv.track.TrackType;

/**
 * Class description
 *
 *
 * @version    Enter version here..., 08/10/24
 * @author     Enter your name here...
 */
public class BasicFeatureRenderer extends FeatureRenderer {

    static final Color AA_COLOR_1 = new Color(92, 92, 164);
    static final Color AA_COLOR_2 = new Color(12, 12, 120);
    static final Color DULL_BLUE = new Color(0, 0, 200);
    static final Color DULL_RED = new Color(200, 0, 0);
    
    /** Field description */
    public static final int NON_CODING_HEIGHT = 8;
    private static Logger log = Logger.getLogger(BasicFeatureRenderer.class);
    private Font font;
    private boolean drawBoundary = false;

    // Use the max of these values to determine where
    // text should be drawn
    private double lastFeatureLineMaxY = 0;
    private double lastFeatureBoundsMaxY = 0;
    protected double lastRegionMaxY = 0;

    // Constants
    static protected final int NORMAL_STRAND_Y_OFFSET = 14;
    static protected final int POSITIVE_STRAND_Y_OFFSET = 5;
    static protected final int NEGATIVE_STRAND_Y_OFFSET = 15;
    static protected final int ARROW_SPACING = 30;
    static protected final int NO_STRAND_THICKNESS = 2;
    static protected final int REGION_STRAND_THICKNESS = 4;
    final int BLOCK_HEIGHT = 14;

    /**
     * Note:  assumption is that featureList is sorted by pStart position.
     *
     * @param featureList
     * @param context
     * @param trackRectangle
     * @param track
     */
    public void renderFeatures(List<Feature> featureList, RenderContext context,
            Rectangle trackRectangle, Track track) {

        double origin = context.getOrigin();
        double locScale = context.getScale();

        // Clear values
        lastFeatureLineMaxY = 0;
        lastFeatureBoundsMaxY = 0;
        lastRegionMaxY = 0;

        // TODO -- use enum instead of string "Color"
        if ((featureList != null) && (featureList.size() > 0)) {

            // Create a graphics object to draw font names.  Graphics are not cached
            // by font, only by color, so its neccessary to create a new one to prevent
            // affecting other tracks.
            int fontSize = 10;
            font = FontManager.getScalableFont(fontSize);
            Graphics2D fontGraphics =
                    (Graphics2D) context.getGraphic2DForColor(Color.BLACK).create();
            fontGraphics.setFont(font);

            // Track coordinates
            double trackRectangleX = trackRectangle.getX();
            double trackRectangleMaxX = trackRectangle.getMaxX();
            double trackRectangleY = trackRectangle.getY();
            double trackRectangleMaxY = trackRectangle.getMaxY();

            // Draw the lines that represent the bounds of
            // a feature's region
            // TODO -- bugs in "Line Placement" style -- hardocde to fishbone


            int lastFeatureEndedAtPixelX = -9999;
            Feature featureArray[] = featureList.toArray(new Feature[featureList.size()]);
            for (int i = 0; i < featureArray.length; i++) {

                Feature feature = featureArray[i];

                // Get the pStart and pEnd of the entire feature  at extreme zoom levels the
                // virtual pixel value can be too large for an int, so the computation is
                // done in double precision and cast to an int only when its confirmed its
                // within the field of view.



                double virtualPixelStart = Math.round((feature.getStart() - origin) / locScale);
                double virtualPixelEnd = Math.round((feature.getEnd() - origin) / locScale);

                boolean hasRegions = ((feature.getExons() != null) && (feature.getExons().size() > 0));

                // If the any part of the feature fits in the
                // Track rectangle draw it
                if ((virtualPixelEnd >= trackRectangleX) && (virtualPixelStart <= trackRectangleMaxX)) {

                    //
                    int pixelEnd = (int) virtualPixelEnd;
                    int pixelStart = (int) virtualPixelStart;
                    Color color = getFeatureColor(feature, track);


                    Graphics2D g2D = context.getGraphic2DForColor(color);

                    // Get the feature's direction
                    Strand strand = feature.getStrand();

                    // Draw lines
                    drawFeatureBlock(pixelStart, pixelEnd, trackRectangle, strand, hasRegions, g2D);

                    // Determine the y offset of features based on strand type

                    // If the width is < 3 there isn't room to draw the
                    // feature, or orientation.  If the feature has any exons
                    // at all indicate by filling a small rect

                    int yOffset = trackRectangle.y + NORMAL_STRAND_Y_OFFSET / 2;

                    if ((pixelEnd - pixelStart < 3) && hasRegions) {

                        drawVerticalFeatureBounds(pixelStart, pixelEnd, yOffset, g2D);
                    } else {

                        Graphics2D arrowGraphics = hasRegions
                                ? g2D
                                : context.getGraphic2DForColor(Color.WHITE);

                        // Draw the directional arrows
                        drawStrandArrows(feature, pixelStart, pixelEnd, yOffset, arrowGraphics);

                        if (hasRegions) {

                            // Draw the individual block of a feature
                            drawBlockRegions(feature, yOffset, context, g2D, trackRectangle);
                        }

                    }
                    int trackHeight = trackRectangle.height;

                    String name = feature.getName();
                    if (name != null) {
                        LineMetrics lineMetrics = font.getLineMetrics(name,
                                g2D.getFontRenderContext());
                        int fontHeight = (int) Math.ceil(lineMetrics.getHeight());


                        // Draw feature name.  Center it over the feature extent,
                        // or if the feature extends beyond the track bounds over
                        // the track rectangle.
                        int nameStart = Math.max(0, pixelStart);
                        int nameEnd = Math.min(pixelEnd, (int) trackRectangle.getWidth());
                        int textBaselineY = getLastLargestMaxY() + fontHeight;

                        // Calculate the minimum amount of vertical track
                        // space required be we  draw the
                        // track name without drawing over the features
                        int verticalSpaceRequiredForText = textBaselineY - (int) trackRectangleY;

                        if (verticalSpaceRequiredForText <= trackHeight) {

                            lastFeatureEndedAtPixelX = drawFeatureName(feature, nameStart, nameEnd,
                                    lastFeatureEndedAtPixelX, fontGraphics, textBaselineY);
                        }
                    }

                }
            }

            if (drawBoundary) {
                Graphics2D g2D = context.getGraphic2DForColor(Color.LIGHT_GRAY);
                g2D.drawLine((int) trackRectangleX, (int) trackRectangleMaxY - 1,
                        (int) trackRectangleMaxX, (int) trackRectangleMaxY - 1);
            }
        }
    }

    /**
     *
     * @param pixelStart
     * @param pixelEnd
     * @param trackRectangle
     * @param strand
     * @param hasRegions
     * @param g
     * @param drawNormalFeatureLine
     *
     * @return The stroke used to draw the line.
     */
    final private void drawFeatureBlock(int pixelStart, int pixelEnd, Rectangle trackRectangle,
            Strand strand, boolean hasRegions, Graphics2D g) {

        Graphics2D g2D = (Graphics2D) g.create();

        if (!hasRegions) {
            int yOffset = trackRectangle.y + NORMAL_STRAND_Y_OFFSET / 2;

            // No regions so draw and even thicker line
            // g2D.fillRect(pStart, yOffset - BLOCK_HEIGHT / 2, width, BLOCK_HEIGHT);

            g2D.fillRect(pixelStart, yOffset - (BLOCK_HEIGHT - 4) / 2,
                    Math.max(1, pixelEnd - pixelStart), (BLOCK_HEIGHT - 4));
            lastFeatureLineMaxY = yOffset + BLOCK_HEIGHT - 4;
        } else {

            // Draw the line that represents the entire feature
            int yOffset = trackRectangle.y + NORMAL_STRAND_Y_OFFSET / 2;
            float lineThickness = ((BasicStroke) g.getStroke()).getLineWidth();
            if (strand == null) {

                // The stroke used to draw the line
                Stroke stroke = g2D.getStroke();

                // Double the line thickness
                lineThickness *= NO_STRAND_THICKNESS;
                stroke = new BasicStroke(lineThickness);
                g2D.setStroke(stroke);
            }
            g2D.drawLine(pixelStart, yOffset, pixelEnd, yOffset);
            lastFeatureLineMaxY = yOffset + lineThickness;
        }
        g2D.dispose();
    }

    private void drawVerticalFeatureBounds(int pixelStart, int pixelEnd, int yOffset,
            Graphics2D g2D) {

        if (pixelEnd == pixelStart) {
            int yStart = yOffset - BLOCK_HEIGHT / 2;
            g2D.drawLine(pixelStart, yStart, pixelStart, yStart + BLOCK_HEIGHT);
        } else {
            g2D.fillRect(pixelStart, yOffset - BLOCK_HEIGHT / 2, pixelEnd - pixelStart,
                    BLOCK_HEIGHT);
        }

        lastFeatureBoundsMaxY = yOffset + BLOCK_HEIGHT / 2;
    }

    protected void drawBlockRegions(Feature gene, int yOffset, RenderContext context,
            Graphics2D g2D, Rectangle trackRectangle) {

        Graphics exonNumberGraphics = g2D.create();
        exonNumberGraphics.setColor(Color.BLACK);
        exonNumberGraphics.setFont(FontManager.getScalableFont(Font.BOLD, 8));

        // Now get the individual regions of the
        // feature are drawn here

        double theOrigin = context.getOrigin();
        double locationScale = context.getScale();

        Graphics2D fontGraphics = context.getGraphic2DForColor(Color.WHITE);

        for (Exon exon : gene.getExons()) {
            int pStart = getPixelFromChromosomeLocation(exon.getChromosome(), exon.getStart(),
                    theOrigin, locationScale);
            int pEnd = getPixelFromChromosomeLocation(exon.getChromosome(), exon.getEnd(),
                    theOrigin, locationScale);

            if ((pEnd > 0) && (pStart < trackRectangle.getMaxX())) {
                int pCdStart =
                        Math.min(pEnd,
                        Math.max(pStart,
                        getPixelFromChromosomeLocation(exon.getChromosome(),
                        exon.getCdStart(), theOrigin, locationScale)));
                int pCdEnd = Math.max(pStart,
                        Math.min(pEnd,
                        getPixelFromChromosomeLocation(exon.getChromosome(),
                        exon.getCdEnd(), theOrigin, locationScale)));


                if (pCdStart > pStart) {
                    g2D.fillRect(pStart, yOffset - NON_CODING_HEIGHT / 2, pCdStart - pStart,
                            NON_CODING_HEIGHT);
                    pStart = pCdStart;
                }
                if (pCdEnd < pEnd) {
                    g2D.fillRect(pCdEnd, yOffset - NON_CODING_HEIGHT / 2, pEnd - pCdEnd,
                            NON_CODING_HEIGHT);
                    pEnd = pCdEnd;
                }

                if ((exon.getCdStart() < exon.getEnd()) && (exon.getCdEnd() > exon.getStart())) {
                    int width = Math.max(2, pEnd - pStart);
                    g2D.fillRect(pStart, yOffset - BLOCK_HEIGHT / 2, width, BLOCK_HEIGHT);
                    if (PreferenceManager.getInstance().getDrawExonNumbers() && (width > 4)) {

                        // int exonNumber = i + 1;
                        // Rectangle numberRect = new Rectangle(pStart, yOffset + BLOCK_HEIGHT / 2,
                        // pEnd - pStart, 10);
                        // GraphicUtils.drawCenteredChar(String.valueOf(exonNumber), numberRect, exonNumberGraphics);
                    }
                }

                Color c = g2D.getColor();
                g2D.setColor(Color.white);
                drawStrandArrows(gene, pStart + ARROW_SPACING / 2, pEnd, yOffset, g2D);
                g2D.setColor(c);



                if (locationScale < 0.25) {
                    labelAminoAcids(pStart, fontGraphics, theOrigin, context, gene, locationScale,
                            yOffset, exon, trackRectangle);
                }

            }

            // TODO -- eliminate reference to lastRegionMaxY, a base class member.
            //
            lastRegionMaxY = yOffset - BLOCK_HEIGHT / 2 + BLOCK_HEIGHT;

        }

    }

    protected void drawStrandArrows(Feature feature, int pixelStart, int pixelEnd, int yOffset,
            Graphics2D g2D) {

        // Don't draw strand arrows for very small regions
        if ((pixelEnd - pixelStart < 6)) {
            return;
        }

        // Limit drawing to visible region, we don't really know the viewport pEnd,
        int vStart = 0;
        int vEnd = 10000;

        // Draw the directional arrows on the feature
        if (feature.getStrand().equals(Strand.POSITIVE)) {

            for (int i = pixelEnd; i > pixelStart; i -= ARROW_SPACING) {

                if (i < vStart) {
                    break;
                }
                if (i < vEnd) {
                    g2D.drawLine(i, yOffset, i - 3, yOffset + 3);
                    g2D.drawLine(i, yOffset, i - 3, yOffset - 3);
                }

            }
        } else if (feature.getStrand().equals(Strand.NEGATIVE)) {

            // Draw starting line.  Should we be doing this?
            g2D.drawLine(pixelEnd, yOffset + 2, pixelEnd, yOffset - 2);

            for (int i = pixelStart; i < pixelEnd; i += ARROW_SPACING) {
                if (i > vEnd) {
                    break;
                }
                if ((i > vStart) && (i < vEnd)) {
                    g2D.drawLine(i, yOffset, i + 3, yOffset + 3);
                    g2D.drawLine(i, yOffset, i + 3, yOffset - 3);
                }
            }
        }
    }

    final private int drawFeatureName(Feature feature, int pixelStart, int pixelEnd,
            int lastFeatureEndedAtPixelX, Graphics2D g2D,
            int textBaselineY) {

        String name = feature.getName();
        if (name == null) {
            return lastFeatureEndedAtPixelX;
        }

        FontMetrics fm = g2D.getFontMetrics();
        int nameWidth = fm.stringWidth(name);
        Rectangle2D stringBounds = fm.getStringBounds(name, g2D);
        int nameStart = (int) (pixelStart + pixelEnd - nameWidth) / 2;

        if (nameStart > (lastFeatureEndedAtPixelX + 10)) {

            // g2D.clearRect(xString2, textBaselineY, (int) stringBounds.getWidth(), (int) stringBounds.getHeight());
            g2D.drawString(name, nameStart, textBaselineY);
            lastFeatureEndedAtPixelX = nameStart + nameWidth;

        }

        return lastFeatureEndedAtPixelX;
    }

    /**
     * Method description
     *
     *
     * @param pStart
     * @param fontGraphics
     * @param theOrigin
     * @param context
     * @param gene
     * @param locationScale
     * @param yOffset
     * @param exon
     * @param trackRectangle
     */
    public void labelAminoAcids(int pStart, Graphics2D fontGraphics, double theOrigin,
            RenderContext context, Feature gene, double locationScale,
            int yOffset, Exon exon, Rectangle trackRectangle) {
        AminoAcidSequence aaSequence = exon.getAminoAcidSequence();
        if ((aaSequence != null) && aaSequence.hasNonNullSequence()) {
            Rectangle aaRect = new Rectangle(pStart, yOffset - BLOCK_HEIGHT / 2, 1, BLOCK_HEIGHT);


            boolean odd = true;
            int aaSeqStartPosition = aaSequence.getStartPosition();

            if (aaSeqStartPosition > exon.getStart()) {

                // The codon for the first amino acid is split between this and the previous exon
                // TODO -- get base from previous exon and compute aaSequence.  For now skipping drawing
                // the AA label
                int cdStart = Math.max(exon.getCdStart(), exon.getStart());
                aaRect.x = getPixelFromChromosomeLocation(exon.getChromosome(), cdStart, theOrigin,
                        locationScale);
                aaRect.width = getPixelFromChromosomeLocation(exon.getChromosome(),
                        aaSeqStartPosition, theOrigin, locationScale) - aaRect.x;

                if (trackRectangle.intersects(aaRect)) {
                    Graphics2D bgGraphics = context.getGraphic2DForColor(odd
                            ? AA_COLOR_1 : AA_COLOR_2);
                    odd = !odd;
                    bgGraphics.fill(aaRect);
                }
            }


            for (AminoAcid acid : aaSequence.getSequence()) {
                if (acid != null) {

                    int px = getPixelFromChromosomeLocation(exon.getChromosome(),
                            aaSeqStartPosition, theOrigin, locationScale);
                    int px2 = getPixelFromChromosomeLocation(exon.getChromosome(),
                            aaSeqStartPosition + 3, theOrigin, locationScale);

                    if ((px < trackRectangle.getMaxX()) && (px2 > 0)) {

                        // {

                        aaRect.x = px;
                        aaRect.width = px2 - px;


                        Graphics2D bgGraphics = context.getGraphic2DForColor(odd
                                ? AA_COLOR_1 : AA_COLOR_2);
                        odd = !odd;
                        if (((acid.getSymbol() == 'M') && (((gene.getStrand() == Strand.POSITIVE) && (aaSeqStartPosition == exon.getCdStart())) || ((gene.getStrand() == Strand.NEGATIVE) && (aaSeqStartPosition == exon.getCdEnd() - 3))))) {
                            bgGraphics = context.getGraphic2DForColor(Color.green);
                        } else if (acid.getSymbol() == 'X') {
                            bgGraphics = context.getGraphic2DForColor(Color.RED);
                        }

                        bgGraphics.fill(aaRect);

                        String tmp = new String(new char[]{acid.getSymbol()});
                        GraphicUtils.drawCenteredText(tmp, aaRect, fontGraphics);
                    }
                    aaSeqStartPosition += 3;

                }
            }

            if (aaSeqStartPosition < exon.getEnd()) {

                // The last codon is not complete (continues on next exon).
                // TODO -- get base from previous exon and compute aaSequence
                int cdEnd = Math.min(exon.getCdEnd(), exon.getEnd());
                aaRect.x = getPixelFromChromosomeLocation(exon.getChromosome(), aaSeqStartPosition,
                        theOrigin, locationScale);
                aaRect.width = getPixelFromChromosomeLocation(exon.getChromosome(), cdEnd,
                        theOrigin, locationScale) - aaRect.x;
                Graphics2D bgGraphics = context.getGraphic2DForColor(odd ? AA_COLOR_1 : AA_COLOR_2);
                odd = !odd;
                bgGraphics.fill(aaRect);
            }
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayName() {
        return "Basic Feature";
    }

    private Color getFeatureColor(Feature feature, Track track) {
        // Set color used to draw the feature
        Color color = feature.getColor();
        if (color == null) {
            // TODO -- hack, generalize this
            if (track.getTrackType() == TrackType.CNV) {
                color = feature.getName().equals("gain") ? DULL_RED : DULL_BLUE;
            } else {
                // Only used if feature color is not set
                color = track.getColor();
            }
        }
        return color;
    }

    private int getLastLargestMaxY() {

        double largestY = Math.max(lastFeatureLineMaxY,
                Math.max(lastFeatureBoundsMaxY, lastRegionMaxY));

        return (int) Math.ceil(largestY);
    }

    protected int getPixelFromChromosomeLocation(String chr, int chromosomeLocation, double origin,
            double locationScale) {
        return (int) Math.round((chromosomeLocation - origin) / locationScale);
    }
}
