/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * Dataset.java
 *
 * Created on October 29, 2007, 11:29 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.preprocess.old;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.data.Dataset;
import org.broad.igv.util.ObjectCache;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FilenameFilter;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.broad.igv.track.TrackProperties;
import org.broad.igv.track.TrackType;

/**
 *      (new SnpBinaryProcessor(name, ds, genome, chipFormat, wSpan, norm)).process(outputFile);
 * @author jrobinso
 */
public class SnpBinaryDataset implements Dataset {

    private String name;

    private TrackType type = TrackType.COPY_NUMBER;

    private boolean solexaFormat = false;

    private String genome;

    private int wSpan = 1;

    private boolean normalized = false;

    /** Temp variable to cache dataCache array for a single chromosome. */

    // ObjectCache<String, float[]> dataCache;
    ObjectCache<String, int[]> locations = new ObjectCache();

    private boolean intLocations = false;

    String sourceDirectory;

    FilenameFilter filter;

    BinaryReader reader;

    Map<String, ChromosomeFileGroup> fileGroups = new HashMap();

    Set<String> dataHeadings = new LinkedHashSet();

    Set<String> chromosomes = new HashSet();
    
    TrackProperties trackProperties = new TrackProperties();

    public SnpBinaryDataset(
            String name,
            TrackType type,
            String genome,
            String sourceDirectory,
            FilenameFilter filter,
            boolean isSolexa,
            boolean intLocations) {
        this.name = name;
        this.type = type;
        this.genome = genome;
        this.sourceDirectory = sourceDirectory;
        this.filter = filter;
        this.intLocations = intLocations;
        this.solexaFormat = isSolexa;
        reader = new BinaryReader();
        reader.setLittleEndian(isSolexa);
        initialize();
        validate();
    }

    public String getName() {
        return name;
    }

    public TrackType getType() {
        return type;
    }

    public String[] getChromosomes() {
        String[] chrs = new String[chromosomes.size()];
        int i = 0;

        for (String chr : chromosomes) {
            chrs[i] = chr;
            i++;
        }

        return chrs;
    }

    // TODO - return Snp ids
    public String[] getFeatureNames(String chr) {
        return null;
    }

    /**
     * Return the dataCache headings (e.g. sample names) for this dataset.
     * @return dataCache headings
     */
    public String[] getDataHeadings() {
        String[] headings = new String[dataHeadings.size()];
        int i = 0;

        for (String h : dataHeadings) {
            headings[i] = h;
            i++;
        }

        return headings;
    }

    /**
     * Scan the dataCache directory and create ChromosomeFileGroup objects for each
     * chromosome. These objects are later used to load the appropriate dataCache for
     * a chromosome.
     */
    private void initialize() {
        List<File> files = reader.getDataFiles(sourceDirectory, filter);

        for (File file : files) {
            String[] tokens = file.getName().split("\\.");

            // Get the chromosome identifier. String the "chr" prefix (first 3
            // characters).  
            String chr = tokens[0];

            chromosomes.add(chr);

            ChromosomeFileGroup fg = getChromosomeDataFileGroup(chr);
            String tmp = tokens[1];

            if (tmp.equals("snpid")) {
                fg.setMarkerIdFile(file);
            } else if (tmp.equals("snplocation") || tmp.equals("snplocations")) {
                fg.setLocationFile(file);
            } else {
                String arrayName = tmp;
                int i = 2;

                while (!(tokens[i].equals("bin") || tokens[i].equals("cn"))) {
                    arrayName += "." + tokens[i];
                    i++;
                }

                fg.addDataFile(arrayName, file);
                dataHeadings.add(arrayName);
            }
        }

    }

    private void validate() {
        boolean valid = true;
        for (ChromosomeFileGroup fg : fileGroups.values()) {
            valid = fg.validate() && valid;
        }
        if (!valid) {
            System.err.println("Validation failed.  Exiting");
            System.exit(-1);
        }
    }

    /**
     * Return a a ChromosomeFileGroup object for the specified chromosome.
     * Create one if it does not alread exist.
     *
     * @param chr
     * @return
     */
    private ChromosomeFileGroup getChromosomeDataFileGroup(String chr) {
        ChromosomeFileGroup fg = fileGroups.get(chr);

        if (fg == null) {
            ChromosomeFileGroup.LocationType locType = intLocations ? ChromosomeFileGroup.LocationType.INT : ChromosomeFileGroup.LocationType.LONG;
            fg = new ChromosomeFileGroup(locType);
            fileGroups.put(chr, fg);
        }

        return fg;
    }

    public String getGenome() {
        return genome;
    }

    public int[] getStartLocations(String chr) {
        int[] start = locations.get(chr);

        if (start == null) {

            // TODO -- handle chip format
            ChromosomeFileGroup fg = this.getChromosomeDataFileGroup(chr);
            File locationFile = fg.getLocationFile();

            if (intLocations) {
                start = reader.readInts(locationFile);
            } else {
                long[] longs = reader.readLongs(locationFile);

                start = new int[longs.length];

                int count = 0;
                for (int i = 0; i < longs.length; i++) {
                    if (solexaFormat == false || longs[i] >= 0) {
                        start[count] = (int) longs[i];
                        count++;
                    }
                }
                //If the count is less than the array size reduce
                if (count != start.length) {
                    int[] tmp = new int[count];
                    System.arraycopy(start, 0, tmp, 0, count);
                    start = tmp;
                }

            }

            locations.put(chr, start);
        }
        return start;
    }

    public int[] getEndLocations(String chr) {
        return null;
    }

    public float[] getData(String heading, String chr) {
        ChromosomeFileGroup fg = this.getChromosomeDataFileGroup(chr);
        File dataFile = fg.getDataFile(heading);
        if (dataFile == null) {
            System.out.println("No data file for sample: " + heading +
                    "  chromosome: " + chr);
            return null;
        } else {
            float[] data = reader.readFloats(dataFile);
            if (solexaFormat == false) {
                return data;
            } else {
                // Get the start length
                int nonZeroLength = getStartLocations(chr).length;
                float[] nonZeroData = new float[nonZeroLength];
                int count = 0;
                for (int i = 0; i < data.length; i++) {
                    if (data[i] > 0) {
                        nonZeroData[count] = data[i];
                    }

                }
                if (count != nonZeroData.length) {
                    System.err.println("Unexpected data length.  Heading = " + heading + " Chr = " + chr +
                            " expected: " + nonZeroLength + " actual: " + count);
                }
                return nonZeroData;
            }

        }
    }

    public int getWindowSpan() {
        return wSpan;
    }

    public void setWindowSpan(int wSpan) {
        this.wSpan = wSpan;
    }

    public boolean isLogNormalized() {
        return normalized;
    }

    public void setNormalized(boolean normalized) {
        this.normalized = normalized;
    }

    public boolean isLogValues() {
        // todo - implement
        return false;
    }

    public void setName(String name) {
        this.name = name;
    }

    public float getDataMin() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public float getDataMax() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public TrackProperties getTrackProperties() {
        return trackProperties;
    }
}
