/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.preprocess.old;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.data.Dataset;
import org.broad.igv.data.GCTDataset;
import org.broad.igv.data.GCTDatasetParser;
import org.broad.igv.data.IGVDataset;
import org.broad.igv.data.ProcessingUtils;
import org.broad.igv.data.WiggleParser;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackType;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.net.URL;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.broad.igv.feature.ParsingUtils;

/**
 *
 * @author jrobinso
 */
public class IGVPreprocessor {

    static Logger log = Logger.getLogger(IGVPreprocessor.class);

    /**
     * Example:
     *    IGVPreprocessor input.file  output.file.h5 spombe
     * @param args
     *
     * @throws Exception
     *
     *
     */
    public static void main(String args[]) throws Exception {

        LinkedHashMap<String, String> argsMap = ProcessingUtils.parseArgs(args);

        //initGenomeMap();

        String inputFileName = args[0];
        String outputFileName = args[1];
        String genomeId = args[2];

        String genomeListUrl = argsMap.get("-genomeList");
        initGenomeMap(genomeListUrl);

        boolean regevLab = argsMap.containsKey("-regev");
        String span = argsMap.get("-windowWidth");
        String maxZoomLevel = argsMap.get("-maxZoomLevel");


        int maxZoom = 6;
        if ((maxZoomLevel != null) && (maxZoomLevel.length() > 0)) {
            try {
                maxZoom = Integer.parseInt(maxZoomLevel);
            } catch (NumberFormatException numberFormatException) {
                System.err.println(numberFormatException.getMessage());
            }
        }

        String typeString = argsMap.get("-type");

        TrackType type = TrackType.UNKNOWN;
        if (typeString != null) {
            try {
                type = TrackType.valueOf(typeString);

            } catch (Exception exception) {
            }
        }

        final String filterString = argsMap.get("-filter");

        //String genomeUrl = genomeUrlMap.get(genomeId);
        //if (genomeUrl == null) {
        //    System.err.println("Unknown genome id: " + genomeId);
        //    System.exit(-1);
        //}

        GenomeManager.getInstance().findGenomeAndLoad(genomeId);

        File inputFile = new File(inputFileName);
        File outputFile = new File(outputFileName);


        if (inputFile.isDirectory() && !outputFile.isDirectory()) {
            System.err.println("Input file is a directory but output file is not.");
            System.exit(-1);

        }
        if (outputFile.isDirectory() && !inputFile.isDirectory()) {
            outputFile = new File(outputFile, inputFile.getName() + ".h5");
        }


        if (inputFile.isDirectory()) {
            for (File iFile : inputFile.listFiles(new FileFilter() {

                public boolean accept(File arg0) {
                    return filterString == null || arg0.getAbsolutePath().endsWith(filterString);
                }
            })) {
                File oFile = new File(outputFile, iFile.getName() + ".h5");
                processFile(iFile, oFile, genomeId, type, maxZoom, regevLab);
            }
        } else {
            processFile(inputFile, outputFile, genomeId, type, maxZoom, regevLab);
        }

    }

    private static void processFile(File inputFile, File outputFile, String genomeId,
            TrackType type, int maxZoom, boolean regevLab) {

        //if (!outputFile.getName().endsWith(".h5"))
        //{
        //    outputFile = new File(outputFile.getAbsolutePath() + ".h5");
        //}
        String name = inputFile.getName();

        if (regevLab) {
            System.out.println("<Resource name=\"" + name + " (" + genomeId + ")\"");
            System.out.println("path=" + "\"" + outputFile.getAbsolutePath() + "\"");
            System.out.println("serverURL=\"http://www.broadinstitute.org/webservices/igv\"/>");
        } else {
            System.out.println("Processing " + inputFile.getAbsolutePath() + " -> " +
                    outputFile.getAbsolutePath());
        }


        AbstractProcessor processor = null;

        String shortFileName = inputFile.getName().toLowerCase();
        if (shortFileName.endsWith(".txt")  || shortFileName.endsWith(".xls")) {
            shortFileName = shortFileName.substring(0, shortFileName.lastIndexOf("."));
        }

        if (shortFileName.endsWith(".gct") || shortFileName.endsWith(".res") ||
                inputFile.getName().endsWith(".tab")) {
            processor = processGCT(inputFile, genomeId, name, type);

        } else if (shortFileName.endsWith(".cn") || shortFileName.endsWith(".xcn") || shortFileName.endsWith(".snp")) {
            processor = processCN(inputFile, genomeId, name, type);

        } else if (shortFileName.endsWith(".igv")) {
            processor = processIGV(inputFile, genomeId, name, type);
        } else if (shortFileName.endsWith(".wig")) {
            String dsName = inputFile.getName().replace(".wig", "");
            processor = processWIG(dsName, genomeId, inputFile);
        } else {
            log.error("Unrecognized input file extension: " + name);
            System.exit(-1);
        }



        if (processor != null) {
            processor.setZoomMax(maxZoom);
            processor.process(outputFile.getAbsolutePath());

        }
    }

    private static AbstractProcessor processGCT(File inputFile, String genomeId, String name,
            TrackType type) {

        if (type == TrackType.UNKNOWN) {
            type = TrackType.GENE_EXPRESSION;
        }

        GCTDatasetParser parser = new GCTDatasetParser(inputFile, null, genomeId);
        Dataset ds = parser.parse();
        ds.setName(name);
        ((GCTDataset) ds).setType(type);
        ((GCTDataset) ds).setNormalized(true);
        ((GCTDataset) ds).setLogValues(true);
        return new OverlappingProcessor(ds, null);
    }

    private static AbstractProcessor processCN(File inputFile, String genomeId, String name,
            TrackType type) {
        if (type == TrackType.UNKNOWN) {
            type = TrackType.COPY_NUMBER;
        }
        IGVDataset ds = new IGVDataset(genomeId, inputFile);
        ds.setName(name);
        ds.setTrackType(type);
        return new NonOverlappingProcessor(ds, null);
    }

    private static AbstractProcessor processIGV(File inputFile, String genomeId, String name,
            TrackType type) {
        if (type == TrackType.UNKNOWN) {
            type = TrackType.OTHER;
        }

        IGVDataset ds = new IGVDataset(genomeId, inputFile);
        ds.setName(name);
        ds.setTrackType(type);
        return new OverlappingProcessor(ds, null);
    }

    private static AbstractProcessor processCNBin(File inputFile, String genomeId, String name,
            TrackType type, boolean chipFormat, boolean intLocations, int wSpan,
            final String filterString) {


        FilenameFilter filter = new FilenameFilter() {

            public boolean accept(File dir, String name) {
                return name.endsWith(".bin") && (!name.contains("_random") && ((filterString == null) || name.contains(filterString)));
            }
        };

        Dataset ds2 = null;
        System.out.println("chipFormat = " + chipFormat);
        if (chipFormat) {
            SolexaBinaryDataset ds = new SolexaBinaryDataset(name, type, genomeId,
                    inputFile.getAbsolutePath(), filter);
            ds.setWindowSpan(wSpan);
            ds.setNormalized(false);
            ds2 = ds;
        } else {
            SnpBinaryDataset ds = new SnpBinaryDataset(name, type, genomeId,
                    inputFile.getAbsolutePath(), filter, chipFormat,
                    intLocations);
            ds.setWindowSpan(wSpan);
            ds.setNormalized(false);
            ds2 = ds;
        }

        for (String nm : ds2.getDataHeadings()) {
            System.out.println(nm);
        }

        return new NonOverlappingProcessor(ds2);
    }

    private static AbstractProcessor processWIG(String dsName, String genomeId, File inputFile) {

        AbstractProcessor processor;
        Dataset ds = (new WiggleParser(new ResourceLocator(inputFile.getAbsolutePath()),
                genomeId)).parse();
        ds.setName(dsName);
        processor = new OverlappingProcessor(ds, null);

        return processor;
    }
    static Map<String, String> genomeUrlMap;

    private static void initGenomeMap(String genomeListUrl) {
        genomeUrlMap = new HashMap();
        BufferedReader reader = null;
        try {
            if (genomeListUrl == null) {
                genomeListUrl = "http://www.broadinstitute.org/igv/resources/genomes/genomes.txt";
            }

            reader = ParsingUtils.openBufferedReader(genomeListUrl);

            String nextLine = null;
            while ((nextLine = reader.readLine()) != null) {
                String[] tokens = nextLine.split("\t");
                if (tokens.length >= 2) {
                    String url = tokens[1].trim();
                    String key = tokens[2].trim();
                    genomeUrlMap.put(key, url);
                }
            }
        } catch (IOException ex) {
            log.error("Error initializing genome map.", ex);
            System.exit(-1);
        } finally {
            try {
                reader.close();
            } catch (IOException ex) {
                log.error("Error closing genome list reader", ex);
            }
        }
    }

    private static void initGenomeMap() {
        genomeUrlMap = new HashMap();
        BufferedReader reader = null;
        try {

            URL genomeListURL = new URL("http://www.broadinstitute.org/igv/resources/genomes/genomes.txt");
            URLConnection connection = genomeListURL.openConnection();
            reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));

            String nextLine = null;
            while ((nextLine = reader.readLine()) != null) {
                String[] tokens = nextLine.split("\t");
                if (tokens.length > 2) {
                    String url = tokens[1].trim();
                    String key = tokens[2].trim();
                    genomeUrlMap.put(key, url);
                }

            }
        } catch (IOException ex) {
            System.err.println("Error initializing genome map.");
            ex.printStackTrace();
            System.exit(-1);
        } finally {
            try {
                reader.close();
            } catch (IOException ex) {
            }
        }
    }

}
