/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and openFile the template in the editor.
 */
package org.broad.igv.preprocess.old;

//~--- non-JDK imports --------------------------------------------------------
import cern.colt.list.DoubleArrayList;

import cern.jet.stat.quantile.DoubleQuantileFinder;
import cern.jet.stat.quantile.QuantileFinderFactory;

import ncsa.hdf.hdf5lib.HDF5Constants;

import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;

import org.broad.igv.data.DataStatistics;
import org.broad.igv.data.Dataset;
import org.broad.igv.data.GenomeSummaryData;
import org.broad.igv.data.ProcessingUtils;

import org.broad.igv.feature.Chromosome;
import org.broad.igv.feature.Genome;

import org.broad.igv.feature.GenomeManager;
import org.broad.igv.h5.HDFWriter;
import org.broad.igv.renderer.RendererFactory;
import org.broad.igv.renderer.RendererFactory.RendererType;
import org.broad.igv.track.TrackProperties;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVModel;

import org.broad.igv.util.ColorUtilities;

import static org.broad.igv.IGVConstants.*;

//~--- JDK imports ------------------------------------------------------------


import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.broad.igv.h5.HDF5LocalWriter;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractProcessor {

    private static Logger log = Logger.getLogger(AbstractProcessor.class);
    private static int version = 3;
    Dataset dataset;
    private Genome genome;
    private int tileSize = 700;
    private int zoomMax = 10;
    private int zoomMin = 2;
    Set<String> chromosomeGroupCache = new HashSet<String>();
    Set<String> zoomGroupCache = new HashSet<String>();
    Set<String> datasetCache = new HashSet<String>();
    static ProcessingUtils procUtils = new ProcessingUtils();
    GenomeSummaryData genomeSummaryData;
    StatusMonitor statusMonitor;
    HDFWriter writer;
    boolean inferZeroes = false;
    Set<WindowFunction> windowFunctions;

    /**
     * Constructs ...
     *
     *
     * @param dataset
     * @param statusMonitor
     */
    public AbstractProcessor(Dataset dataset, StatusMonitor statusMonitor) {
        this(dataset);
        this.statusMonitor = statusMonitor;

        // Hardcoded  TODO -- pass these in
        windowFunctions = new HashSet();
        windowFunctions.add(WindowFunction.median);
        windowFunctions.add(WindowFunction.mean);
        windowFunctions.add(WindowFunction.median);
        windowFunctions.add(WindowFunction.percentile90);
        windowFunctions.add(WindowFunction.percentile10);
        windowFunctions.add(WindowFunction.max);
    }

    /**
     * Constructs ...
     *
     *
     * @param dataset
     */
    public AbstractProcessor(Dataset dataset) {
        this.dataset = dataset;
        String genomeId = dataset.getGenome();
        if (genomeId == null) {
            IGVModel.getInstance().getViewContext().getGenomeId();
        }
        genomeId = dataset.getGenome();
        genome = GenomeManager.getInstance().getGenome(genomeId);
        if (genome == null) {
            throw new RuntimeException("Unknown genome: " + genomeId);
        }
        genomeSummaryData = new GenomeSummaryData(genome);
    }

    /**
     * Method description
     *
     *
     * @param outputFile
     *
     * @return
     */
    public boolean process(String outputFile) {
        writer = new HDF5LocalWriter(); // new H5BinWriter(new File(outputFile)); //new ZipWriter(); //
        int file = writer.createFile(outputFile);
        try {

            // openFile root group
            int root = writer.openGroup(file, "/");
            writeRootAttributes(root);

            // Process features, and as a side effect compute bin information
            double estTimeFraction = 0.1;    // Estimate of total time spent processing features
            int featureGroup = writer.createGroup(root, "features");
            Map<String, List<BinnedData>> binInfoMap = processFeatures(featureGroup,
                    estTimeFraction);
            writer.closeGroup(featureGroup);

            int dataGroup = writer.createGroup(root, "data");

            // Process data
            // If the "name" attribute is set in a track line, and there is a single track in the
            // dataset,  use this as the name of the track.  This special case is to accomodate
            // wig files.
            TrackProperties tp = dataset.getTrackProperties();
            if ((tp.getName() != null) && (dataset.getDataHeadings().length == 1)) {
                writer.createAndWriteStringDataset(dataGroup, "track.id",
                        new String[]{tp.getName()});
            } else {
                writer.createAndWriteStringDataset(dataGroup, "track.id",
                        dataset.getDataHeadings());
            }

            estTimeFraction = 0.7;
            processData(dataGroup, binInfoMap, estTimeFraction);
            writer.closeGroup(dataGroup);

            // cleanup
            writer.closeGroup(root);


            return true;

        } catch (InterruptedException ex) {
            System.out.println("Thread interrupted" + ex.getMessage());

            // TODO -- cleanup
            return false;

        } finally {
            try {
                writer.closeFile(file);
            } catch (Exception e) {
                if (!Thread.interrupted()) {

                    // log.error(e);
                }
            }
        }
    }

    /**
     * Check the tread for interrupt status.  Used to support cancelling
     */
    private void checkForInterrupt() throws InterruptedException {
        if (Thread.interrupted()) {
            System.out.println("Interrupted");
            throw new InterruptedException();
        }
    }

    /**
     *
     * @param featureGroup
     * @return
     */
    private Map<String, List<BinnedData>> processFeatures(int featureGroup, double estTimeFraction)
            throws InterruptedException {

        double progIncrement = (estTimeFraction * 100) / dataset.getChromosomes().length;

        Map<String, List<BinnedData>> binInfoMap = new HashMap();
        for (String chr : dataset.getChromosomes()) {

            checkForInterrupt();

            List<BinnedData> binnedData = processFeaturesForChromosome(chr, featureGroup);
            if (binnedData != null) {
                binInfoMap.put(chr, binnedData);
            }

            if (statusMonitor != null) {
                statusMonitor.incrementStatus(progIncrement);
            }


        }
        binInfoMap.put(CHR_ALL, processFeaturesForChromosome(CHR_ALL, featureGroup));

        return binInfoMap;
    }

    /**
     *
     * @param chr
     * @param featureGroup
     * @return
     */
    private List<BinnedData> processFeaturesForChromosome(String chr, int featureGroup)
            throws InterruptedException {

        // Chromosome c = genome.getChromosome(chr);
        int maxLength = 0;

        if (chr.equals(CHR_ALL)) {
            int chrLength = (int) (genome.getLength() / 1000);
            maxLength = chrLength;
        } else {
            Chromosome c = genome.getChromosome(chr);


            if (c == null) {
                System.out.println("Missing chromosome: " + chr);
                return null;
            }
            System.out.println("Processing chromosome: " + chr);
            int chrLength = c.getLength();
            int[] locs = dataset.getEndLocations(chr);
            if (locs == null) {
                locs = dataset.getStartLocations(chr);
            }

            maxLength = Math.max(chrLength, locs[locs.length - 1]);

            if (maxLength > chrLength + 1) {
                log.info("Warning: " + chr + " max end location (" + maxLength + " exceeds chr length (" + chrLength + ") Wrong genome?");
            }
        }


        int chrGroup = writer.createGroup(featureGroup, chr);

        writer.writeAttribute(chrGroup, "length", maxLength);

        List<BinnedData> binnedDataList = computeAllBins(chr, maxLength);

        // Record maximum zoom level
        int numZoomLevels = (binnedDataList.size() == 0)
                ? 0 : binnedDataList.get(binnedDataList.size() - 1).getZoomLevel() + 1;

        writer.writeAttribute(chrGroup, "zoom.levels", numZoomLevels);

        for (BinnedData binnedData : binnedDataList) {

            checkForInterrupt();


            String zoomName = "z" + binnedData.getZoomLevel();
            int zoomGroup = writer.createGroup(chrGroup, zoomName);

            writer.writeAttribute(zoomGroup, "bin.size", binnedData.getBinSize());

            double tileWidth = tileSize * binnedData.getBinSize();

            writer.writeAttribute(zoomGroup, "tile.width", tileWidth);

            // TODO  mean.count, data.count,  max.count
            writer.writeAttribute(zoomGroup, "mean.count", binnedData.getMeanCount());
            writer.writeAttribute(zoomGroup, "median.count", binnedData.getMedianCount());
            writer.writeAttribute(zoomGroup, "max.count", binnedData.getMaxCount());
            writer.writeAttribute(zoomGroup, "percentile90.count",
                    binnedData.getPercentile90Count());

            // Record bin starting startLocations
            int[] locations = binnedData.getLocations();

            writer.createAndWriteVectorDataset(zoomGroup, "start", locations);

            // Record boundary indices (bin number) for each tile
            int[] tileBoundaries = binnedData.getTileBoundaries();

            writer.createAndWriteVectorDataset(zoomGroup, "tile.boundary", tileBoundaries);

            // Record # pts for each bin
            float[] ptsPerBin = binnedData.getCounts();

            writer.createAndWriteVectorDataset(zoomGroup, "count", ptsPerBin);
            writer.closeGroup(zoomGroup);
        }

        // Record unprocessed coordinates

        int rawGroup = writer.createGroup(chrGroup, "raw");

        int[] startLocations = getStartLocationsForChromosome(chr);
        writer.createAndWriteVectorDataset(rawGroup, "start", startLocations);

        int[] endLocations = getEndLocationsForChromosome(chr);
        if (endLocations != null) {
            writer.createAndWriteVectorDataset(rawGroup, "end", endLocations);
            int longestFeature = 0;
            for (int i = 1; i < startLocations.length; i++) {
                longestFeature = Math.max(endLocations[i] - startLocations[i], longestFeature);
            }
            writer.writeAttribute(rawGroup, "longest.feature", new Integer(longestFeature));
        }


        String[] featureNames = dataset.getFeatureNames(chr);

        if (featureNames != null) {
            writer.createAndWriteStringDataset(rawGroup, "feature", featureNames);
        }

        assert (maxLength < Integer.MAX_VALUE);
        recordRawIndex(rawGroup, (int) maxLength, startLocations);
        writer.closeGroup(rawGroup);

        if (!chr.equals(IGVConstants.CHR_ALL) && (startLocations.length > 0)) {
            genomeSummaryData.addLocations(chr, startLocations);
        }


        writer.closeGroup(chrGroup);

        return binnedDataList;
    }

    protected int getZoomMax() {
        return zoomMax;
    }

    /**
     *
     * @param chr
     * @param maxLength
     * @return
     */
    private List<BinnedData> computeAllBins(String chr, long maxLength) {
        double binCountCutoff = 3;
        List<BinnedData> binInfoList = new ArrayList();
        int adjustedZoomMax = (chr.equals(CHR_ALL) ? 1 : getZoomMax());

        int[] startLocations = getStartLocationsForChromosome(chr);
        int[] endLocations = getEndLocationsForChromosome(chr);

        for (int z = 0; z < adjustedZoomMax; z++) {
            int nTiles = (int) Math.pow(2, z);
            int nBins = nTiles * this.tileSize;
            double binSize = ((double) maxLength) / nBins;

            if (binSize < 0) {
                System.out.println("Negative bin size");
            }


            List<Bin> bins = allocateBins(chr, nBins, binSize, startLocations, endLocations);
            BinnedData binInfo = computeBinnedData(z, maxLength, nTiles, bins, binSize);

            binInfoList.add(binInfo);

            if ((binInfo.getMeanCount() < binCountCutoff) && (z > zoomMin)) {
                break;
            }
        }

        return binInfoList;
    }

    protected abstract List<Bin> allocateBins(String chr, int nBins, double binSize,
            int[] startLocations, int[] endLocations);

    private void recordRawIndex(int groupId, int chrLength, int[] locations) {
        double chunkSize = 10000;
        int nChunks = (int) (chrLength / chunkSize) + 2;
        int[] indices = new int[nChunks];
        int i = 0;
        int n = 0;

        while ((n < nChunks) && (i < locations.length)) {
            int boundary = (int) (n * chunkSize);

            try {
                while ((locations[i] < boundary) && (i < locations.length - 1)) {
                    i++;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }

            indices[n] = i;
            n++;
        }

        // If we haven't filled the index array it means we have run out of
        // startLocations.  In other words there is no data (startLocations) in the
        // remaining chunks.  Record there indeces = the max location index.
        while (n < nChunks) {
            indices[n] = locations.length - 1;
            n++;
        }

        writer.writeAttribute(groupId, "index.span", chunkSize);
        writer.createAndWriteVectorDataset(groupId, "index", indices);
    }

    private void recordStats(DataStatistics[] stats, int group) {
        int nPts = stats.length;
        float[] min = new float[nPts];
        float[] mean = new float[nPts];
        float[] max = new float[nPts];
        float[] median = new float[nPts];
        float[] percentile10 = new float[nPts];
        float[] percentile90 = new float[nPts];
        float[] percentile98 = new float[nPts];
        float[] stddev = new float[nPts];
        for (int i = 0; i < nPts; i++) {
            DataStatistics stat = stats[i];
            min[i] = (stat == null) ? Float.NaN : (float) stat.getMin();
            mean[i] = (stat == null) ? Float.NaN : (float) stats[i].getMean();
            max[i] = (stat == null) ? Float.NaN : (float) stats[i].getMax();
            median[i] = (stat == null) ? Float.NaN : (float) stats[i].getMedian();
            percentile10[i] = (stat == null) ? Float.NaN : (float) stats[i].getPercentile10();
            percentile90[i] = (stat == null) ? Float.NaN : (float) stats[i].getPercentile90();
            percentile98[i] = (stat == null) ? Float.NaN : (float) stats[i].getPercentile90();
            min[i] = (stat == null) ? Float.NaN : (float) stats[i].getPercentile98();
            stddev[i] = (stat == null) ? Float.NaN : (float) stats[i].getStdDev();
        }

        writer.createAndWriteVectorDataset(group, "min", min);
        writer.createAndWriteVectorDataset(group, "mean", mean);
        writer.createAndWriteVectorDataset(group, "max", max);
        writer.createAndWriteVectorDataset(group, "median", median);
        writer.createAndWriteVectorDataset(group, "percentile10", percentile10);
        writer.createAndWriteVectorDataset(group, "percentile90", percentile90);
        writer.createAndWriteVectorDataset(group, "percentile98", percentile98);
        writer.createAndWriteVectorDataset(group, "stddev", stddev);

    }

    // TODO consider making dataGroup an instance variable.  There is one
    // per file, it doesn't change
    /**
     * Method description
     *
     *
     * @param chr
     * @param dataGroup
     *
     * @return
     */
    public int openChromosomeGroup(String chr, int dataGroup) {
        if (chromosomeGroupCache.contains(chr)) {
            return writer.openGroup(dataGroup, chr);
        } else {
            int chrGroup = writer.createGroup(dataGroup, chr);

            chromosomeGroupCache.add(chr);

            return chrGroup;
        }
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param zoomName
     * @param chrGroup
     *
     * @return
     */
    public int openZoomGroup(String chr, String zoomName, int chrGroup) {
        String key = chr + zoomName;

        if (zoomGroupCache.contains(key)) {
            return writer.openGroup(chrGroup, zoomName);
        } else {
            int zoomGroup = writer.createGroup(chrGroup, zoomName);

            zoomGroupCache.add(key);

            return zoomGroup;
        }
    }

    /**
     * Open a 2-D dataArrayDataset, create if neccessary.
     *
     * @param chr
     * @param zoomName
     * @param dsName
     * @param nCols
     * @param zoomGroup
     * @return
     */
    public int openDataset(String chr, String zoomName, String dsName, int nCols, int zoomGroup) {

        String key = chr + zoomName + dsName;


        if (datasetCache.contains(key)) {
            int datasetId = writer.openDataset(zoomGroup, dsName);

            return datasetId;
        } else {
            int nRows = dataset.getDataHeadings().length;
            int datasetId = writer.createDataset(zoomGroup, dsName, HDF5Constants.H5T_NATIVE_FLOAT,
                    new long[]{nRows,
                        nCols});

            datasetCache.add(key);

            return datasetId;
        }
    }

    /**
     *
     * @param chr
     * @param zoomName
     * @param dsName
     * @param zoomGroup
     * @return
     */
    public int openVectorDataset(String chr, String zoomName, String dsName, int zoomGroup) {
        String key = chr + zoomName + dsName;

        if (datasetCache.contains(key)) {
            int datasetId = writer.openDataset(zoomGroup, dsName);

            return datasetId;
        } else {
            int nRows = dataset.getDataHeadings().length;
            int datasetId = writer.createDataset(zoomGroup, dsName, HDF5Constants.H5T_NATIVE_FLOAT,
                    new long[]{nRows});

            datasetCache.add(key);

            return datasetId;
        }
    }

    private List<String> getAllChromosomes() {
        List<String> allChromosomes = new ArrayList(Arrays.asList(dataset.getChromosomes()));
        allChromosomes.add(CHR_ALL);
        return allChromosomes;

    }

    private void processData(int dataGroup, Map<String, List<BinnedData>> binInfoMap,
            double estTimeFraction)
            throws InterruptedException {

        List<String> allChromosomes = getAllChromosomes();

        int nSteps = allChromosomes.size() * dataset.getDataHeadings().length;
        double procProgIncrement = (estTimeFraction * 0.8 * 100) / nSteps;
        double rawDataProgIncrement = (estTimeFraction * 0.2 * 100) / nSteps;

        // Loop through chromosomes
        for (String chr : allChromosomes) {

            checkForInterrupt();

            if (chr.equals(CHR_ALL) || (genome.getChromosome(chr) != null)) {

                int chrGroup = openChromosomeGroup(chr, dataGroup);

                // Loop through samples
                int rowNumber = 0;
                boolean hasNulls = false;
                int nCols = 0;
                float[][] allData = new float[dataset.getDataHeadings().length][];
                DataStatistics[] stats = new DataStatistics[dataset.getDataHeadings().length];
                for (String heading : dataset.getDataHeadings()) {

                    float[] data = this.getDataForChromosome(heading, chr);
                    allData[rowNumber] = data;

                    if ((data == null) || (data.length == 0)) {
                        allData[rowNumber] = null;
                        stats[rowNumber] = null;
                        log.info("No data for array: " + heading + " chr: " + chr);
                    } else {
                        checkForInterrupt();
                        nCols = data.length;
                        processDataForChromosome(rowNumber, heading, chrGroup, binInfoMap, chr);

                        if (statusMonitor != null) {
                            statusMonitor.incrementStatus(procProgIncrement);
                        }

                        if (statusMonitor != null) {
                            statusMonitor.incrementStatus(rawDataProgIncrement);
                        }

                        genomeSummaryData.addData(heading, chr, data);

                        stats[rowNumber] = ProcessingUtils.computeStats(data);

                    }

                    rowNumber++;
                }

                // If there are any null rows replace them with NaN
                if (hasNulls && (nCols > 0)) {
                    float[] nanArray = new float[nCols];
                    Arrays.fill(nanArray, Float.NaN);
                    for (int i = 0; i < allData.length; i++) {
                        if (allData[i] == null) {
                            allData[i] = nanArray;
                        }
                    }
                }

                int rawGroup = openZoomGroup(chr, "raw", chrGroup);

                writer.createAndWriteDataset(rawGroup, "value", allData);
                recordStats(stats, rawGroup);

                writer.closeGroup(rawGroup);

                writer.closeGroup(chrGroup);
            }
        }
    }

    float[] getDataForChromosome(String sample, String chr) {
        if (chr.equals(CHR_ALL)) {

            // TODO
            return genomeSummaryData.getData(sample);
        } else {
            return dataset.getData(sample, chr);
        }
    }

    int[] getStartLocationsForChromosome(String chr) {
        if (chr.equals(CHR_ALL)) {
            return genomeSummaryData.getLocations();
        } else {
            return dataset.getStartLocations(chr);
        }
    }

    int[] getEndLocationsForChromosome(String chr) {
        if (chr.equals(CHR_ALL)) {
            return null;
        } else {
            return dataset.getEndLocations(chr);
        }
    }

    /**
     *
     * @param rowNumber
     * @param sample
     * @param chrGroup
     * @param binInfoMap
     * @param chr
     */
    private void processDataForChromosome(int rowNumber, String sample, int chrGroup,
            Map<String, List<BinnedData>> binInfoMap, String chr)
            throws InterruptedException {

        // Loop through zoom levels
        for (BinnedData binInfo : binInfoMap.get(chr)) {

            checkForInterrupt();

            List<? extends Bin> bins = binInfo.getBins();

            if (bins.size() > 0) {

                // Arrays for the statistics, 1 element per bin.
                float[] median = new float[bins.size()];
                float[] percent10 = new float[bins.size()];
                float[] percent90 = new float[bins.size()];
                float[] min = new float[bins.size()];
                float[] max = new float[bins.size()];
                float[] mean = new float[bins.size()];
                float[] stdDev = new float[bins.size()];

                float[] data = getDataForChromosome(sample, chr);

                for (int b = 0; b < bins.size(); b++) {
                    Bin bin = bins.get(b);
                    float[] binData = getDataForBin(data, bin);

                    if (binData == null) {
                        median[b] = percent10[b] = percent10[b] = max[b] = mean[b] = stdDev[b] =
                                Float.NaN;
                    } else {

                        //
                        DataStatistics stats = ProcessingUtils.computeStats(binData);

                        median[b] = (float) stats.getMedian();
                        percent10[b] = (float) stats.getPercentile10();
                        percent90[b] = (float) stats.getPercentile90();
                        max[b] = (float) stats.getMax();
                        min[b] = (float) stats.getMin();
                        mean[b] = (float) stats.getMean();
                        stdDev[b] = (float) stats.getStdDev();
                    }
                }

                String zoomName = "z" + binInfo.getZoomLevel();
                int zoomGroup = openZoomGroup(chr, zoomName, chrGroup);

                if (windowFunctions.contains(WindowFunction.median)) {
                    recordStats("median", rowNumber, median, zoomGroup, chr, zoomName);
                }

                if (windowFunctions.contains(WindowFunction.percentile10)) {
                    recordStats("percentile10", rowNumber, percent10, zoomGroup, chr, zoomName);
                }
                if (windowFunctions.contains(WindowFunction.percentile90)) {
                    recordStats("percentile90", rowNumber, percent90, zoomGroup, chr, zoomName);
                }
                if (windowFunctions.contains(WindowFunction.min)) {
                    recordStats("min", rowNumber, min, zoomGroup, chr, zoomName);
                }
                if (windowFunctions.contains(WindowFunction.max)) {
                    recordStats("max", rowNumber, max, zoomGroup, chr, zoomName);
                }
                if (windowFunctions.contains(WindowFunction.mean)) {
                    recordStats("mean", rowNumber, mean, zoomGroup, chr, zoomName);
                }
                if (windowFunctions.contains(WindowFunction.stddev)) {
                    recordStats("stdDev", rowNumber, stdDev, zoomGroup, chr, zoomName);
                }
                writer.closeGroup(zoomGroup);
            }
        }
    }

    protected void recordStats(String type, int rowNumber, float[] data, int zoomGroup, String chr,
            String zoomName) {
        int dataArrayDataset = openDataset(chr, zoomName, type, data.length, zoomGroup);

        writer.writeDataRow(dataArrayDataset, rowNumber, data, data.length);
        writer.closeDataset(dataArrayDataset);

        float median = (float) procUtils.computeMedian(data);
        int medianDataset = openVectorDataset(chr, zoomName, "median." + type, zoomGroup);

        writer.writeDataValue(medianDataset, rowNumber, median);
        writer.closeDataset(medianDataset);
    }

    protected abstract float[] getDataForBin(float[] data, Bin bin);

    /**
     *
     * @param zoomLevel
     * @param maxLength
     * @param nTiles
     * @param bins
     * @param binSize
     * @return
     */
    private BinnedData computeBinnedData(int zoomLevel, double chrLength, int nTiles,
            List<Bin> bins, double binSize) {

        // Find tile breaks.  Could possibly do this n loop above.
        int[] tileBoundaries = new int[nTiles];
        int binNumber = 0;
        double tileLength = chrLength / nTiles;

        for (int tileNumber = 0; (tileNumber < nTiles - 1) && (binNumber < bins.size());
                tileNumber++) {

            // Find end binIndex for this tile.  Using a linear search, might
            // need to use a faster scheme.
            while (bins.get(binNumber).getStart() < (tileNumber + 1) * tileLength) {
                binNumber++;

                if (binNumber == bins.size()) {
                    break;
                }
            }

            tileBoundaries[tileNumber] = binNumber;
        }

        // Boundary for last tile number is end
        tileBoundaries[nTiles - 1] = bins.size() - 1;

        BinnedData binInfo = new BinnedData(zoomLevel, binSize, bins, tileBoundaries);

        // Compute the mean, data, and 90th percentile of occupied beans.
        float mean = 0.0F;
        float max = 0.0F;
        DoubleArrayList percentiles = new DoubleArrayList(3);

        percentiles.add(0.1);
        percentiles.add(0.5);
        percentiles.add(0.96);

        DoubleQuantileFinder qf = QuantileFinderFactory.newDoubleQuantileFinder(false,
                Long.MAX_VALUE, 0.0010, 1.0E-4, percentiles.size(), null);

        for (Bin bin : bins) {
            int count = bin.getFeatureCount();

            mean += count;
            max = Math.max(max, count);
            qf.add(count);
        }

        binInfo.setMeanCount(mean / bins.size());
        binInfo.setMaxCount(max);

        DoubleArrayList quantiles = qf.quantileElements(percentiles);

        binInfo.setPercentile10(quantiles.get(0));
        binInfo.setMedianCount(quantiles.get(1));
        binInfo.setPercentile90(quantiles.get(2));

        return binInfo;
    }

    /**
     * Method description
     *
     *
     * @param zoomMax
     */
    public void setZoomMax(int zoomMax) {
        this.zoomMax = zoomMax;
    }

    private void writeRootAttributes(int root) {

        writer.writeAttribute(root, "name", dataset.getName());
        writer.writeAttribute(root, "has.data", 1);
        writer.writeAttribute(root, "normalized", dataset.isLogNormalized() ? 1 : 0);
        writer.writeAttribute(root, "log.values", dataset.isLogNormalized() ? 1 : 0);
        writer.writeAttribute(root, "version", version);
        writer.writeAttribute(root, "type", dataset.getType());

        if (dataset.getWindowSpan() > 0) {
            writer.writeAttribute(root, "window.span", dataset.getWindowSpan());
        }

        // Track properties.  These properties will apply to all tracks in this dataset
        TrackProperties properties = dataset.getTrackProperties();

        if (properties.getAltColor() != null) {
            writer.writeAttribute(root, "track.altColor",
                    ColorUtilities.convertColorToRGBString(properties.getAltColor()));
        }
        if (properties.getMidColor() != null) {
            writer.writeAttribute(root, "track.altColor",
                    ColorUtilities.convertColorToRGBString(properties.getMidColor()));
        }
        if (properties.getColor() != null) {
            writer.writeAttribute(root, "track.color",
                    ColorUtilities.convertColorToRGBString(properties.getColor()));

        }
        if (properties.getDescription() != null) {
            writer.writeAttribute(root, "track.description", properties.getDescription());
        }
        if (properties.getGenome() != null) {
        }
        if (properties.getHeight() > 0) {
            writer.writeAttribute(root, "track.height", properties.getHeight());

        }
        writer.writeAttribute(root, "track.autoscale", String.valueOf(properties.isAutoScale()));

        // autoscale
        if (!Float.isNaN(properties.getMinValue())) {
            writer.writeAttribute(root, "track.minValue", (double) properties.getMinValue());
        }
        if (!Float.isNaN(properties.getMidValue())) {
            writer.writeAttribute(root, "track.midValue", (double) properties.getMidValue());
        }
        if (!Float.isNaN(properties.getMaxValue())) {
            writer.writeAttribute(root, "track.maxValue", (double) properties.getMaxValue());
        }
        writer.writeAttribute(root, "track.drawMidValue", String.valueOf(properties.isDrawMidValue()));

        if (properties.getName() != null) {
            writer.writeAttribute(root, "track.name", properties.getName());
        }
        if (properties.getOffset() > 0) {
        }
        if (properties.getRendererClass() != null) {
            RendererType rType = RendererFactory.getRenderType(properties.getRendererClass());

            if (rType != null) {
                writer.writeAttribute(root, "track.renderer", rType.toString());
            }
        }

        if (properties.getUrl() != null) {
        }
        if (properties.getWindowingFunction() != null) {
            writer.writeAttribute(root, "track.windowFunction",
                    properties.getWindowingFunction().toString());

        }
    }
}
