/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this locator except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.preprocess;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.feature.*;
import org.broad.igv.util.ResourceLocator;

/*
 * browser hide all
browser pack knownGene ensGene refGene genscan multiz17way
track name="nucCount_CM12_Jul2808" description="nucCount_CM12_Jul2808" type="wiggle_0" color=50,50,150 yLineMark=0.0 yLineOnOff=on visibility=2
fixedStep chrom=0 strt=1  step=1
 */



import java.io.*;

import java.util.*;
import org.broad.igv.util.AsciiLineReader;

/**
 * Class description
 *
 *
 * @version    Enter version here..., 08/10/31
 * @author     Enter your name here...
 */
public class WiggleParser {

    /**
     * @return the dataConsumer
     */
    public DataConsumer getDataConsumer() {
        return dataConsumer;
    }

    /**
     * @param dataConsumer the dataConsumer to set
     */
    public void setDataConsumer(DataConsumer dataConsumer) {
        this.dataConsumer = dataConsumer;
    }

    private enum Type {

        FIXED, VARIABLE, BED
    }

     private DataConsumer dataConsumer  ;
    /**
     * The type of wiggle locator (see UCSC documentation).
     */
    private  Type type  = Type.BED;

    // State variables.  This is a serial type parser,  these variables are used to hold temporary
    // state.
    private String chr;
    String lastChr = "";
    int lastPosition = 0;
    private int start;
    private int step;
    private int windowSpan;    // <- ignored for now
    private double minValue;
    private double maxValue;
    ResourceLocator resourceLocator;
    Set<String> unsortedChromosomes;

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param genomeId
     */
    public WiggleParser(String file) {
        this.resourceLocator = new ResourceLocator(file);

    }

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param genomeId
     */
    public WiggleParser(ResourceLocator locator) {
        this.resourceLocator = locator;

    }

    /**
     * Utility method.  Returns true if this looks like a wiggle locator.  The criteria is to scan
     * the first 100 lines looking for a valid "track" line.  According to UCSC documentation
     * track lines must contain a type attribute,  which must be equal to "wiggle_0".
     *
     *
     * @param file
     * @return
     */
    public static boolean isWiggle(ResourceLocator file) {
        AsciiLineReader reader = null;
        try
        {
            reader = ParsingUtils.openAsciiReader(file);
            String nextLine = null;
            int lineNo = 0;
            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0))
            {
                if (nextLine.startsWith("track") && nextLine.contains("wilggle_0"))
                {
                    return true;
                }
                if (lineNo++ > 100)
                {
                    break;
                }
            }
        } catch (IOException e)
        {
            e.printStackTrace();
            return false;
        } finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }
        return false;
    }

    /**
     * Method description
     *
     *
     *
     *
     *
     *
     * @return
     */
    public void parse() {


        String[] tokens = new String[10];

        lastPosition = 0;
        unsortedChromosomes = new HashSet();

        AsciiLineReader reader = null;
        try
        {

            reader = ParsingUtils.openAsciiReader(resourceLocator);
            String nextLine = null;
            int position = -1;
            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0))
            {

                // Skip comment lines
                if (nextLine.startsWith("#") || nextLine.startsWith("data") || nextLine.startsWith(
                    "browser"))
                {
                    continue;
                }


                if (nextLine.startsWith("track"))
                {
                    type = Type.BED;
                //DatasetParserUtils.parseTrackLine(nextLine, dataset.getTrackProperties());

                } else if (nextLine.startsWith("fixedStep"))
                {
                    type = Type.FIXED;
                    parseStepLine(nextLine);
                    position = start;
                    if (start <= lastPosition)
                    {
                        unsortedChromosomes.add(chr);
                    }



                } else if (nextLine.startsWith("variableStep"))
                {
                    type = Type.VARIABLE;
                    parseStepLine(nextLine);
                    if (start <= lastPosition)
                    {
                        unsortedChromosomes.add(chr);
                    }



                } else
                {

                    // Must be data
                    int nTokens = ParsingUtils.split(nextLine, tokens, '\t');
                    if (nTokens == 0)
                    {
                        continue;
                    }
                    try
                    {
                        if (type.equals(Type.BED))
                        {
                            if (nTokens > 3)
                            {
                                chr = tokens[0].trim();
                                if (!chr.equals(lastChr))
                                {
                                    newChromosome();

                                }
                                lastChr = chr;

                                int startPosition = Integer.parseInt(tokens[1].trim());
                                if (startPosition <= lastPosition)
                                {
                                    unsortedChromosomes.add(chr);
                                }
                                lastPosition = startPosition;

                                int endPosition = Integer.parseInt(tokens[2].trim());
                                float data = Float.parseFloat(tokens[3].trim());
                                getDataConsumer().addData(chr, startPosition, endPosition, data);
                            }
                        } else if (type.equals(Type.VARIABLE))
                        {
                            if (nTokens > 1)
                            {

                                // Per UCSC specification variable and fixed step coordinates are "1" based.
                                // We need to subtract 1 to convert to the internal "zero" based coordinates.
                                int startPosition = Integer.parseInt(tokens[0]) - 1;
                                if (startPosition <= lastPosition)
                                {
                                    unsortedChromosomes.add(chr);
                                }
                                lastPosition = startPosition;

                                int endPosition = startPosition + windowSpan;
                                float data = Float.parseFloat(tokens[1].trim());
                                getDataConsumer().addData(chr, startPosition, endPosition, data);
                            }
                        } else
                        {    // Fixed step -- sorting is checked when step line is parsed
                            if (position >= 0)
                            {
                                int endPosition = position + windowSpan;
                                float data = Float.parseFloat(tokens[0].trim());
                                getDataConsumer().addData(chr, position, endPosition, data);

                            }
                            position += step;
                            lastPosition = position;

                        }

                    } catch (NumberFormatException e)
                    {
                        System.out.println("Cannot parse: " + nextLine);
                    }


                }

            }

            parsingComplete();

        } catch (IOException e)
        {
            e.printStackTrace();
        } finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }
    }

    // fixedStep chrom=chrM strt=1 step=1
    private void parseStepLine(String header) {
        String[] tokens = header.split(" ");
        for (String token : tokens)
        {
            String[] keyValue = token.split("=");
            if (keyValue.length >= 2)
            {
                if (keyValue[0].equalsIgnoreCase("chrom"))
                {
                    chr = keyValue[1];
                    if (!chr.equals(lastChr))
                    {
                        newChromosome();
                    }
                    lastChr = chr;

                } else if (keyValue[0].equalsIgnoreCase("start"))
                {
                    // Per UCSC specification variable and fixed step coordinates are "1" based.
                    // We need to subtract 1 to convert to the internal "zero" based coordinates.

                    start = Integer.parseInt(keyValue[1]) - 1;
                    if (start <= lastPosition)
                    {
                        unsortedChromosomes.add(chr);
                    }

                } else if (keyValue[0].equalsIgnoreCase("step"))
                {
                    step = Integer.parseInt(keyValue[1]);
                } else if (keyValue[0].equalsIgnoreCase("span"))
                {
                    windowSpan = Integer.parseInt(keyValue[1]);
                }

            }
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getMinValue() {
        return minValue;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getMaxValue() {
        return maxValue;
    }

    private void newChromosome() {
        getDataConsumer().newChromosome(chr);
        lastPosition = 0;
    }

    private void parsingComplete() {

    }
}
