/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.maf;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.PreferenceManager;
import org.broad.igv.maf.MAFTile.MASequence;
import org.broad.igv.renderer.GraphicUtils;
import org.broad.igv.renderer.Renderer;
import org.broad.igv.track.AbstractTrack;
import org.broad.igv.track.RegionScoreType;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.FontManager;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.util.ResourceLocator;

/**
 *
 * @author jrobinso
 */
public class MAFTrack extends AbstractTrack {

    public static final int margin = 5;
    private static Logger log = Logger.getLogger(MAFTrack.class);
    private static int EXPANDED_HEIGHT = 14;
    private static int GAPS_HEIGHT = 25;
    //List<Rectangle> featureRects = new ArrayList();
    MAFManager mgr;
    MAFRenderer renderer = new MAFRenderer();
    // A hack until full MAF track is implemented.
    Rectangle visibleNameRect;

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param name
     * @param alignments
     */
    public MAFTrack(ResourceLocator locator, String name) {
        super(locator, name);
        this.mgr = new MAFManager(locator);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public int getHeight() {
        return GAPS_HEIGHT + (mgr.getSelectedSpecies().size() + 1) * EXPANDED_HEIGHT;
    }

    @Override
    public void renderName(Graphics2D g2D, Rectangle trackRectangle, Rectangle visibleRect) {

        this.visibleNameRect = visibleRect;
        if (isSelected()) {
            g2D.setBackground(Color.LIGHT_GRAY);
        } else {
            g2D.setBackground(Color.WHITE);
        }

        Rectangle rect = new Rectangle(trackRectangle);
        g2D.clearRect(rect.x, rect.y, rect.width, rect.height);


        Font font = FontManager.getScalableFont(10);
        g2D.setFont(font);
        FontManager.applyScalableTextHints(g2D);

        int y = trackRectangle.y;

        rect.height = GAPS_HEIGHT;
        rect.y = y;
        //text, rect.x + rect.width - margin, yPos, g2D

        GraphicUtils.drawVerticallyCenteredText("Gaps", margin, rect, g2D, true);
        rect.y += rect.height;

        rect.height = EXPANDED_HEIGHT;

        String ref = MAFManager.speciesNames.getProperty(mgr.refId);
        if(ref == null) {
            ref = mgr.refId;
        }
        GraphicUtils.drawVerticallyCenteredText(ref, margin, rect, g2D, true);
        rect.y += rect.height;

        for (String sp : mgr.getSelectedSpecies()) {

            String name = MAFManager.speciesNames.getProperty(sp);
            if (name == null) {
                name = sp;
            }
            GraphicUtils.drawVerticallyCenteredText(name, margin, rect, g2D, true);
            rect.y += rect.height;
        }

    }

    public void render(RenderContext context, Rectangle rect) {

        double locScale = context.getScale();

        if (locScale > 1) {
            Rectangle r = new Rectangle(rect);
            if (visibleNameRect != null) {
                r.y = visibleNameRect.y;
                r.height = visibleNameRect.height;
            }

            Graphics2D g = context.getGraphic2DForColor(Color.black);
            GraphicUtils.drawCenteredText("Zoom in to see alignments.", r, g);
            return;

        }

        double origin = context.getOrigin();
        String chr = context.getChr();
        String genome = IGVModel.getInstance().getViewContext().getGenomeId();

        int start = (int) origin;
        int end = (int) (origin + rect.width * locScale) + 1;

        // TODO -- check genome and chr to load correct MAF file
        //if (!genome.equals("hg18")) {
        //    return;
        //}

// Get tiles
        MAFTile[] tiles = mgr.getTiles(chr, start, end);
        for (MAFTile tile : tiles) {
            // render tile
            if (tile != null) {
                renderTile(context, rect, tile);
            }
        }

    }

    private void renderTile(RenderContext context, Rectangle trackRectangle,
            MAFTile tile) {

        int y = trackRectangle.y;

        MASequence reference = tile.refSeq;
        if (reference == null) {
            return;
        }

        Rectangle rect = new Rectangle(trackRectangle);
        rect.height = GAPS_HEIGHT;
        rect.y = y;
        renderer.renderGaps(tile.getGaps(), context, rect);
        rect.y += rect.height;

        rect.height = EXPANDED_HEIGHT;
        renderer.renderAligment(reference, reference, null, context, rect, this);
        rect.y += rect.height;

        // TODO Render gaps
        for (String sp : mgr.getSelectedSpecies()) {

            MASequence seq = tile.alignedSequences.get(sp);
            if (seq != null) {
                renderer.renderAligment(seq, reference, tile.getGaps(), context, rect, this);
            }

            rect.y += rect.height;

        }

    }

    public void setStatType(WindowFunction type) {
        // Ignored
    }

    public WindowFunction getWindowFunction() {
        return null;
    }

    public void setRendererClass(Class rc) {
        // Ignored
    }

    public Renderer getRenderer() {
        return renderer;
    }

    public boolean isLogNormalized() {
        return false;
    }

    public String getValueStringAt(
            String chr, double position, int y) {
        return "Multiple alignments";
    }

    public float getRegionScore(String chr, int start, int end, int zoom,
            RegionScoreType type) {
        return 0;
    }

    @Override
    public boolean handleClick(MouseEvent e) {
        if (e.isPopupTrigger()) {
            configureTrack();
            return true;
        }
        return false;
    }

    private void configureTrack() {
        MAFConfigurationDialog dialog = new MAFConfigurationDialog(
                IGVMainFrame.getInstance(), true, mgr);
        dialog.setLocationRelativeTo(IGVMainFrame.getInstance());
        dialog.addWindowListener(new java.awt.event.WindowAdapter() {

            public void windowClosing(java.awt.event.WindowEvent e) {
                System.exit(0);
            }
        });
        dialog.setVisible(true);

        if (dialog.cancelled) {
        } else {
            List<String> selectedSpecies = dialog.getSelectedSpecies();
            mgr.setSelectedSpecies(selectedSpecies);
            PreferenceManager.getInstance().setMafSpecies(selectedSpecies);
            IGVMainFrame.getInstance().repaint();
        }

    }
}

