/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ibf;

import com.mindprod.ledatastream.LEDataOutputStream;
import org.broad.igv.h5.*;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.log4j.Logger;
import org.broad.igv.track.TrackType;

/**
 * Assumptions
 *
 * Little endian is used throughout
 * Strings are null terminated ascii (single byte
 *
 * @author jrobinso
 */
public class IBFWriter {

    static private Logger log = Logger.getLogger(IBFWriter.class);
    static private int version = 1;
    BufferedFileOutputStream fos = null;
    File file;
    Map<String, IBFGroup> groupCache = new LinkedHashMap();
    Map<String, IBFDataArray> datasetCache = new LinkedHashMap();
    Map<String, Long> index = new LinkedHashMap();

    public IBFWriter(File file, TrackType trackType, String trackLine, String[] trackNames) {
        this.file = file;
        String filename = file.getAbsolutePath();

        String fullpath = new File(filename).getAbsolutePath();
        if (!this.file.getAbsolutePath().equals(fullpath)) {
            throw new IllegalArgumentException("Filename mismatch.  Expected " + this.file.getAbsolutePath() +
                    " argument: " + filename);
        }

        try {
            fos = new BufferedFileOutputStream(new FileOutputStream(file));
            writeHeader(trackType, trackLine, trackNames);


            IBFGroup rootGroup = new IBFGroup("/");
            groupCache.put(rootGroup.name, rootGroup);


        } catch (IOException ex) {
            log.error("Error creating file: " + filename, ex);
            throw new RuntimeException("Error creating file: " + filename, ex);
        }

    }

    private void writeHeader(TrackType trackType, String trackLine, String[] trackNames) throws IOException {

        // Magic number -- 4 bytes
        byte [] magicNumber = new byte [] {'I', 'B', 'F', '1'};
        write(magicNumber);

        writeInt(version);
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        LEDataOutputStream dos = new LEDataOutputStream(buffer);
        writeString(dos, trackType.toString());
        writeString(dos, trackLine == null ? " " : trackLine);
        dos.writeInt(trackNames.length);
        for (String nm : trackNames) {
            writeString(dos, nm);
        }
        byte[] bytes = buffer.toByteArray();
        writeInt(bytes.length);
        write(bytes);
    }

    /**
     * Write out the group and dataset index and close the underlying file.
     * 
     * @param fileId
     */
    public void closeFile() {

        try {
            writeDatasets();
            writeGroups();
            long indexPosition = writeIndex();
            writeLong(indexPosition);
            fos.close();
        } catch (IOException ex) {
            log.error("Error closing file");
        }

    }

    public void createDataset(String name, IBFDataArray.Format format, IBFDataArray.DataType dataType,
            float tileWidth, float binWidth, int nTiles) {
        datasetCache.put(name, new IBFDataArray(name, format, dataType, tileWidth, binWidth, nTiles));
    }

    public void createGroup(String name) {
        groupCache.put(name, new IBFGroup(name));
    }

    public void writeAttribute(String name, String key, String value) {
        IBFGroup group = groupCache.get(name);
        if (group == null) {
            throw new java.lang.NoSuchFieldError("Group: " + name + " doese not exist.  " +
                    "Call createGroup first");
        }
        group.attributes.put(key, value);
    }

    // Note this will only work for "fixed step" format.  Others need location arrays
    public void writeTile(String dsId, int tileNumber, int start, Object data) {
        try {
            IBFDataArray dataset = datasetCache.get(dsId);
            if (dataset == null) {
                throw new java.lang.NoSuchFieldError("Dataset: " + dsId + " doese not exist.  " +
                        "Call createDataset first");
            }

            dataset.tilePositions[tileNumber] = fos.position();

            // Write out the data

            byte[] bytes = null;
            switch (dataset.dataType) {
                case INT:
                    bytes = convertToBytes((int[][]) data);
                    break;
                case SHORT:
                    bytes = convertToBytes((short[][]) data);
                    break;
                case FLOAT:
                    bytes = convertToBytes((float[][]) data);
                    break;
                default:
                    throw new java.lang.IllegalArgumentException("Dataset type not supported: " +
                            dataset.dataType.toString());
            }

            // Record # of bytes.  Add 4 bytes for the start location
            writeInt(bytes.length + 4);
            writeInt(start);
            write(bytes);

        } catch (IOException ex) {
            ex.printStackTrace();
        }

    }

    private void writeDatasets() throws IOException {
        for (IBFDataArray dataset : datasetCache.values()) {
            index.put(dataset.name, fos.position());
            writeDataset(dataset);
        }
    }

    private void writeDataset(IBFDataArray dataset) throws IOException {
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        LEDataOutputStream dos = new LEDataOutputStream(buffer);
        writeString(dos, dataset.dataType.toString());
        writeString(dos, dataset.format.toString());
        dos.writeFloat(dataset.tileWidth);
        dos.writeFloat(dataset.binWidth);
        dos.writeInt(dataset.tilePositions.length);
        for (int i = 0; i < dataset.tilePositions.length; i++) {
            dos.writeLong(dataset.tilePositions[i]);
        }
        byte[] bytes = buffer.toByteArray();
        writeInt(bytes.length);
        write(bytes);
    }

    private void writeGroups() throws IOException {
        for (IBFGroup dataset : groupCache.values()) {
            index.put(dataset.name, fos.position());
            writeGroup(dataset);
        }
    }

    private void writeGroup(IBFGroup group) throws IOException {
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        LEDataOutputStream dos = new LEDataOutputStream(buffer);
        dos.writeInt(group.attributes.size());
        for (Map.Entry<String, String> entry : group.attributes.entrySet()) {
            writeString(dos, entry.getKey());
            writeString(dos, entry.getValue());
        }
        byte[] bytes = buffer.toByteArray();
        writeInt(bytes.length);
        write(bytes);
    }

    private long writeIndex() throws IOException {

        long position = fos.position();

        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        LEDataOutputStream dos = new LEDataOutputStream(buffer);
        // Now write out dataset index
        dos.writeInt(index.size());
        for (Map.Entry<String, Long> entry : index.entrySet()) {
            writeString(dos, entry.getKey());
            dos.writeLong(entry.getValue());
        }
        byte[] bytes = buffer.toByteArray();

        writeInt(bytes.length);
        write(bytes);

        return position;
    }

    private void write(byte[] bytes) throws IOException {
        fos.write(bytes);
    }

    // Convert an array of floats to bytes.  The byte array includes the size
    private byte[] convertToBytes(float[][] data) {
        byte[] bytes = new byte[data.length * data[0].length * 4];
        int byteIdx = 0;
        for (int i = 0; i < data.length; i++) {
            for (int j = 0; j < data[i].length; j++) {
                int v = Float.floatToIntBits(data[i][j]);
                bytes[byteIdx++] = (byte) ((v >>> 0) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 8) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 16) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 24) & 0xFF);
            }
        }
        return bytes;
    }

    private byte[] convertToBytes(int[][] data) {
        byte[] bytes = new byte[data.length * data[0].length * 4];
        int byteIdx = 0;
        for (int i = 0; i < data.length; i++) {
            for (int j = 0; i < data[i].length; j++) {
                int v = data[i][j];
                bytes[byteIdx++] = (byte) ((v >>> 0) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 8) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 16) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 24) & 0xFF);
            }
        }

        return bytes;
    }

    private byte[] convertToBytes(short[][] data) {
        byte[] bytes = new byte[data.length * data[0].length * 2];
        int byteIdx = 0;
        for (int i = 0; i < data.length; i++) {
            for (int j = 0; i < data[i].length; j++) {
                short v = data[i][j];
                bytes[byteIdx++] = (byte) ((v >>> 0) & 0xFF);
                bytes[byteIdx++] = (byte) ((v >>> 8) & 0xFF);
            }
        }
        return bytes;
    }

    private void writeInt(int v) throws IOException {
        fos.write((v >>> 0) & 0xFF);
        fos.write((v >>> 8) & 0xFF);
        fos.write((v >>> 16) & 0xFF);
        fos.write((v >>> 24) & 0xFF);
    }
    /**
     * Writes a <code>long</code> to the underlying output stream as eight
     * bytes, high byte first. In no exception is thrown, the counter
     * <code>written</code> is incremented by <code>8</code>.
     *
     * @param      v   a <code>long</code> to be written.
     * @exception  IOException  if an I/O error occurs.
     * @see        java.io.FilterOutputStream#out
     */
    private byte writeBuffer[] = new byte[8];

    private final void writeLong(long v) throws IOException {
        writeBuffer[7] = (byte) (v >>> 56);
        writeBuffer[6] = (byte) (v >>> 48);
        writeBuffer[5] = (byte) (v >>> 40);
        writeBuffer[4] = (byte) (v >>> 32);
        writeBuffer[3] = (byte) (v >>> 24);
        writeBuffer[2] = (byte) (v >>> 16);
        writeBuffer[1] = (byte) (v >>> 8);
        writeBuffer[0] = (byte) (v >>> 0);
        fos.write(writeBuffer, 0, 8);
    }

    /**
     * IGV requires all strings to be ascii,  so single byte storaged is enough
     * @param dos
     * @param s
     * @throws java.io.IOException
     */
    private void writeString(LEDataOutputStream dos, String s) throws IOException {
        dos.write(s.getBytes());
        dos.writeByte(0);
    }
}
