/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ibf;

import cern.colt.list.ByteArrayList;
import com.mindprod.ledatastream.LEDataInputStream;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.log4j.Logger;
import org.broad.igv.track.TrackType;

/**
 *
 * @author jrobinso
 */
public class IBFReader {

    static final Logger log = Logger.getLogger(IBFReader.class);
    File file;
    //DataInputStream dis = null;
    FileInputStream fis = null;
    FileChannel channel = null;
    Map<String, Long> index;
    TrackType trackType;
    String trackLine;
    String[] trackNames;

    public IBFReader(File file) {
        this.file = file;

        try {
            int fileId = file.getAbsolutePath().hashCode();

            fis = new FileInputStream(file);
            channel = fis.getChannel();

            readHeader();
            readIndex();

        } catch (IOException ex) {
            log.error("Error creating file: " + file, ex);
            throw new RuntimeException("Error creating file: " + file, ex);
        }

    }

    private void readHeader() throws IOException {
        byte[] magicNumber = new byte[4];
        readFully(magicNumber);

        int version = readInt();
        int nBytes = readInt();
        byte[] bytes = new byte[nBytes];
        readFully(bytes);

        LEDataInputStream bdis = new LEDataInputStream(new ByteArrayInputStream(bytes));
        trackType = TrackType.valueOf(readString(bdis));
        trackLine = readString(bdis).trim();
        int nTracks = bdis.readInt();
        trackNames = new String[nTracks];
        for (int i = 0; i < nTracks; i++) {
            trackNames[i] = readString(bdis);
        }


    }

    private void readIndex() throws IOException {

        // Position channel at the start of the index
        long lastPosition = file.length() - Long.SIZE / 8;
        channel.position(lastPosition);
        long magicNumber = readLong();
        channel.position(magicNumber);

        // Read index chunk
        int nBytes = readInt();
        byte[] bytes = new byte[nBytes];
        readFully(bytes);

        LEDataInputStream bdis = new LEDataInputStream(new ByteArrayInputStream(bytes));

        // Read dataset index
        int nDatasets = bdis.readInt();
        index = new LinkedHashMap(nDatasets);
        for (int i = 0; i < nDatasets; i++) {
            String name = readString(bdis);
            long fPosition = bdis.readLong();
            index.put(name, fPosition);
        }
    }

    public synchronized IBFDataArray readDataset(String name) {

        try {
            long position = index.get(name);
            channel.position(position);

            int nBytes = readInt();
            byte[] buffer = new byte[nBytes];
            readFully(buffer);

            LEDataInputStream dis = new LEDataInputStream(new ByteArrayInputStream(buffer));
            IBFDataArray.DataType dataType = IBFDataArray.DataType.valueOf(readString(dis));
            IBFDataArray.Format format = IBFDataArray.Format.valueOf(readString(dis));
            float tileWidth = dis.readFloat();
            float binWidth = dis.readFloat();
            int nTiles = dis.readInt();
            IBFDataArray ds = new IBFDataArray(name, format, dataType, tileWidth, binWidth, nTiles);

            for (int i = 0; i < nTiles; i++) {
                ds.tilePositions[i] = dis.readLong();
            }

            return ds;
        } catch (IOException ex) {
            log.error("Error reading dataset: " + name, ex);
            throw new RuntimeException("System error occured while reading dataset: " + name);
        }
    }

    public synchronized IBFGroup readGroup(String name) {

        try {
            long position = index.get(name);
            channel.position(position);

            int nBytes = readInt();
            byte[] buffer = new byte[nBytes];
            readFully(buffer);

            LEDataInputStream dis = new LEDataInputStream(new ByteArrayInputStream(buffer));
            int nAttributes = dis.readInt();
            Map<String, String> attrs = new HashMap(nAttributes);
            while (nAttributes-- > 0) {
                String key = readString(dis);
                String value = readString(dis);
                attrs.put(key, value);
            }

            return new IBFGroup(name, attrs);
        } catch (IOException ex) {
            log.error("Error reading group: " + name, ex);
            throw new RuntimeException("System error occured while reading group: " + name);
        }
    }

    // For now only support "fixed tile"
    public synchronized IBFBin readTile(IBFDataArray ds, int tileNumber) {

        try {
            long position = ds.tilePositions[tileNumber];
            channel.position(position);

            int nBytes = readInt();
            byte[] buffer = new byte[nBytes];
            readFully(buffer);

            LEDataInputStream dis = new LEDataInputStream(new ByteArrayInputStream(buffer));
            int start = dis.readInt();
            int nFloats = (buffer.length - 4) / 4;
            int nCols = nFloats / trackNames.length;

            float[] data = new float[nFloats];
            for (int i = 0; i < nFloats; i++) {
                data[i] = dis.readFloat();
            }

            return new IBFFixedTile(nCols, start, ds.binWidth, data);
        } catch (IOException ex) {
            String tileName = ds.name + "[" + tileNumber + "]";
            log.error("Error reading data tile: " + tileName, ex);
            throw new RuntimeException("System error occured while reading group: " + tileName);
        }
    }

    public void closeFile() {
        try {
            fis.close();
        } catch (IOException ex) {
            log.error("Error closing file: " + file, ex);
        }
    }

    /**
     * Bytes for this operation are read from the contained
     * input stream.
     *
     * @param      b     the buffer into which the data is read.
     *
     * @exception  EOFException  if this input stream reaches the end before
     *               reading all the bytes.
     * @exception  IOException   if an I/O error occurs.
    
     */
    private void readFully(byte b[]) throws IOException {
        int off = 0;
        int len = b.length;
        if (len < 0) {
            throw new IndexOutOfBoundsException();
        }
        int n = 0;
        while (n < len) {
            int count = fis.read(b, off + n, len - n);
            if (count < 0) {
                throw new EOFException();
            }
            n += count;
        }
    }

    private int readInt() throws IOException {
        int ch1 = fis.read();
        int ch2 = fis.read();
        int ch3 = fis.read();
        int ch4 = fis.read();
        if ((ch1 | ch2 | ch3 | ch4) < 0) {
            throw new EOFException();
        }
        return ((ch4 << 24) + (ch3 << 16) + (ch2 << 8) + (ch1 << 0));
    }

    private long readLong() throws IOException {
        byte[] readBuffer = new byte[8];
        readFully(readBuffer);
        return (((long) readBuffer[7] << 56) +
                ((long) (readBuffer[6] & 255) << 48) +
                ((long) (readBuffer[6] & 255) << 40) +
                ((long) (readBuffer[4] & 255) << 32) +
                ((long) (readBuffer[3] & 255) << 24) +
                ((long) (readBuffer[2] & 255) << 16) +
                ((long) (readBuffer[1] & 255) << 8) +
                ((long) (readBuffer[0] & 255) << 0));
    }

    private String readString(LEDataInputStream dis) throws IOException {
        ByteArrayList bytes = new ByteArrayList(100);
        byte b = -1;
        while ((b = dis.readByte()) != 0) {
            bytes.add(b);
        }
        bytes.trimToSize();
        return new String(bytes.elements());
    }
}
