/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.h5;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;

/**
 *
 * @author jrobinso
 */
public class ZipReader extends ZipBase implements HDF5Reader {

    int fileId;

    ZipFile zipFile;

    Map<Integer, ZipEntry> zipEntries = new HashMap();

    Map<Integer, Map<String, String>> attributeMaps = new HashMap();

    public ZipReader(String filename) {
        openFile(filename);
    }

    Map<String, String> getAttributes() {
        Map<String, String> attrs = attributeMaps.get(fileId);
        if (attrs == null) {
            attrs = readAttributes();
            attributeMaps.put(fileId, attrs);
        }
        return attrs;
    }

    private Map<String, String> readAttributes() {
        String entryName = "attributes";
        ZipEntry entry = zipFile.getEntry(entryName);
        List<String> lines = readAsStrings(zipFile, entry);

        Map<String, String> map = new HashMap(lines.size());
        for (String kv : lines) {
            String[] tokens = kv.split("\t");
            map.put(tokens[0], tokens[1]);
        }
        return map;

    }

    private int openFile(String path) {
        try {
            fileId = path.hashCode();
            zipFile = new ZipFile(new File(path));
            addEntity(fileId, "/");


            return fileId;
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }

    public void closeFile() {
        try {
            zipFile.close();
            removeEntity(fileId);

        //TODO -- need to remove all entities associated with this file
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }

    public int openDataset(String datasetName) {

        String entryName = datasetName;
        ZipEntry entry = zipFile.getEntry(entryName);
        int id = entry.hashCode();
        addEntity(id, datasetName);
        zipEntries.put(id, entry);
        return id;
    }

    public void closeDataset(int datasetId) {
        zipEntries.remove(datasetId);
    }

    public int openGroup(String groupPath) {
        // Nothing to do here other than create the path name and cache it.
        String fullName = getFullName(fileId, groupPath);
        int id = fullName.hashCode();
        addEntity(id, fullName);

        return id;
    }

    public void closeGroup(int groupId) {
        removeEntity(groupId);
    }

    public String[] getChildNames(String groupPath) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public int readIntegerAttribute(int groupId, String attrName) throws ObjectNotFoundException {
        return Integer.parseInt(readStringAttribute(groupId, attrName));
    }

    public String readStringAttribute(int groupId, String attrName) {
        String groupName = getEntityName(groupId);
        String fullAttrName = groupName + "|" + attrName;
        return this.getAttributes().get(fullAttrName);
    }

    public double readDoubleAttribute(int groupId, String attrName) {
        return Double.parseDouble(readStringAttribute(groupId, attrName));
    }

    public float[] readAllFloats(int datasetId) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        return readAsFloats(zipFile, entry);

    }

    public int[] readAllInts(int datasetId) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        return readAsInts(zipFile, entry);
    }

    public String[] readAllStrings(int datasetId) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        return readAsStrings(zipFile, entry).toArray(new String[]{});
    }

    public float[] readFloats(int datasetId, int fromIndex, int toIndex) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        return readAsFloats(zipFile, entry, fromIndex, toIndex);
    }

    public int[] readInts(int datasetId, int fromIndex, int toIndex) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        return readAsInts(zipFile, entry, fromIndex, toIndex);
    }

    //////////// TODO 
    // This implementation is restricted to  1-d datasets.   
    public float[][] readDataSlice(int datasetId, int fromIndex, int toIndex) {
        String entryName = getEntityName(datasetId);
        ZipEntry entry = zipFile.getEntry(entryName);
        float[] data = readAsFloats(zipFile, entry, fromIndex, toIndex);

        return new float[][]{data};

    }

    private List<String> readAsStrings(ZipFile file, ZipEntry entry) {
        List<String> strings = new ArrayList();
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(file.getInputStream(entry)));
            String nextLine = null;
            while ((nextLine = br.readLine()) != null) {
                strings.add(nextLine.trim());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return strings;

    }

    /**
     * Read the contents of a zip entry as an array of ints.
     * 
     * @param entryName name of the entry
     * @return array of ints
     */
    private int[] readAsInts(ZipFile file, ZipEntry entry) {
        try {

            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(file.getInputStream(entry)));

            int nInts = (int) (entry.getSize() / 4);
            int[] ints = new int[nInts];
            for (int i = 0; i < nInts; i++) {
                ints[i] = is.readInt();
            }

            return ints;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * Read the contents of a zip entry as an array of ints.
     * 
     * @param entryName name of the entry
     * @return array of ints
     */
    private int[] readAsInts(ZipFile file, ZipEntry entry, int startIndex, int endIndex) {
        try {

            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(file.getInputStream(entry)));

            int nPts = endIndex - startIndex;
            assert nPts <= (entry.getSize() / 4);
  
            int startPosition = startIndex * 4;
            int nBytes = nPts * 4;
            byte [] bytes = this.readBytes(is, startPosition, nBytes);
            
            return convertBytesToInts(bytes);
            
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * Read the content of a zip entry as an array of floats.
     * @param entryName name of the entry
     * @return contents as an array of floats
     */
    private float[] readAsFloats(ZipFile file, ZipEntry entry) {
        try {
            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(file.getInputStream(entry)));

            int nInts = (int) (entry.getSize() / 4);
            float[] values = new float[nInts];
            for (int i = 0; i < nInts; i++) {
                values[i] = is.readFloat();
            }

            return values;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Read the content of a zip entry as an array of floats.
     * @param entryName name of the entry
     * @return contents as an array of floats
     */
    private float[] readAsFloats(ZipFile file, ZipEntry entry, int startIndex, int endIndex) {
        try {
            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(file.getInputStream(entry)));

            int nPts = endIndex - startIndex;
            assert nPts <= (entry.getSize() / 4);
            float[] values = new float[nPts];

            int startByte = startIndex * 4;
            is.skipBytes(startByte);
            for (int i = 0; i < nPts; i++) {
                values[i] = is.readFloat();
            }

            return values;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Read the number of bytes specificed from the input stream
     */
    protected byte[] readBytes(InputStream inStream, int startPosition, int nBytes) throws IOException {
        byte[] bytes = new byte[nBytes];
        int bytesRead = 0;
        inStream.skip(startPosition);
        while (bytesRead < nBytes) {
            bytesRead += inStream.read(bytes, bytesRead, inStream.available());
        }
        return bytes;
    }

    public static int[] convertBytesToInts(byte[] bytes) {
        try {
            int nInts = bytes.length / (Integer.SIZE / 8);
            int[] ints = new int[nInts];
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(bytes));
            for (int i = 0; i < nInts; i++) {
                if (dis.available() >= (Integer.SIZE / 8)) {
                    ints[i] = dis.readInt();
                }
            }
            dis.close();
            return ints;
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }

    public static float[] convertBytesToFloats(byte[] bytes) {
        try {
            int nFloats = bytes.length / (Float.SIZE / 8);
            float[] floats = new float[nFloats];
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(bytes));
            for (int i = 0; i < nFloats; i++) {
                if (dis.available() >= (Float.SIZE / 8)) {
                    floats[i] = dis.readFloat();
                }
            }
            dis.close();
            return floats;
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }

    @Override
    protected void finalize() throws Throwable {
        closeFile();
        super.finalize();
    }
}
