/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

package org.broad.igv.h5;

/**
 * An HDF55 reader handles read access to an HDF5 file,  either local or
 * remote.  
 * 
 * @author jrobinso
 */
public interface HDF5Reader {

    public void closeFile();

    /**
     * Open an HDF5 dataset in the given file.  
     * 
     * @param fileId - the HDF5 handle to the file
     * @param datasetName
     * @return a handle to the dataset
     */
    int openDataset(String datasetName);

    /**
     * 
     * @param datasetId
     */
    void closeDataset(int datasetId);

    /**
     * Open an HDF5 data group.  
     * @param parentNodeId - HDF5 handle to a parent object.  The group path is relative to this node
     * 
     * @param groupPath
     * @return
     */
    int openGroup(String groupPath);

    /**
     * Close an hdf5 group
     * 
     * @param groupId
     */
    public void closeGroup(int groupId);

    /**
     * Return value of the integer attribute for the specified group
     * 
     * @param groupId
     * @param attrName
     * @return
     * @throws org.broad.igv.exception.ObjectNotFoundException
     */
    public int readIntegerAttribute(int groupId, String attrName) throws ObjectNotFoundException;

    /**
     * Return value of the string attribute for the specified group
     * 
     * @param groupId
     * @param attrName
     * @return
     * @throws org.broad.igv.exception.ObjectNotFoundException
     */
    public String readStringAttribute(int groupId, String attrName);

    /**
     * Return value of the double attribute for the specified group
     * 
     * @param groupId
     * @param attrName
     * @return
     */
    public double readDoubleAttribute(int groupId, String attrName);

    /**
     * Read the entire dataset as an array of floats.
     * 
     * @param datasetId
     * @return
     */
    float[] readAllFloats(int datasetId);

    /**
     * Read the entire dataset as an array of ints.  
     * 
     * @param datasetId
     * @return
     */
    int[] readAllInts(int datasetId);

    /**
     * Read the entire dataset as an array of strings.
     * 
     * @param datasetId
     * @return
     */
    String[] readAllStrings(int datasetId);

    /**
     * Read a section of the dataset as an array of floats.
     * 
     * @param datasetId
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public float[] readFloats(int datasetId, int fromIndex, int toIndex);

    /**
     * Read a section of the dataset as an array of ints.
     * 
     * @param datasetId
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public int[] readInts(int datasetId, int fromIndex, int toIndex);

    /**
     * Read a section of the dataset as a 2-D array of floats.
     * 
     * @param fromIndex  Start column index
     * @param toIndex  End column index
     * @return
     */
    public float[][] readDataSlice(int datasetId, int fromIndex, int toIndex);
}
