/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.h5;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.log4j.Logger;

/**
 *
 * @author jrobinso
 */
public class H5BinReader extends H5BinBase implements HDF5Reader {

    static final Logger log = Logger.getLogger(H5BinReader.class);
    File file;
    //DataInputStream dis = null;
    FileInputStream fis = null;
    FileChannel channel = null;
    Map<String, Long> datasetIndex;

    public H5BinReader(File file) {
        this.file = file;

        try {
            int fileId = file.getAbsolutePath().hashCode();
            addEntity(fileId, "/");

            fis = new FileInputStream(file);
            channel = fis.getChannel();

            int version = readInt();
            readGroups();

        } catch (IOException ex) {
            log.error("Error creating file: " + file, ex);
            throw new RuntimeException("Error creating file: " + file, ex);
        }

    }

    /**
     * Read groups and database indexes
     */
    private synchronized void readGroups() throws IOException {

        long lastPosition = file.length() - Long.SIZE / 8;
        channel.position(lastPosition);
        long magicNumber = readLong();
        channel.position(magicNumber);

        int nBytes = (int) (lastPosition - magicNumber);
        byte[] bytes = new byte[nBytes];
        readFully(bytes);

        DataInputStream bdis = new DataInputStream(new ByteArrayInputStream(bytes));

        // Read dataset index
        int nDatasets = bdis.readInt();
        datasetIndex = new LinkedHashMap(nDatasets);
        for (int i = 0; i < nDatasets; i++) {
            String name = bdis.readUTF();
            long fPosition = bdis.readLong();
            datasetIndex.put(name, fPosition);
        }

        // read groups
        int nGroups = bdis.readInt();
        groupCache = new HashMap(nGroups);
        for (int i = 0; i < nGroups; i++) {
            String name = bdis.readUTF();
            int id = name.hashCode();
            Group g = new Group(name);
            this.groupCache.put(id, g);

            int nAttributes = bdis.readInt();

            g.attributes = new HashMap(nAttributes);
            for (int j = 0; j < nAttributes; j++) {
                String key = bdis.readUTF();
                String value = bdis.readUTF();
                g.attributes.put(key, value);
            }

        // TODO -- put group in cache

        }
    }

    private synchronized Dataset readDataset() throws IOException {

        int nBytes = readInt();
        byte[] buffer = new byte[nBytes];
        readFully(buffer);

        DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));
        String fullName = dis.readUTF();
        String typeString = dis.readUTF();
        DataType type = DataType.valueOf(typeString);
        int nRows = dis.readInt();
        int nCols = dis.readInt();
        Dataset dataset = new Dataset(fullName, nRows, nCols, type);

        int nAttrs = dis.readInt();
        if (nAttrs > 0) {
        }

        dataset.rowPositions = new long[nRows];


        for (int i = 0; i < nRows; i++) {
            dataset.rowPositions[i] = dis.readLong();
        }
        return dataset;

    }

    public void closeFile() {
        try {
            fis.close();
        } catch (IOException ex) {
            log.error("Error closing file: " + file, ex);
        }
    }

    public synchronized int openDataset(String datasetName) {
        try {
            if (entityIdMap.containsKey(datasetName)) {
                return entityIdMap.get(datasetName);
            } else {
                int id = datasetName.hashCode();
                entityIdMap.put(datasetName, id);
                long dsFilePosition = datasetIndex.get(datasetName);

                channel.position(dsFilePosition);
                Dataset dataset = readDataset();
                datasetCache.put(id, dataset);

                return id;
            }
        } catch (IOException ex) {
            log.error("Error opening dataset: " + datasetName, ex);
            throw new RuntimeException("Error opening dataset: " + datasetName, ex);
        }
    }

    public void closeDataset(int datasetId) {
        // No-op
    }

    public int openGroup(String groupPath) {
        int id = groupPath.hashCode();
        entityIdMap.put(groupPath, id);
        return id;
    }

    public void closeGroup(int groupId) {
        // no op
    }

    public int readIntegerAttribute(int groupId, String attrName) throws ObjectNotFoundException {
        String value = readStringAttribute(groupId, attrName);
        return Integer.parseInt(value);
    }

    public String readStringAttribute(int groupId, String attrName) {
        Group group = groupCache.get(groupId);
        if (group == null) {
            throw new ObjectNotFoundException("Group not found");
        }
        String value = group.attributes.get(attrName);
        if (value == null) {
            throw new ObjectNotFoundException("Attribute not found: " + attrName);
        }
        return value;
    }

    public double readDoubleAttribute(int groupId, String attrName) {
        String value = readStringAttribute(groupId, attrName);
        return Double.parseDouble(value);
    }

    /**
     * Read all values from the dataset as an array of floats.  Note that this method implicitly
     * assumes a 1-d dataset.
     * @param datasetId
     * @return
     */
    public synchronized float[] readAllFloats(int datasetId) {
        try {
            Dataset ds = datasetCache.get(datasetId);
            assert (ds.nRows == 1);

            int nBytes = ds.nColumns * Float.SIZE / 8;
            byte[] buffer = new byte[nBytes];
            readFully(buffer);
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));

            float[] data = new float[ds.nColumns];
            channel.position(ds.rowPositions[0]);
            for (int i = 0; i < ds.nColumns; i++) {
                data[i] = dis.readFloat();
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    public synchronized int[] readAllInts(int datasetId) {
        try {
            Dataset ds = datasetCache.get(datasetId);
            assert (ds.nRows == 1);
            int[] data = new int[ds.nColumns];
            channel.position(ds.rowPositions[0]);

            int nBytes = ds.nColumns * Integer.SIZE / 8;
            byte[] buffer = new byte[nBytes];
            readFully(buffer);
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));

            for (int i = 0; i < ds.nColumns; i++) {
                data[i] = dis.readInt();
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    public synchronized String[] readAllStrings(int datasetId) {
        try {
            Dataset ds = datasetCache.get(datasetId);
            assert (ds.nRows == 1);
            String[] data = new String[ds.nColumns];
            channel.position(ds.rowPositions[0]);

            int nBytes = readInt();
            byte[] buffer = new byte[nBytes];
            readFully(buffer);
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));

            for (int i = 0; i < ds.nColumns; i++) {
                data[i] = dis.readUTF();
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    public synchronized float[] readFloats(int datasetId, int fromIndex, int toIndex) {
        try {
            Dataset ds = datasetCache.get(datasetId);
            assert (ds.nRows == 1);
            assert (fromIndex >= 0);
            assert (toIndex <= ds.nColumns);

            int nPts = toIndex - fromIndex;
            float[] data = new float[nPts];
            long offset = fromIndex * Float.SIZE / 8;
            channel.position(ds.rowPositions[0] + offset);

            int nBytes = nPts * Float.SIZE / 8;
            byte[] buffer = new byte[nBytes];
            readFully(buffer);
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));


            for (int i = 0; i < nPts; i++) {
                data[i] = dis.readFloat();
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    public synchronized int[] readInts(int datasetId, int fromIndex, int toIndex) {
        try {

            Dataset ds = datasetCache.get(datasetId);
            assert (ds.nRows == 1);
            assert (fromIndex >= 0);
            assert (toIndex <= ds.nColumns);

            int nPts = toIndex - fromIndex;
            int[] data = new int[nPts];
            long offset = fromIndex * Integer.SIZE / 8;
            channel.position(ds.rowPositions[0] + offset);

            int nBytes = nPts * Integer.SIZE / 8;
            byte[] buffer = new byte[nBytes];
            readFully(buffer);
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));

            for (int i = 0; i < nPts; i++) {
                data[i] = dis.readInt();
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    public synchronized float[][] readDataSlice(int datasetId, int fromIndex, int toIndex) {
        try {
            Dataset ds = datasetCache.get(datasetId);
            assert (fromIndex >= 0);
            assert (toIndex <= ds.nColumns);

            int nPts = toIndex - fromIndex;
            int nBytes = nPts * Float.SIZE / 8;
            byte[] buffer = new byte[nBytes];
            long offset = fromIndex * Float.SIZE / 8;

            float[][] data = new float[ds.nRows][nPts];
            for (int row = 0; row < ds.nRows; row++) {
                channel.position(ds.rowPositions[row] + offset);

                readFully(buffer);
                DataInputStream dis = new DataInputStream(new ByteArrayInputStream(buffer));

                for (int i = 0; i < nPts; i++) {
                    data[row][i] = dis.readFloat();
                }
            }
            return data;
        } catch (IOException ex) {
            log.error("Error reading datset", ex);
            throw new RuntimeException("Error reading datset", ex);
        }
    }

    /**
    
     * Bytes for this operation are read from the contained
     * input stream.
     *
     * @param      b     the buffer into which the data is read.
     *
     * @exception  EOFException  if this input stream reaches the end before
     *               reading all the bytes.
     * @exception  IOException   if an I/O error occurs.
    
     */
    private void readFully(byte b[]) throws IOException {
        int off = 0;
        int len = b.length;
        if (len < 0) {
            throw new IndexOutOfBoundsException();
        }
        int n = 0;
        while (n < len) {
            int count = fis.read(b, off + n, len - n);
            if (count < 0) {
                throw new EOFException();
            }
            n += count;
        }
    }

    private int readInt() throws IOException {
        int ch1 = fis.read();
        int ch2 = fis.read();
        int ch3 = fis.read();
        int ch4 = fis.read();
        if ((ch1 | ch2 | ch3 | ch4) < 0) {
            throw new EOFException();
        }
        return ((ch1 << 24) + (ch2 << 16) + (ch3 << 8) + (ch4 << 0));
    }

    private long readLong() throws IOException {
        byte[] readBuffer = new byte[8];
        readFully(readBuffer);
        return (((long) readBuffer[0] << 56) +
                ((long) (readBuffer[1] & 255) << 48) +
                ((long) (readBuffer[2] & 255) << 40) +
                ((long) (readBuffer[3] & 255) << 32) +
                ((long) (readBuffer[4] & 255) << 24) +
                ((readBuffer[5] & 255) << 16) +
                ((readBuffer[6] & 255) << 8) +
                ((readBuffer[7] & 255) << 0));
    }
}
