/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.h5;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;

/**
 *
 * @author jrobinso
 */
public class H5BinBase {

    enum DataType {

        BYTE, SHORT, INT, FLOAT, DOUBLE, STRING
    };
    /** Index of chr -> datasetName -> position */
    protected Map<String, Long> groupIndex;
    protected Map<String, Integer> entityIdMap;
    protected Map<Integer, Group> groupCache;
    protected Map<Integer, Dataset> datasetCache;

    public static class Entity {

        String fullName;
        Map<String, String> attributes = new HashMap();
    }

    public static class Group extends Entity {

        Group(String fullName) {
            this.fullName = fullName;
        }

        boolean isEmpty() {
            return attributes.isEmpty();
        }

        void dump(PrintStream out) {
            out.println(fullName);
            for (Map.Entry<String, String> entry : attributes.entrySet()) {
                out.println("\t" + entry.getKey() + " = " + entry.getValue());
            }
        }
    }

    public static class Dataset extends Entity {
        int nRows;
        int nColumns;
        DataType type;
        long[] rowPositions;

        Dataset(String fullName, int nRows, int nColumns, DataType type) {
            this.fullName = fullName;
            this.type = type;
            this.nRows = nRows;
            this.nColumns = nColumns;
            rowPositions = new long[nRows];
        }

        void dump(PrintStream out) {
            out.println(fullName + "  [" + nRows + "," + nColumns + "]");
            for (Map.Entry<String, String> entry : attributes.entrySet()) {
                out.println("\t" + entry.getKey() + " = " + entry.getValue());
            }
        }
    }

    public H5BinBase() {
        entityIdMap = new HashMap();
        groupCache = new HashMap();
        datasetCache = new HashMap();

    }

    protected String getFullName(int locId, String name) {

        // Names that start with / are absolute already
        if (name.startsWith("/")) {
            return name;
        }

        String entityName = groupCache.get(locId).fullName;
        if (!entityName.endsWith("/")) {
            entityName = entityName + "/";
        }
        String fullName = entityName + name;
        return fullName;
    }

    protected Group getParentGroup(int locId, String name) {
        String fullPath = getFullName(locId, name);
        int idx = fullPath.lastIndexOf("/");
        String parentName = fullPath.substring(0, idx);
        if (!entityIdMap.containsKey(idx)) {
            throw new java.lang.IllegalArgumentException("No parent group found for: " + fullPath);
        } else {
            int parentId = entityIdMap.get(parentName);
            return groupCache.get(parentId);
        }

    }

    protected void addEntity(Integer id, String name) {
        groupCache.put(id, new Group(name));
        entityIdMap.put(name, id);
    }

    protected int getEntityId(String name) {
        if (!entityIdMap.containsKey(name)) {
            System.out.println();
        }
        return entityIdMap.get(name);
    }

    protected Group getEntity(int id) {
        return groupCache.get(id);
    }

    protected String getEntityName(int id) {
        return groupCache.get(id).fullName;
    }

    protected void removeEntity(int id) {
        Group entity = groupCache.get(id);
        groupCache.remove(id);
        entityIdMap.remove(entity.fullName);
    }
}
