/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */



/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------

import org.apache.log4j.Logger;

import org.broad.igv.util.ObjectCache;
import org.broad.igv.remote.SequenceServletWrapper;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

/**
 *
 * @author jrobinso
 */
public class SequenceManager {

    private static Logger logger = Logger.getLogger(SequenceManager.class);
    private static boolean cacheSequences = true;
    private static int chunkSize = 1000;
    static ObjectCache<String, SequenceChunk> sequenceCache = new ObjectCache(50);

    /**
     * Return the reference dna sequence for the exact interval specified.
     * @param genome
     * @param chr
     * @param start
     * @param end
     * @return
     */
    public static byte[] readSequence(String genome, String chr, int start, int end) {

        File rootDirectory = null;

        try
        {
            GenomeDescriptor descriptor = 
                GenomeManager.getInstance().getGenomeDescriptor(genome);
            if (descriptor != null)
            {
                String location = descriptor.getSequenceLocation();
                if(location != null) {
                    if (location.toLowerCase().startsWith("http:"))
                    {
                        return readSequenceRemote(genome, chr, start, end);
                    }
                    else
                    {                
                        rootDirectory = new File(location);                   
                        return readSequenceLocal(rootDirectory, genome, chr, start, end);
                    }
                }
            }
        }
        catch (Exception e)
        {
            logger.error(e.getMessage(), e);
        }

        return new byte[0];
    }

    /**
     * Return the reference dna sequence for the exact interval specified.
     * @param genome
     * @param chr
     * @param start
     * @param end
     * @return
     */
    private static byte[] readSequenceRemote(String genome, String chr, int start, int end) {
        if (cacheSequences)
        {
            byte[] sequence = new byte[end - start];
            int startTile = start / chunkSize;
            int endTile = end / chunkSize;

            // Get first chunk
            SequenceChunk chunk = getSequenceChunk(genome, chr, startTile);
            int offset = start - chunk.getStart();
            byte[] seqBytes = chunk.getBytes();

            if (seqBytes == null)
            {
                return null;
            }

            int nBytes = Math.min(seqBytes.length - offset, sequence.length);

            System.arraycopy(chunk.getBytes(), offset, sequence, 0, nBytes);

            for (int tile = startTile + 1; tile <= endTile; tile++)
            {
                chunk = getSequenceChunk(genome, chr, tile);

                int nNext = Math.min(sequence.length - nBytes, chunk.getSize());

                System.arraycopy(chunk.getBytes(), 0, sequence, nBytes, nNext);
                nBytes += nNext;
            }

            return sequence;
        }
        else
        {
            return SequenceServletWrapper.readBytes(genome, chr, start, end);
        }
    }

    /**
     * Read sequence from a local file directory, as opposed to a servlet connection.
     * @param rootDir
     * @param genome
     * @param chr
     * @param start
     * @param end
     * @return
     */
    private static byte[] readSequenceLocal(File seqDir, String genome, String chr, int start,
            int end) {
        DataInputStream is = null;
        try {
            // File inputFile = new File("hg18.chr1.bin.gz");
            if (seqDir.exists()) {
                File inputFile = new File(seqDir, chr + ".txt");

                if (inputFile.exists()) {
                    byte[] bytes = new byte[end - start + 1];
                    is = new DataInputStream(
                            new BufferedInputStream(
                            new FileInputStream(inputFile)));
                    is.skip(start);
                    is.read(bytes);
                    return bytes;
                }
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        finally {
            if(is != null) {
                try {
                    is.close();
                } catch (IOException ex) {
                    logger.error("Error closing sequence file.", ex);
                }
            }
        }

        return null;
    }

    private static SequenceChunk getSequenceChunk(String genome, String chr, int tileNo) {
        String key = getKey(genome, chr, tileNo);
        SequenceChunk chunk = sequenceCache.get(key);

        if (chunk == null)
        {
            int start = tileNo * chunkSize;
            int end = start + chunkSize - 1;
            byte[] seq = SequenceServletWrapper.readBytes(genome, chr, start, end);

            chunk = new SequenceChunk(start, end, seq);
            sequenceCache.put(key, chunk);
        }

        return chunk;
    }

    static String getKey(String genome, String chr, int tileNo) {
        return genome + chr + tileNo;
    }

    /**
     * This accessor provided to support unit tests.
     * @param aChunkSize
     */
    static void setChunkSize(int aChunkSize) {
        chunkSize = aChunkSize;
    }

    /**
     * Accessor to support unit tests.
     * @param aCacheSequences
     */
    static void setCacheSequences(boolean aCacheSequences) {
        cacheSequences = aCacheSequences;
    }

    static class SequenceChunk {

        private int start;
        private int end;
        private byte[] bytes;

        SequenceChunk(int start, int end, byte[] bytes) {
            this.start = start;
            this.end = end;
            this.bytes = bytes;
        }

        public int getStart() {
            return start;
        }

        public int getEnd() {
            return end;
        }

        public int getSize() {
            return bytes.length;
        }

        public byte[] getBytes() {
            return bytes;
        }
    }
}
