/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import cern.colt.map.OpenIntObjectHashMap;
import org.apache.log4j.Logger;

import org.broad.igv.ui.IGVMainFrame;


//~--- JDK imports ------------------------------------------------------------

import java.io.*;

import java.net.URL;
import java.net.URLConnection;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.zip.GZIPInputStream;

import javax.swing.JOptionPane;
import org.broad.igv.PreferenceManager;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.util.AsciiLineReader;

/**
 * @author: Marc-Danie Nazaire
 *     private static String affyMappingURL =
"http://www.broadinstitute.org/igv/resources/probes/affy_probe_gene_mapping.txt.gz";
private static String agilentMappingURL =
"http://www.broadinstitute.org/igv/resources/probes/agilent_probe_gene_mapping.txt.gz";
private static String illuminaMappingURL =
"http://www.broadinstitute.org/igv/resources/probes/illumina_probe_gene_mapping.txt.gz";
 */
public class ProbeToGeneMap {

    private static Logger log = Logger.getLogger(ProbeToGeneMap.class);
    private static String affyGenesMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/affy/affy_probe_gene_mapping.txt.gz";
    private static String affyHumanMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/affy/affy_human_mappings.txt.gz";
    private static String affyMouseMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/affy/affy_mouse_mappings.txt.gz";
    private static String affyOtherMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/affy_other_mappings.txt.gz";
    private static String agilentGenesMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/agilent/agilent_probe_gene_mapping.txt.gz";
    private static String agilentHumanMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/agilent/agilent_human_mappings.txt.gz";
    private static String agilentMouseMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/agilent/agilent_mouse_mappings.txt.gz";
    private static String agilentOtherMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/agilent/agilent_other_mappings.txt.gz";
    private static String illuminaMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/illumina/illumina_allMappings.txt.gz";
    private static String illuminaGenesMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/illumina/illumina_probe_gene_mapping.txt.gz";
    private static String methylationGeneMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/meth/methylation_pobeToGene.tab.gz";
    private static String methylationLociMappingURL =
            "http://www.broadinstitute.org/igv/resources/probes/meth/methylation_probeToLoci.mappings.txt.gz";
    private static OpenIntObjectHashMap rnaiMap;
    private static ProbeToGeneMap instance;
    Map<String, OpenIntObjectHashMap> probeMaps = new HashMap();
    MappingUrlCache mappingUrlCache = new MappingUrlCache();

    enum Platform {

        Affymetrix, Agilient, Illumina, Methylation, Mirna, unknown
    };

    /**
     * Method description
     *
     *
     * @return
     */
    public static synchronized ProbeToGeneMap getInstance() {
        if (instance == null) {
            instance = new ProbeToGeneMap();
        }

        return instance;
    }

    public void clearProbeMappings() {
        if (probeMaps != null) {
            probeMaps.clear();
        }
        if (mappingUrlCache != null) {
            mappingUrlCache.clear();
        }
    }

    /**
     * Method description
     *
     *
     * @param urlString
     * @param map
     *
     * @throws Exception
     */
    public void loadMapping(String urlString, OpenIntObjectHashMap map) throws Exception {
        AsciiLineReader bufReader = null;
        try {
            InputStream is = null;

            if (urlString.startsWith("http") || urlString.startsWith("file")) {
                URL url = new URL(urlString);
                URLConnection connection = url.openConnection();
                is = connection.getInputStream();
            } else {
                is = new FileInputStream(urlString);
            }
            if (urlString.endsWith("gz")) {
                is = new GZIPInputStream(is);
            }
            bufReader = new AsciiLineReader(is);
            loadMapping(bufReader, map);

        } catch (Exception e) {
            log.error("Error loading probe mapping", e);

            throw e;
        } finally {
            if (bufReader != null) {
                bufReader.close();
            }
        }
    }

    public void loadMapping(AsciiLineReader bufReader, OpenIntObjectHashMap map) throws IOException {
        String line;
        String[] result = new String[100];
        while ((line = bufReader.readLine()) != null) {
            int nTokens = ParsingUtils.split(line, result, '\t');
            if (nTokens != 2) {
                continue;
            }
            // check if more than one gene symbol found
            String[] genes = result[1].split("///");

            map.put(result[0].trim().hashCode(), genes);
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public OpenIntObjectHashMap getRNAiProbeMap() {
        return rnaiMap;
    }

    public String getMappingURL(String genomeId, Platform platform) {

        if (platform == Platform.unknown) {
            return null;
        }

        String mappingUrl = mappingUrlCache.getMappingUrl(genomeId, platform);
        if (mappingUrl == null) {

            boolean mapToGenes = PreferenceManager.getInstance().isMapProbesToGenes();
            if (!mapToGenes) {
                boolean hasLociMapping = checkForLociMapping(platform, genomeId);
                if (!hasLociMapping) {
                    Genome genome = GenomeManager.getInstance().getGenome(genomeId);
                    String genomeName = genome.getName();
                    JOptionPane.showMessageDialog(IGVMainFrame.getInstance(),
                            "<html>" + platform.toString() + " probe locations are not available for the selected genome " +
                            " (" + genomeName + "). <br>Expression data will be mapped to gene locations.");
                    mapToGenes = true;
                }
            }

            if (platform == Platform.Affymetrix) {
                if (mapToGenes) {
                    mappingUrl = affyGenesMappingURL;
                } else if (genomeId.startsWith("hg")) {
                    mappingUrl = affyHumanMappingURL;

                } else if (genomeId.startsWith("mm")) {
                    mappingUrl = affyMouseMappingURL;

                } else {
                    mappingUrl = affyOtherMappingURL;
                }
            } else if (platform == Platform.Agilient) {
                if (mapToGenes) {
                    mappingUrl = agilentGenesMappingURL;
                } else if (genomeId.startsWith("hg")) {
                    mappingUrl = agilentHumanMappingURL;

                } else if (genomeId.startsWith("mm")) {
                    mappingUrl = agilentMouseMappingURL;
                } else {
                    mappingUrl = agilentOtherMappingURL;
                }
            } else if (platform == Platform.Illumina) {
                if (mapToGenes) {
                    mappingUrl = illuminaGenesMappingURL;
                } else {
                    mappingUrl = illuminaMappingURL;
                }
            } else if (platform == Platform.Methylation) {
                if (mapToGenes) {
                    mappingUrl = methylationGeneMappingURL;
                } else {
                    mappingUrl = methylationLociMappingURL;
                }
            } else {
                return null;
            }
            mappingUrlCache.put(genomeId, platform, mappingUrl);
        }
        return mappingUrl;

    }

    public static Platform getPlatform(String probeId) {
        if (probeId.endsWith("_at") || probeId.endsWith("_st")) {
            return Platform.Affymetrix;
        } else if (probeId.startsWith("A_")) {
            return Platform.Agilient;
        } else if (probeId.startsWith("ILMN_") || probeId.startsWith("GI_") || probeId.startsWith(
                "NM_") || probeId.startsWith("XM_")) {
            return Platform.Illumina;
        } else if (probeId.startsWith("cg")) {
            return Platform.Methylation;
        }
        return Platform.unknown;
    }

    //mm9 (Affymetrix), mm5 (Agilent) or mm8 (Illumina)
    private static boolean checkForLociMapping(Platform platform, String genomeId) {

        boolean hasLociMapping = true;
        if (genomeId.startsWith("hg") && !genomeId.equals("hg18")) {
            hasLociMapping = false;
        } else if (genomeId.startsWith("mm")) {
            switch (platform) {
                case Affymetrix:
                    if (!genomeId.equals("mm9")) {
                        hasLociMapping = false;
                    }
                    break;

                case Agilient:
                    if (!genomeId.equals("mm5")) {
                        hasLociMapping = false;
                    }
                    break;

                case Illumina:
                    if (!genomeId.equals("mm8")) {
                        hasLociMapping = false;
                    }
                    break;
            }
        }
        return hasLociMapping;
    }

    public String[] getGenesForProbe(String probeId) throws Exception {

        int probeHash = probeId.hashCode();
        String genomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        Platform platform = getPlatform(probeId);
        if (platform == Platform.unknown) {
            return null;
        }
        String mappingURL = getMappingURL(genomeId, platform);
        if (mappingURL == null) {
            return null;
        }

        try {

            OpenIntObjectHashMap pMap = probeMaps.get(mappingURL);
            if (pMap == null) {
                pMap = new OpenIntObjectHashMap(500000);
                loadMapping(mappingURL, pMap);
                probeMaps.put(mappingURL, pMap);
            }
            return pMap == null ? null : (String[]) pMap.get(probeHash);

        } catch (Exception e) {
            JOptionPane.showMessageDialog(
                    IGVMainFrame.getInstance(),
                    "<html>Could not connect to the server to download probe to gene mappings.<br>" + "Please check your internet connection");
            throw e;
        }
    }


    // TODO -- this could be generalized to a Generic 2 key cache
    static class MappingUrlCache {

        Map<Platform, Map<String, String>> cache = new Hashtable();

        public void clear() {
            cache.clear();
        }

        public String getMappingUrl(String genomeId, Platform platform) {
            if (cache.containsKey(platform)) {
                Map<String, String> urlMap = cache.get(platform);
                return urlMap.get(genomeId);
            } else {
                return null;
            }
        }

        public void put(String genomeId, Platform platform, String mappingUrl) {
            if (!cache.containsKey(platform)) {
                cache.put(platform, new Hashtable());
            }
            cache.get(platform).put(genomeId, mappingUrl);
        }
    }

    public static void main(String[] args) throws Exception {
        OpenIntObjectHashMap map = new OpenIntObjectHashMap(1000000);
    //ProbeToGeneMap.getInstance().loadMapping(ProbeToGeneMap.agilentMappingURL, map);
    //System.out.println("sz = " + map.size());
    //System.in.read();
    }
}
