/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.util.AsciiLineReader;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.net.URL;
import java.net.URLConnection;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import org.broad.igv.ui.IGVModel;

/**
 *
 * @author jrobinso
 */
public class ParsingUtils {

    private static Logger log = Logger.getLogger(ParsingUtils.class);

    public static BufferedReader openBufferedReader(String pathOrUrl)
            throws FileNotFoundException, IOException {
        BufferedReader reader = null;

        if (pathOrUrl.startsWith("http:") || pathOrUrl.startsWith("file:")) {
            URL url = new URL(pathOrUrl);
            URLConnection connection = url.openConnection();
            reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
        } else {
            File file = new File(pathOrUrl);

            FileInputStream fileInput = new FileInputStream(file);
            if (file.getName().endsWith("gz")) {
                GZIPInputStream in = new GZIPInputStream(fileInput);
                reader = new BufferedReader(new InputStreamReader(in));
            } else {
                reader = new BufferedReader(new InputStreamReader(fileInput));
            }
        }

        return reader;
    }

    public static int estimateLineCount(String filename) {

        AsciiLineReader reader = null;
        try {
            File file = new File(filename);
            if (!file.exists()) {
                return -1;
            }
            reader = openAsciiReader(new ResourceLocator(filename));
            int totalBytes = 0;
            String nextLine;
            int lines = 0;
            while ((nextLine = reader.readLine(true)) != null & lines < 100) {
                lines++;
                totalBytes += nextLine.length();
            }
            int bytesPerLine = totalBytes / lines;
            int nLines = (int) (file.length() / bytesPerLine);
            return nLines;

        } catch (Exception e) {
            log.error("Error estimating line count", e);
            return -1;
        } finally {
            reader.close();
        }

    }

    /**
     * Method description
     *
     *
     * @param locator
     *
     * @return
     *
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static AsciiLineReader openAsciiReader(ResourceLocator locator)
            throws FileNotFoundException, IOException {
        InputStream stream = openInputStream(locator);
        return new AsciiLineReader(stream);

    }

    /**
     * Method description
     *
     *
     * @param locator
     *
     * @return
     *
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static InputStream openInputStream(ResourceLocator locator)
            throws FileNotFoundException, IOException {

        if (locator.isLocal()) {
            File file = new File(locator.getPath());

            FileInputStream fileInput = new FileInputStream(file);
            if (file.getName().endsWith("gz")) {
                return new GZIPInputStream(fileInput);

            } else {
                return fileInput;
            }
        } else {
            URL url = new URL(
                    locator.getServerURL() + "?method=getContents&file=" + locator.getPath());
            URLConnection connection = url.openConnection();

            // Note -- assumption that url stream is compressed!
            try {
                return new GZIPInputStream(connection.getInputStream());
            } catch (Exception ex) {
                log.error("Error with gzip stream", ex);
                throw new RuntimeException(
                        "There was a server error loading file: " + locator.getDisplayName() +
                        ". Please report to igv-help@broadinstitute.org");

            }
        }
    }

    /**
     * Split the string into tokesn separated by the given delimiter.  Profiling has
     * revealed that the standard string.split() method typically takes > 1/2
     * the total time when used for parsing ascii files.
     *
     * @param aString  the string to split
     * @param tokens an array to hold the parsed tokens
     * @param delim  character that delimits tokens
     * @return the number of tokens parsed
     */
    public static int split(String aString, String[] tokens, char delim) {

        int maxTokens = tokens.length;
        int nTokens = 0;
        int start = 0;
        int end = aString.indexOf(delim);
        if (end < 0) {
            tokens[nTokens++] = aString;
            return nTokens;
        }
        while ((end > 0) && (nTokens < maxTokens)) {
            //tokens[nTokens++] = new String(aString.toCharArray(), start, end-start); //  aString.substring(start, end);
            tokens[nTokens++] = aString.substring(start, end);
            start = end + 1;
            end = aString.indexOf(delim, start);

        }

        // Add the trailing string,  if there is room and if it is not empty.
        if (nTokens < maxTokens) {
            String trailingString = aString.substring(start);
            if (trailingString.length() > 0) {
                tokens[nTokens++] = trailingString;
            }
        }
        return nTokens;
    }

    /**
     * Method description
     *
     *
     * @param file
     *
     * @return
     */
    public static List<String> loadRegions(File file) {
        try {
            FileInputStream fileInput = new FileInputStream(file);
            BufferedReader reader = new BufferedReader(new InputStreamReader(fileInput));
            String nextLine;
            List<String> features = new ArrayList<String>();
            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0)) {
                try {
                    if (nextLine.startsWith("chr")) {
                        String[] tokens = nextLine.split("\t");
                        String region = tokens[0] + ":" + tokens[1] + "-" + tokens[2];
                        features.add(region);
                    }
                } catch (NumberFormatException e) {
                    log.error("Error parsing numer in line: " + nextLine);
                }
            }

            reader.close();
            return features;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Method description
     *
     *
     * @param gene
     */
    public static void computeReadingShifts(Feature gene) {
        List<Exon> exons = gene.getExons();
        if (exons.size() == 0) {
            return;
        }

        int startIndex = (gene.getStrand() == Strand.POSITIVE) ? 0 : exons.size() - 1;
        int endIndex = (gene.getStrand() == Strand.POSITIVE) ? exons.size() : -1;
        int increment = (gene.getStrand() == Strand.POSITIVE) ? 1 : -1;
        int cds = 0;
        int exonNumber = 1;
        for (int i = startIndex; i != endIndex; i += increment) {
            Exon exon = exons.get(i);
            exon.setNumber(exonNumber);
            int modCds = cds % 3;
            int phase = (modCds == 0) ? 0 : 3 - modCds;
            exon.setPhase(phase);
            cds += exon.getCodingLength();
            exonNumber++;
        }
    }
    /*
     * Legacy conversion for human genome files.
     *
     * @param chr
     * @return
     */
    /**
     * Method description
     *
     *
     * @param genomeId
     * @param chr
     *
     * @return
     */
    static Set<String> legacyGenomes = new HashSet();


    static {
        legacyGenomes.add("S._cerevisiae");
        legacyGenomes.add("Yeast_S._pombe");
        legacyGenomes.add("Chicken_galGal3");
    }
    /**
     * Method description
     *
     *
     * @param genomeId
     * @param chr
     *
     * @return
     */
    private static Map<String, String> chrLookupTable = new HashMap(100);

    // Preseed chr table with "1,2,3,  etc"


    static {
        for (int i = 0; i < 23; i++) {
            chrLookupTable.put(String.valueOf(i), "chr" + i);
        }
        chrLookupTable.put("X", "chrX");
        chrLookupTable.put("Y", "chrY");
        chrLookupTable.put("M", "chrM");
        chrLookupTable.put("x", "chrX");
        chrLookupTable.put("y", "chrY");
        chrLookupTable.put("m", "chrM");
    }

    public static String convertChrString(String str) {
        String genomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        return convertChrString(genomeId, str);
    }

    public static String convertChrString(String genomeId, String str) {

        if (str == null) {
            return null;
        }

        if (genomeId != null) {
            if (genomeId.startsWith("hg")) {
                str = str.replace("23", "X");
                str = str.replace("24", "Y");
            } else if (genomeId.startsWith("mm")) {
                str = str.replace("20", "X");
                str = str.replace("21", "Y");
            }
        }
        String chr = chr = chrLookupTable.get(str);
        if (chr == null) {
            chr = str;
            chrLookupTable.put(str, chr);
        }

        return chr;

    }
}
