/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.feature;

import java.io.*;

import java.util.*;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.FeatureTrack;
import org.broad.igv.track.Track;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.util.AsciiLineReader;

/**
 * Parses a ".mut" file
 * 
 * @author jrobinso
 */
public class MutationParser {

    public static List<Track> loadMutationTracks(ResourceLocator locator) {

        List<Track> tracks = new ArrayList();
        Map<String, List<Feature>> features = loadMutations(locator);
        for (String runId : features.keySet())
        {
            FeatureTrack track = new FeatureTrack(locator, runId, features.get(runId));

            // Annotate the track as a "mutation" track. The only purpose of this
            // property is to control popup menus, its an ugly hack that will be
            // refactored out at some point.
            track.setMutationTrack(true);

            // Overrid default minimum height (10 for feature tracks).
            track.setMinimumHeight(0);

            computeWholeGenome(track);
            tracks.add(track);
        }
        return tracks;
    }

    /**
     * Return a map of runId -> list of mutation objects.   The "runId" field
     * is the track identifier (name) for mutation files.
     * 
     * @param file
     * @return
     */
    private static Map<String, List<Feature>> loadMutations(ResourceLocator locator) {
        try
        {

            AsciiLineReader reader = ParsingUtils.openAsciiReader(locator);
            String nextLine;

            // Skip first line
            reader.readLine();

            Map<String, List<Feature>> mutationMap = new LinkedHashMap();
            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0))
            {
                try
                {
                    String[] tokens = nextLine.split("\t");
                    if (tokens.length > 4)
                    {
                        String chr = ParsingUtils.convertChrString(tokens[0]);
                        int start = Integer.parseInt(tokens[1].trim());
                        int end = Integer.parseInt(tokens[2].trim());
                        String runId = tokens[3].trim();
                        String typeString = tokens[4].trim().replace(' ', '_');


                        Mutation.Type type;
                        try
                        {
                            type = Mutation.Type.valueOf(typeString);
                        } catch (Exception exception)
                        {
                            type = Mutation.Type.Unknown;
                        }

                        List<Feature> features = mutationMap.get(runId);
                        if (features == null)
                        {
                            features = new ArrayList();
                            mutationMap.put(runId, features);
                        }

                        Mutation mut = new Mutation(runId, chr, start, end, type);
                        if (tokens.length > 14)
                        {
                            mut.setMClass(tokens[13].trim());
                            mut.setNewBase(tokens[14].trim());
                        }

                        features.add(mut);
                    }
                } catch (NumberFormatException e)
                {
                    System.err.println("Error parsing line: " + nextLine);
                }
            }

            reader.close();
            return mutationMap;
        } catch (IOException e)
        {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Compute features for "chr All".  This is probably not the best way to do this, 
     * definitely not good for large tracks.  Need it now for mutation data.
     */
    public static void computeWholeGenome(FeatureTrack track) {

        String genomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        int unit = 1000;  // KB

        String chrAll = "All";
        List<Feature> allFeatures = new ArrayList(1000);
        Genome genome = GenomeManager.getInstance().getGenome(genomeId);
        if (genome == null)
        {
            throw new RuntimeException("Unknown genome: " + genomeId);
        }
        genome.getChromosomeNames();
        long offset = 0;
        for (String chr : genome.getChromosomeNames())
        {
            int chrLength = genome.getChromosome(chr).getLength();
            Collection<Feature> features = track.getFeatures(chr, 0, chrLength);
            if (features != null)
            {
                for (Feature f : features)
                {
                    Mutation m = (Mutation) f;
                    int start = (int) ((offset + m.getStart()) / unit);
                    int end = (int) ((offset + m.getEnd()) / unit);
                    String runId = m.getRunId();
                    Mutation.Type type = m.getMutationType();
                    allFeatures.add(new Mutation(runId, chrAll, start, end, type));
                }
            }
            offset += chrLength;
        }

        track.setFeatures(chrAll, allFeatures);
    }
}
