/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.track.WindowFunction;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a mutation
 * // TODO -- refactor this to not implement "Feature"
 *
 * @author jrobinso
 */
public class Mutation implements Feature {

    private String runId;
    private String chromosome;
    private int start;
    private int end;
    private Type type;
    private Color color;
    private String newBase;
    private String mClass;
    private static Map<Type, Color> colors;

    public enum Type {

        Indel, Missense, Nonsense, Splice_site, Synonymous, Targeted_Region, Unknown
    }
    private static Map<Mutation.Type, Color> theColorScheme;
    static Mutation.Type[] types = new Mutation.Type[]{
        Mutation.Type.Indel, Mutation.Type.Missense, Mutation.Type.Nonsense,
        Mutation.Type.Splice_site, Mutation.Type.Synonymous, Mutation.Type.Targeted_Region,
        Mutation.Type.Unknown
    };

    // TODO It would be nice is we make the ColorSchemeFactory produce
    // ColorScheme objects instead of ContinuousColorScheme objects. This way
    // sets of Color specific behavior could be retrieved from a single place
    // with common behavior. For now I need to use a Map since this object was
    // not using a ColorScheme at all. In the future we should replace
    // Map<Mutation.Type,Color> by ColorScheme.
    /**
     * Method description
     *
     *
     * @return
     */
    public static Map<Mutation.Type, Color> getColorScheme() {

        if (theColorScheme != null) {
            return theColorScheme;
        }

        theColorScheme = PreferenceManager.getInstance().getMutationColorScheme();

        // If Color Scheme info was not found in user's preferences
        // so the scheme is still null we use the default
        if (theColorScheme == null) {
            theColorScheme = getDefaultColorScheme();
        }

        return theColorScheme;
    }

    // TODO same changes as getColorScheme
    /**
     * Method description
     *
     *
     * @param colorScheme
     */
    public static void setColorScheme(Map<Mutation.Type, Color> colorScheme) {
        theColorScheme = colorScheme;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public static Map<Mutation.Type, Color> getDefaultColorScheme() {

        HashMap<Mutation.Type, Color> scheme = new HashMap<Mutation.Type, Color>();
        for (Mutation.Type mutType : types) {
            scheme.put(mutType, Mutation.getColor(mutType));
        }

        return scheme;
    }

    /**
     * Method description
     *
     */
    public static void reloadPreferences() {
        theColorScheme = PreferenceManager.getInstance().getMutationColorScheme();
    }

    private static Color getColor(Type type) {

        if (colors == null) {
            colors = new HashMap();
            colors.put(Type.Indel, Color.GREEN);
            colors.put(Type.Missense, new Color(161, 45, 238));
            colors.put(Type.Nonsense, new Color(103, 61, 123));
            colors.put(Type.Splice_site, Color.MAGENTA);
            colors.put(Type.Synonymous, Color.CYAN);
            colors.put(Type.Targeted_Region, Color.ORANGE);
        }

        return colors.containsKey(type) ? colors.get(type) : Color.black;
    }

    /**
     * Constructs ...
     *
     *
     * @param runId
     * @param chromosome
     * @param start
     * @param end
     * @param type
     */
    public Mutation(String runId, String chromosome, int start, int end, Type type) {
        this.runId = runId;
        this.chromosome = ParsingUtils.convertChrString(chromosome);
        this.start = start;
        this.end = end;
        this.type = type;
    }

    /**
     * Constructs ...
     *
     *
     * @param mutation
     */
    public Mutation(Mutation mutation) {
        this.runId = mutation.runId;
        this.chromosome = mutation.chromosome;
        this.start = mutation.start;
        this.end = mutation.end;
        this.type = mutation.type;
        this.color = mutation.color;
        this.newBase = mutation.newBase;
        this.mClass = mutation.mClass;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Mutation copy() {
        return new Mutation(this);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getRunId() {
        return runId;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getType() {
        return "mutation";
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Type getMutationType() {
        return type;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getName() {
        return type.toString();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDescription() {
        return getName();
    }

    /**
     * Method description
     *
     *
     * @param ignored
     *
     * @return
     */
    public String getValueString(double position, WindowFunction ignored) {
        if ((mClass != null) && (newBase != null)) {
            return getName() + "<br>" + "Class     = " + mClass + "<br>New Base = " + newBase;
        } else {
            return getName();
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean hasScore() {
        return false;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Strand getStrand() {
        return Strand.NONE;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean hasStrand() {
        return false;
    }

    /**
     * Method description
     *
     *
     * @param track
     *
     * @return
     */
    public boolean overlaps(Feature track) {
        return false;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getChromosome() {
        return chromosome;
    }

    /**
     * Method description
     *
     *
     * @param color
     */
    public void setColor(Color color) {

        // Ignore
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getColor() {
        Color c = getColorScheme().get(getMutationType());
        return c;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getStart() {
        return start;
    }

    /**
     * Method description
     *
     *
     * @param start
     */
    public void setStart(int start) {
        this.start = start;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getEnd() {
        return end;
    }

    /**
     * Method description
     *
     *
     * @param end
     */
    public void setEnd(int end) {
        this.end = end;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int[] getStarts() {
        return null;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int[] getEnds() {
        return null;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public float getScore() {
        return 0;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public float getConfidence() {
        return 1;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getNewBase() {
        return newBase;
    }

    /**
     * Method description
     *
     *
     * @param newBase
     */
    public void setNewBase(String newBase) {
        this.newBase = newBase;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getMClass() {
        return mClass;
    }

    /**
     * Method description
     *
     *
     * @param mClass
     */
    public void setMClass(String mClass) {
        this.mClass = mClass;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getLevel() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Return true if the feature is completely contained within the bounds of this
     * featre.
     * 
     * 
     * @param feature
     *
     * @return
     */
    public boolean contains(Feature feature) {

        if (feature == null || !this.getChromosome().equals(feature.getChromosome())) {
            return false;
        }
        if ((feature.getStart() >= this.getStart()) && (feature.getEnd() <= this.getEnd())) {
            return true;
        } else {
            return false;
        }
    }

    public boolean contains(double location) {
        return location >= start && location <= end;
    }

    /**
     * Method description
     *
     *
     * @param confidence
     */
    public void setConfidence(float confidence) {

        // ignore
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public List<Exon> getExons() {
        return null;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getIdentifier() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Method description
     *
     *
     * @param exonIndex
     *
     * @return
     */
    public AminoAcidSequence getAminoAcidSequence(int exonIndex) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getCdEnd() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getCdStart() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getLength() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public Map<String, String> getAttributes() {
        return null;
    }
}
