/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * GisticFilesParser.java
 *
 * Created on June 22, 2007, 8:32 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.GisticTrack;

//~--- JDK imports ------------------------------------------------------------

import java.io.FileNotFoundException;
import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public class GisticFileParser {

    private static Logger log = Logger.getLogger(GisticFileParser.class);

    /**
     * Method description
     *
     *
     * @param locator
     *
     * @return
     */
    public static GisticTrack loadData(ResourceLocator locator) {

        // TODO -- handle remote resource
        AsciiLineReader reader = null;

        try {
            reader = ParsingUtils.openAsciiReader(locator);

            // Parse header
            reader.readLine();

            // Parse data
            // parameters to track chromosome breaks. Used in wholeGenome case.
            GisticTrack track = new GisticTrack(locator, locator.getDisplayName());
            List<GisticScore> scores = new ArrayList();

            String nextLine = reader.readLine();
            while ((nextLine != null) && (nextLine.length() > 0)) {
                String[] tokens = nextLine.split("\t");
                try {
                    GisticScore.Type type = getType(tokens[0].trim());
                    String chr = "chr" + tokens[1].trim();
                    int start = Integer.parseInt(tokens[2]);
                    int end = Integer.parseInt(tokens[3]);
                    float qValue = 0;
                    try {
                        qValue = Float.parseFloat(tokens[4]);
                    } catch (NumberFormatException numberFormatException) {
                        if (tokens[4].toLowerCase().equals("inf")) {
                            qValue = Float.POSITIVE_INFINITY;
                        } else {
                            qValue = Float.NaN;
                        }
                    }
                    float gScore = Float.parseFloat(tokens[5]);
                    scores.add(new GisticScore(chr, start, end, qValue, gScore,
                            type));

                } catch (Exception ex) {
                    log.error("Skipping line: " + nextLine, ex);
                }

                nextLine = reader.readLine();
            }

            if (scores.isEmpty()) {
                return null;
            } else {
                track.setScores(scores);
                computeWholeGenome(IGVModel.getInstance().getViewContext().getGenomeId(), track);
                return track;
            }

        } catch (FileNotFoundException e) {
            log.error("File not found: " + locator.getPath());
            throw new RuntimeException(e);
        } catch (IOException e) {
            log.error("Exception loading: " + locator.getPath(), e);
            throw new RuntimeException(e);
        } finally {
            if (reader != null) {
                reader.close();
            }
        }

    }

    /**
     * Compute features for "chr All".  This is probably not the best way to do this,
     * definitely not good for large tracks.  Need it now for mutation data.
     *
     * @param genomeId
     * @param track
     */
    public static void computeWholeGenome(String genomeId, GisticTrack track) {
        int unit = 1000;    // KB

        String chrAll = "All";
        List<GisticScore> allFeatures = new ArrayList(1000);
        Genome genome = GenomeManager.getInstance().getGenome(genomeId);
        if (genome == null) {
            throw new RuntimeException("Unknown genome: " + genomeId);
        }
        genome.getChromosomeNames();
        long offset = 0;
        for (String chr : genome.getChromosomeNames()) {
            int chrLength = genome.getChromosome(chr).getLength();
            int chrStart = (int) (offset / unit);
            allFeatures.add(new GisticScore(chrAll, chrStart, chrStart, 0, 0, GisticScore.Type.AMP));
            allFeatures.add(new GisticScore(chrAll, chrStart, chrStart, 0, 0, GisticScore.Type.DEL));
            List<GisticScore> ampScores = track.getAmpScores(chr);
            if (ampScores != null) {
                for (GisticScore m : ampScores) {
                    int start = (int) ((offset + m.getStart()) / unit);
                    int end = (int) ((offset + m.getEnd()) / unit);

                    allFeatures.add(new GisticScore(chrAll, start, end,
                            (float) m.getQValue(),
                            (float) m.getGScore(),
                            m.getType()));
                }
            }
            List<GisticScore> delScores = track.getDelScores(chr);
            if (delScores != null) {
                for (GisticScore m : delScores) {
                    int start = (int) ((offset + m.getStart()) / unit);
                    int end = (int) ((offset + m.getEnd()) / unit);

                    allFeatures.add(new GisticScore(chrAll, start, end,
                            (float) m.getQValue(),
                            (float) m.getGScore(),
                            m.getType()));
                }
            }

            offset += chrLength;
        }
        track.setScores(allFeatures);

    }

    static GisticScore.Type getType(String typeString) {
        if (typeString.toUpperCase().equals("AMP")) {
            return GisticScore.Type.AMP;
        } else if (typeString.toUpperCase().equals("DEL")) {
            return GisticScore.Type.DEL;
        } else {
            throw new IllegalArgumentException("Unkown score type: " + typeString);
        }
    }
}
