/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * GenomeManager.java
 *
 * Created on November 9, 2007, 9:12 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.ui.util.ProgressMonitor;
import org.broad.igv.util.Utilities;


import static org.broad.igv.IGVConstants.*;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.net.UnknownHostException;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import javax.swing.JOptionPane;
import org.broad.igv.ui.IGVMainFrame;

/**
 *
 * @author jrobinso
 */
public class GenomeManager {

    /** Field description */
    public static final String SEQUENCE_FILE_EXTENSION = ".txt";
    private static Logger log = Logger.getLogger(GenomeManager.class);
    private static GenomeManager theInstance = null;
    private Map<String, GenomeDescriptor> genomeDescriptorMap;
    /** Field description */
    final public static String USER_DEFINED_GENOME_LIST_FILE = "user-defined-genomes.txt";
    /** Field description */
    final public static String GENOME_ID_KEY = "id";
    /** Field description */
    final public static String GENOME_NAME_KEY = "name";
    /** Field description */
    final public static String GENOME_CYTOBAND_FILE_KEY = "cytobandFile";
    /** Field description */
    final public static String GENOME_GENE_FILE_KEY = "geneFile";
    /** Field description */
    final public static String GENOME_SEQUENCE_URL_KEY = "sequenceURL";

    /**
     * A container for specific genome information which can be used to
     * manage loaded genomes.
     */
    public static class GenomeListItem {

        private int version;
        private String displayableName;
        private String location;
        private String id;
        private boolean isUserDefined = false;

        ;

        /**
         * Constructor.
         *
         * @param displayableName The name that can be shown to a user.
         * @param url The url of the genome archive.
         * @param id The id of the genome.
         * @param isUserDefined
         */
        /**
         * Constructs ...
         *
         *
         * @param displayableName
         * @param url
         * @param id
         * @param isUserDefined
         */
        public GenomeListItem(
                String displayableName, String url, String id, int version,
                boolean isUserDefined) {

            this.displayableName = displayableName;
            this.location = url;
            this.id = id;
            this.version = version;
            this.isUserDefined = isUserDefined;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getDisplayableName() {
            return displayableName;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getId() {
            return id;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getLocation() {
            return location;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public boolean isUserDefined() {
            return isUserDefined;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        @Override
        public String toString() {
            return getDisplayableName();
        }

        /**
         * Method description
         *
         *
         * @return
         */
        @Override
        public int hashCode() {

            int hash = 1;
            hash = hash * 31 + ((displayableName == null) ? 0 : displayableName.trim().hashCode());
            hash = hash * 13 + ((id == null) ? 0 : id.trim().hashCode());
            return hash;
        }

        /**
         * Method description
         *
         *
         * @param object
         *
         * @return
         */
        @Override
        public boolean equals(Object object) {

            if (!(object instanceof GenomeListItem)) {
                return false;
            }

            GenomeListItem item = (GenomeListItem) object;

            // First check name field
            if ((getDisplayableName() == null) && (item.getDisplayableName() != null)) {
                return false;
            }
            if ((item.getDisplayableName() == null) && (getDisplayableName() != null)) {
                return false;
            }
            if (!item.getDisplayableName().trim().equals(getDisplayableName().trim())) {
                return false;
            }

            // Now check id field
            if ((getId() == null) && (item.getId() != null)) {
                return false;
            }
            if ((item.getId() == null) && (getId() != null)) {
                return false;
            }
            if (!item.getId().trim().equals(getId().trim())) {
                return false;
            }

            return true;
        }
    }

    /** Creates a new instance of GenomeManager */
    private GenomeManager() {
        genomeDescriptorMap = new HashMap();
    }

    /**
     * Get the shared instance of the GenomeManager.
     *
     * @return GenomeManager
     */
    public static synchronized GenomeManager getInstance() {
        if (theInstance == null) {
            theInstance = new GenomeManager();
        }
        return theInstance;
    }

    /**
     * Return the genome identified by the id (e.g. mm8, hg17, etc).
     *
     * @param id Genome id.
     *
     * @return
     */
    public Genome getGenome(String id) {

        GenomeDescriptor genomeDescriptor = genomeDescriptorMap.get(id);
        if (genomeDescriptor != null) {
            InputStream is = null;
            try {

                InputStream inputStream = genomeDescriptor.getCytoBandFileInputStream();
                if (inputStream == null) {
                    return null;
                }

                BufferedReader reader = null;
                if (genomeDescriptor.isCytoBandFileGZipFormat()) {
                    is = new GZIPInputStream(inputStream);
                    reader = new BufferedReader(new InputStreamReader(is));
                } else {
                    is = new BufferedInputStream(inputStream);
                    reader = new BufferedReader(new InputStreamReader(is));
                }

                Genome tmpGenome = new Genome(id);
                tmpGenome.setChromosomeMap(CytoBandFileParser.loadData(reader));
                return tmpGenome;

            } catch (IOException ex) {
                log.warn("Error loading the genome!", ex);
            } finally {
                try {
                    if (is != null) {
                        is.close();
                    }
                } catch (IOException ex) {
                    log.warn("Error closing zip stream!", ex);
                }
            }
        }
        return null;
    }

    /**
     * Load a set of genome into the application's memory from specific
     * genome archive files.
     * @param zipFile A single genome archives.
     *
     * @return GenomeListItem.
     */
    private GenomeListItem loadGenomesFromLocalFiles(File zipFile) {

        GenomeListItem genomeListItem = null;
        if (zipFile == null) {
            return null;
        }

        if (zipFile.isDirectory()) {
            return null;
        }

        try {
            boolean isUserDefined = false;
            File parentFolder = zipFile.getParentFile();
            if (!parentFolder.equals(GENOME_CACHE_DIRECTORY)) {
                isUserDefined = true;
            }
            genomeListItem = loadGenome(zipFile.toURI().toURL(), isUserDefined);

        } catch (Exception e) {

            // Log any error and keep going
            log.warn("Error loading genome archive: " + zipFile, e);
        }

        return genomeListItem;
    }

    /**
     * Determines if the specified genome id has a genome archive file in
     * the cache.
     *
     * @param genomeId
     *
     * @return true if genome was found in local cache.
     *
     * @throws IOException
     */
    public boolean isGenomeCached(String genomeId) throws IOException {
        boolean isCached = false;

        LinkedHashSet<GenomeListItem> cachedGenomeItemList = getCachedGenomeArchiveList(null);

        if ((cachedGenomeItemList != null) && !cachedGenomeItemList.isEmpty()) {
            for (GenomeListItem item : cachedGenomeItemList) {
                if (item.getId().equalsIgnoreCase(genomeId)) {
                    isCached = true;
                }
            }
        }
        return isCached;
    }

    /**
     * Determine if the specified genome is already loaded and a descriptor
     * has been created for it.
     *
     * @param id Genome id.
     * @return true if genome is loaded.
     */
    public boolean isGenomeLoaded(String id) {
        return (genomeDescriptorMap.get(id) == null) ? false : true;
    }

    /**
     * Checks if genome archive file is in cache.
     * @param genomeFilename Just the filename - not the path.
     *
     * @return true if genome was found in local cache.
     */
    public boolean isGenomeArchiveInCached(String genomeFilename) {
        File genomeArchiveFile = new File(IGVConstants.GENOME_CACHE_DIRECTORY, genomeFilename);
        return genomeArchiveFile.exists();
    }

    /**
     * Locates a genome in the set of known genome archive locations -
     * then loads it. This method is used ONLY for command line loading
     * of genomes.
     *
     * @param genome The genome to load.
     *
     */
    public void findGenomeAndLoad(String genome) {

        try {
            LinkedHashSet<GenomeListItem> genomes = getAllGenomeArchives(null);
            for (GenomeListItem item : genomes) {
                if (item.getId().equalsIgnoreCase(genome)) {
                    String url = item.getLocation();
                    importGenome(url, false, item.isUserDefined(), null);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Loads a genome into the application's memory from a specific
     * genome archive file url.
     * @param zipUrl The genome archive.
     *
     * @return The GenomeListItem which represents the loaded genome.
     */
    private GenomeListItem loadGenome(URL zipUrl, boolean userDefined) throws IOException {

        GenomeListItem genomeListItem = null;
        if (zipUrl == null) {
            return null;
        }

        try {
            // user imported genomes are not versioned
            GenomeDescriptor genomeDescriptor = createGenomeDescriptor(zipUrl, userDefined);
            genomeDescriptorMap.put(genomeDescriptor.getId(), genomeDescriptor);
            genomeListItem = new GenomeListItem(genomeDescriptor.getName(),
                    genomeDescriptor.getSequenceLocation(), genomeDescriptor.getId(),
                    genomeDescriptor.getVersion(),
                    userDefined);
        } catch (IOException ex) {
            log.error("Error loading the genome!", ex);
        }
        return genomeListItem;
    }

    /**
     * Imports a system IGV genome archive into IGV's local genome cache and
     * then request it be loaded. If the genome is user-define it is loaded
     * directly from it's own location.
     *
     * @param item A GenomeListItem.
     * @param ignoreCached If true, existing genome archives will be over
     * written in the cache. If false, the existing cached genome will be used.
     * @param isUserDefined true if a user genome.
     *
     * @see GenomeListItem
     *
     * @throws FileNotFoundException
     */
    public void importGenome(GenomeListItem item, boolean ignoreCached, boolean isUserDefined)
            throws FileNotFoundException {
        importGenome(item, ignoreCached, isUserDefined, null);
    }

    /**
     * Imports a system IGV genome archive into IGV's local genome cache and
     * then request it be loaded. If the genome is user-define it is loaded
     * directly from it's own location.
     *
     * @param item A GenomeListItem.
     * @param ignoreCached If true, existing genome archives will be over
     * written in the cache. If false, the existing cached genome will be used.
     * @param isUserDefined true if a user genome.
     * @param monitor ProgressMonitor
     *
     * @return GenomeListItem.
     * @see GenomeListItem
     *
     * @throws FileNotFoundException
     */
    private void importGenome(GenomeListItem item, boolean ignoreCached, boolean isUserDefined,
            ProgressMonitor monitor)
            throws FileNotFoundException {

        if (isGenomeLoaded(item.getId())) {
            return;
        }
        importGenome(item.getLocation(), ignoreCached, isUserDefined, null);
    }

    /**
     * Imports a system IGV genome archive into IGV's local genome cache and
     * then request it be loaded. If the genome is user-define it is loaded
     * directly from it's own location.
     *
     * @param genomeArchiveFileLocation
     * @param ignoreCached If true, existing genome archives will be over
     * written in the cache. If false, the existing cached genome will be used.
     * @param isUserDefined true if a user genome.
     *
     * @return GenomeListItem.
     * @see GenomeListItem
     *
     * @throws FileNotFoundException
     */
    public GenomeListItem importGenome(String genomeArchiveFileLocation, boolean ignoreCached,
            boolean isUserDefined)
            throws FileNotFoundException {
        return importGenome(genomeArchiveFileLocation, ignoreCached, isUserDefined, null);
    }

    /**
     * Imports a system IGV genome archive into IGV's local genome cache and
     * then request it be loaded. If the genome is user-define it is loaded
     * directly from it's own location.
     *
     * @param genomeArchiveFileLocation
     * @param ignoreCached If true, existing genome archives will be over
     * written in the cache. If false, the existing cached genome will be used.
     * @param isUserDefined true if a user genome.
     * @param monitor ProgressMonitor
     *
     * @return GenomeListItem.
     * @see GenomeListItem
     *
     * @throws FileNotFoundException
     */
    public GenomeListItem importGenome(String genomeArchiveFileLocation, boolean ignoreCached,
            boolean isUserDefined, ProgressMonitor monitor)
            throws FileNotFoundException {
        return importGenome(genomeArchiveFileLocation, ignoreCached, isUserDefined, monitor, false);
    }

    /**
     * Imports a system IGV genome archive into IGV's local genome cache and
     * then request it be loaded. If the genome is user-define it is loaded
     * directly from it's own location.
     *
     * @param genomeArchiveFileLocation
     * @param ignoreCached If true, existing genome archives will be over
     * written in the cache. If false, the existing cached genome will be used.
     * @param isUserDefined true if a user genome.
     * @param monitor ProgressMonitor
     * @param isFromCommandLine true if this method is call outside of the
     * IGV UI (as in a command line process).
     *
     * @return GenomeListItem.
     * @see GenomeListItem
     *
     * @throws FileNotFoundException
     */
    public GenomeListItem importGenome(String genomeArchiveFileLocation,
            boolean ignoreCached,
            boolean isUserDefined,
            ProgressMonitor monitor,
            boolean isFromCommandLine)
            throws FileNotFoundException {

        GenomeListItem genomeListItem = null;

        if (genomeArchiveFileLocation == null) {
            return null;
        }

        if (!genomeArchiveFileLocation.trim().endsWith(GENOME_FILE_EXTENSION)) {
            throw new RuntimeException(
                    "The extension of archive [" + genomeArchiveFileLocation + "] is not an IGV genome archive extension");
        }

        InputStream archiveInputStream = null;
        try {

            File archiveFile = null;

            // If a System genome copy to local cache
            if (!isUserDefined && !isFromCommandLine) {

                if (!GENOME_CACHE_DIRECTORY.exists()) {
                    GENOME_CACHE_DIRECTORY.mkdir();
                }

                if (genomeArchiveFileLocation.toLowerCase().startsWith("http:") || genomeArchiveFileLocation.toLowerCase().startsWith(
                        "file:")) {

                    URL genomeArchiveURL = new URL(genomeArchiveFileLocation);
                    String fileName = Utilities.getFileNameFromURL(
                            URLDecoder.decode(
                            new URL(genomeArchiveFileLocation).getFile(),
                            "UTF-8"));

                    archiveFile = new File(GENOME_CACHE_DIRECTORY, fileName);

                    // Do we need to read from server directly?
                    if (ignoreCached) {
                        archiveInputStream = genomeArchiveURL.openStream();
                    } else {        // Look in cache first
                        if (!archiveFile.exists()) {
                            archiveInputStream = genomeArchiveURL.openStream();
                        } else {    // If not in cache get from server
                            archiveInputStream = new FileInputStream(archiveFile);
                        }
                    }
                } else {            // Not user-define it must be a cached server file
                    archiveFile = new File(genomeArchiveFileLocation);
                    archiveInputStream = new FileInputStream(archiveFile);
                }

                // Load from local file
                if (!ignoreCached && archiveFile.exists()) {
                    if (monitor != null) {
                        monitor.fireProgressChange(25);
                    }

                    genomeListItem = loadGenomesFromLocalFiles(archiveFile);

                    if (monitor != null) {
                        monitor.fireProgressChange(25);
                    }

                    return genomeListItem;
                } else {

                    if (monitor != null) {
                        monitor.fireProgressChange(25);
                    }

                    final int READ_CHUNK_SIZE = 64000;
                    Utilities.createFileFromStream(archiveInputStream, archiveFile,
                            READ_CHUNK_SIZE);


                    if (monitor != null) {
                        monitor.fireProgressChange(25);
                    }
                }
            } else {

                // New genome archive file
                archiveFile = new File(genomeArchiveFileLocation);

                if (!archiveFile.exists()) {
                    throw new FileNotFoundException(genomeArchiveFileLocation);
                }

                if (!isFromCommandLine) {

                    File userDefinedListFile = new File(GENOME_CACHE_DIRECTORY,
                            USER_DEFINED_GENOME_LIST_FILE);
                    Properties listProperties =
                            retrieveUserDefinedGenomeListFromFile(userDefinedListFile);

                    if (listProperties == null) {
                        listProperties = new Properties();
                    }

                    String record = buildClientSideGenomeListRecord(archiveFile, isUserDefined);

                    if (record != null) {
                        // user defined genomes are not versioned
                        int version = 0;
                        String[] fields = record.split("\t");
                        listProperties.setProperty(fields[2], record);
                        GenomeImporter.storeUserDefinedGenomeListToFile(userDefinedListFile,
                                listProperties);
                        genomeListItem = new GenomeListItem(fields[0], fields[1], fields[2], version,
                                isUserDefined);
                    }
                }
            }

            // If archive file exist import it into IGV
            if (archiveFile.exists()) {
                if (monitor != null) {
                    monitor.fireProgressChange(25);
                }

                loadGenomesFromLocalFiles(archiveFile);

                if (monitor != null) {
                    monitor.fireProgressChange(25);
                }
            }
        } catch (MalformedURLException e) {
            log.warn("Error Importing Genome!", e);
        } catch (FileNotFoundException e) {
            throw e;
        } catch (SocketException e) {
            throw new GenomeServerException("Server connection error", e);
        } catch (IOException e) {
            log.warn("Error Importing Genome!", e);
        }
        return genomeListItem;
    }

    /**
     * Creates a genome descriptor.
     *
     * @param name The genome display name.
     * @param id The genome id.
     * @param ctyoBandFileName The cytoband file name (not path).
     * @param geneFileName The gene file name (not path).
     * @param sequenceUrl The full path to the chromosome sequence location.
     * @param zipFile The genome archive zip file.
     * @param zipEntries The entries in the genome archive (stored for quick access).
     * @param userDefined true if a user genome.
     *
     * @return
     * @see GenomeDescriptor
     */
    private GenomeDescriptor createGenomeDescriptor(String name,
            int version,
            String id,
            String ctyoBandFileName,
            String geneFileName,
            String geneTrackName,
            String sequenceUrl,
            ZipFile zipFile,
            Map<String, ZipEntry> zipEntries,
            boolean userDefined) {

        GenomeDescriptor genomeDescriptor = new GenomeDescriptor(name, version, id, ctyoBandFileName,
                geneFileName, geneTrackName, sequenceUrl, zipFile,
                zipEntries, userDefined);
        return genomeDescriptor;
    }

    /**
     * Creates a genome descriptor.
     *
     * @param genomeArchive The genome archive file.
     * @param userDefined true if genome is user-defined.
     *
     * @return GenomeDescriptor
     * @see GenomeDescriptor
     * @throws java.io.IOException
     */
    private GenomeDescriptor createGenomeDescriptor(File genomeArchive, boolean userDefined)
            throws IOException {

        if (genomeArchive == null) {
            return null;
        }
        try {
            URL zipUrl = genomeArchive.toURI().toURL();
            return createGenomeDescriptor(zipUrl, userDefined);
        } catch (MalformedURLException e) {
            log.error("Could not create a genome descriptor from archive file!", e);
        }
        return null;
    }

    /**
     * Creates a genome descriptor.
     *
     * @param zipUrl The URL of the genome archive file.
     * @param userDefined true if genome is user-defined.
     *
     * @return GenomeDescriptor
     * @see GenomeDescriptor
     * @throws java.io.IOException
     */
    private GenomeDescriptor createGenomeDescriptor(URL zipUrl, boolean userDefined)
            throws IOException {

        if (zipUrl == null) {
            return null;
        }

        String zipFilePath = URLDecoder.decode(zipUrl.getFile(), "UTF-8");
        if ((zipFilePath == null) || !zipFilePath.endsWith(GENOME_FILE_EXTENSION)) {
            return null;
        }

        GenomeDescriptor genomeDescriptor = null;
        Map<String, ZipEntry> zipEntries = new HashMap();
        ZipFile zipFile = new ZipFile(zipFilePath);
        ZipInputStream zipInputStream = null;
        try {
            zipInputStream = new ZipInputStream(zipUrl.openStream());
            ZipEntry zipEntry = zipInputStream.getNextEntry();

            while (zipEntry != null) {
                String zipEntryName = zipEntry.getName();
                zipEntries.put(zipEntryName, zipEntry);

                if (zipEntryName.equalsIgnoreCase(GENOME_ARCHIVE_PROPERTY_FILE_NAME)) {
                    InputStream inputStream = zipFile.getInputStream(zipEntry);
                    Properties properties = new Properties();
                    properties.load(inputStream);

                    // Cytoband
                    String cytobandZipEntryName =
                            properties.getProperty(GENOME_ARCHIVE_CYTOBAND_FILE_KEY);

                    // RefFlat
                    String geneFileName =
                            properties.getProperty(GENOME_ARCHIVE_GENE_FILE_KEY);

                    String sequenceLocation =
                            properties.getProperty(GENOME_ARCHIVE_SEQUENCE_FILE_LOCATION_KEY);

                    if ((sequenceLocation != null) && !sequenceLocation.startsWith("http:")) {
                        File tempZipFile = new File(zipFilePath);
                        File sequenceFolder = new File(tempZipFile.getParent(), sequenceLocation);
                        sequenceLocation = sequenceFolder.getCanonicalPath();
                        sequenceLocation.replace('\\', '/');
                    }


                    int version = 0;
                    String versionString = properties.getProperty(GENOME_ARCHIVE_VERSION_KEY);
                    if (versionString != null) {
                        try {
                            version = Integer.parseInt(versionString);
                        } catch (Exception e) {
                            log.error("Error parsing version string: " + versionString);
                        }
                    }

                    // The new descriptor
                    genomeDescriptor =
                            createGenomeDescriptor(
                            properties.getProperty(GENOME_ARCHIVE_NAME_KEY),
                            version,
                            properties.getProperty(GENOME_ARCHIVE_ID_KEY),
                            cytobandZipEntryName,
                            geneFileName,
                            properties.getProperty(IGVConstants.GENOME_GENETRACK_NAME, "Gene"),
                            sequenceLocation,
                            zipFile,
                            zipEntries,
                            userDefined);
                }
                zipEntry = zipInputStream.getNextEntry();
            }
        } finally {
            try {
                if (zipInputStream != null) {
                    zipInputStream.close();
                }
            } catch (IOException ex) {
                log.warn("Error closing imported genome zip stream!", ex);
            }
        }
        return genomeDescriptor;
    }

    /**
     * Gets a map of all genome descriptor stored in IGV.
     *
     * @return A map of GenomeDescriptor objects keyed by genome id.
     */
    public Map<String, GenomeDescriptor> getGenomeDescriptorMap() {
        return genomeDescriptorMap;
    }

    /**
     * Gets the descriptor for a specific genome.
     *
     *
     * @param id
     *
     * @return GenomeDescriptor
     */
    public GenomeDescriptor getGenomeDescriptor(String id) {
        return genomeDescriptorMap.get(id);
    }

    /**
     * Sort chromosome names
     * @param descriptors
     */
    public void sort(List<GenomeDescriptor> descriptors) {

        Comparator comparator = new Comparator() {

            public int compare(Object o1, Object o2) {
                GenomeDescriptor g1 = (GenomeDescriptor) o1;
                GenomeDescriptor g2 = (GenomeDescriptor) o2;
                return (int) g1.getName().compareTo(g2.getName());
            }
        };
        Collections.sort(descriptors, comparator);
    }

    /**
     * Gets a list of all the server genome archive files that
     * IGV knows about.
     *
     * @param excludedArchivesUrls The set of file location to exclude in the
     * return list.
     *
     * @return LinkedHashSet<GenomeListItem>
     * @see GenomeListItem
     *
     * @throws IOException
     */
    public LinkedHashSet<GenomeListItem> getServerGenomeArchiveList(Set excludedArchivesUrls)
            throws IOException {

        LinkedHashSet<GenomeListItem> genomeItemList = new LinkedHashSet();
        BufferedReader dataReader = null;
        InputStream inputStream = null;
        try {

            URL serverGenomeArchiveList = new URL(PreferenceManager.getInstance().getGenomeListURL());
            final URLConnection connection = serverGenomeArchiveList.openConnection();

            connection.setReadTimeout(5000);
            inputStream = connection.getInputStream();
            dataReader = new BufferedReader(new InputStreamReader(inputStream));

            boolean wasHeaderRead = false;
            String genomeRecord = null;
            while ((genomeRecord = dataReader.readLine()) != null) {

                // Check for valid data header
                if (!wasHeaderRead) {
                    wasHeaderRead = true;
                    if ((genomeRecord == null) || !genomeRecord.trim().equalsIgnoreCase(
                            SERVER_GENOME_LIST_HEADER)) {
                        String message = "The genome list returned " + "from the server had an invalid header record!";
                        log.error(message);
                        throw new GenomeException(message);
                    }
                    continue;
                }

                if (genomeRecord != null) {
                    genomeRecord = genomeRecord.trim();

                    String[] fields = genomeRecord.split("\t");

                    if ((fields != null) && (fields.length >= 3)) {

                        // Throw away records we don't want to see
                        if (excludedArchivesUrls != null) {
                            if (excludedArchivesUrls.contains(fields[1])) {
                                continue;
                            }
                        }
                        int version = 0;
                        if (fields.length > 3) {
                            try {
                                version = Integer.parseInt(fields[3]);
                            } catch (Exception e) {
                                log.error("Error parsing genome version: " + fields[0], e);
                            }
                        }

                        try {
                            GenomeListItem item = new GenomeListItem(fields[0], fields[1],
                                    fields[2], version, false);
                            genomeItemList.add(item);
                        } catch (Exception e) {
                            log.error(
                                    "Error reading a line from server genome list" + " line was: [" + genomeRecord + "]",
                                    e);
                            continue;
                        }
                    } else {
                        log.error("Found invalid server genome list record: " + genomeRecord);
                    }
                }
            }
        } catch (Exception e) {
            log.error("Error fetching genome list: ", e);
            JOptionPane.showMessageDialog(
                    IGVMainFrame.getInstance(),
                    IGVConstants.CANNOT_ACCESS_SERVER_GENOME_LIST);
        } finally {
            if (dataReader != null) {
                dataReader.close();
            }
            if (inputStream != null) {
                inputStream.close();
            }
        }
        return genomeItemList;
    }

    /**
     * Gets a list of all the user-defined genome archive files that
     * IGV knows about.
     *
     * @param excludedArchivesUrls The set of file location to exclude in the
     * return list.
     *
     * @return LinkedHashSet<GenomeListItem>
     * @see GenomeListItem
     *
     * @throws IOException
     */
    public LinkedHashSet<GenomeListItem> getUserDefinedGenomeArchiveList(Set excludedArchivesUrls)
            throws IOException {

        boolean clientGenomeListNeedsRebuilding = false;
        LinkedHashSet<GenomeListItem> genomeItemList = new LinkedHashSet();

        File listFile = new File(GENOME_CACHE_DIRECTORY, USER_DEFINED_GENOME_LIST_FILE);

        Properties listProperties = retrieveUserDefinedGenomeListFromFile(listFile);

        if (listProperties != null) {

            Collection records = listProperties.values();

            for (Object value : records) {

                String record = (String) value;
                if (record.trim().equals("")) {
                    continue;
                }

                String[] fields = record.split("\t");

                // Throw away records we don't want to see
                if (excludedArchivesUrls != null) {
                    if (excludedArchivesUrls.contains(fields[1])) {
                        continue;
                    }
                }

                File file = new File(fields[1]);
                if (file.isDirectory()) {
                    continue;
                }
                if (!file.exists()) {
                    clientGenomeListNeedsRebuilding = true;
                    continue;
                }

                if (!file.getName().toLowerCase().endsWith(GENOME_FILE_EXTENSION)) {
                    continue;
                }
                GenomeListItem item = new GenomeListItem(fields[0], file.getAbsolutePath(),
                        fields[2], 0, true);
                genomeItemList.add(item);
            }
        }
        if (clientGenomeListNeedsRebuilding) {
            rebuildClientGenomeList(genomeItemList);
        }
        return genomeItemList;
    }

    /**
     * Method description
     *
     */
    public void clearGenomeCache() {

        File[] files = GENOME_CACHE_DIRECTORY.listFiles();
        for (File file : files) {

            if (file.getName().toLowerCase().endsWith(GENOME_FILE_EXTENSION)) {
                file.delete();
            }
        }

    }

    /**
     * Gets a list of all the locally cached genome archive files that
     * IGV knows about.
     *
     * @param excludedArchivesUrls The set of file location to exclude in the
     * return list.
     *
     * @return LinkedHashSet<GenomeListItem>
     * @see GenomeListItem
     *
     * @throws IOException
     */
    public LinkedHashSet<GenomeListItem> getCachedGenomeArchiveList(Set excludedArchivesUrls)
            throws IOException {

        LinkedHashSet<GenomeListItem> genomeItemList = new LinkedHashSet();

        if (!GENOME_CACHE_DIRECTORY.exists()) {
            return genomeItemList;
        }

        File[] files = GENOME_CACHE_DIRECTORY.listFiles();
        for (File file : files) {

            if (file.isDirectory()) {
                continue;
            }

            if (!file.getName().toLowerCase().endsWith(GENOME_FILE_EXTENSION)) {
                continue;
            }

            URL zipUrl = file.toURI().toURL();

            // Throw away records we don't want to see
            if (excludedArchivesUrls != null) {
                if (excludedArchivesUrls.contains(URLDecoder.decode(zipUrl.getFile(), "UTF-8"))) {
                    continue;
                }
            }

            ZipInputStream zipInputStream = null;
            try {

                ZipFile zipFile = new ZipFile(file);
                zipInputStream =
                        new ZipInputStream(new BufferedInputStream(new FileInputStream(file)));

                ZipEntry zipEntry = zipFile.getEntry(GENOME_ARCHIVE_PROPERTY_FILE_NAME);
                if (zipEntry == null) {
                    continue;    // Should never happen
                }

                InputStream inputStream = zipFile.getInputStream(zipEntry);
                Properties properties = new Properties();
                properties.load(inputStream);

                int version = 0;
                if (properties.containsKey(GENOME_ARCHIVE_VERSION_KEY)) {
                    try {
                        version = Integer.parseInt(
                                properties.getProperty(GENOME_ARCHIVE_VERSION_KEY));
                    } catch (Exception e) {
                        log.error("Error parsing genome version: " + version, e);
                    }
                }

                GenomeListItem item =
                        new GenomeListItem(properties.getProperty(GENOME_ARCHIVE_NAME_KEY),
                        file.getAbsolutePath(),
                        properties.getProperty(GENOME_ARCHIVE_ID_KEY),
                        version,
                        false);
                genomeItemList.add(item);
            } catch (ZipException ex) {
                log.warn("\nError building cached genome list!", ex);
            } catch (IOException ex) {
                log.warn("\nError building cached genome list!", ex);
            } finally {
                try {
                    if (zipInputStream != null) {
                        zipInputStream.close();
                    }
                } catch (IOException ex) {
                    log.warn("Error closing genome zip stream!", ex);
                }
            }
        }

        return genomeItemList;
    }

    /**
     * Gets a list of all the server and client-side genome archive files that
     * IGV knows about.
     *
     * @param excludedArchivesUrls The set of file location to exclude in the
     * return list.
     *
     * @return LinkedHashSet<GenomeListItem>
     * @see GenomeListItem
     *
     * @throws IOException
     */
    public LinkedHashSet<GenomeListItem> getAllGenomeArchives(Set excludedArchivesUrls)
            throws IOException {

        LinkedHashSet<GenomeListItem> genomes = new LinkedHashSet();

        // Build a single available genome list from both client, server
        // and cached information. This allows us to process
        // everything the same way.
        LinkedHashSet<GenomeListItem> serverSideItemList = null;
        try {
            serverSideItemList = GenomeManager.getInstance().getServerGenomeArchiveList(null);
        } catch (UnknownHostException e) {
            log.error(IGVConstants.CANNOT_ACCESS_SERVER_GENOME_LIST, e);
        } catch (SocketException e) {
            log.error(IGVConstants.CANNOT_ACCESS_SERVER_GENOME_LIST, e);
        }

        LinkedHashSet<GenomeListItem> cacheGenomeItemList =
                GenomeManager.getInstance().getCachedGenomeArchiveList(excludedArchivesUrls);

        LinkedHashSet<GenomeListItem> userDefinedItemList =
                GenomeManager.getInstance().getUserDefinedGenomeArchiveList(excludedArchivesUrls);


        if (userDefinedItemList != null) {
            genomes.addAll(userDefinedItemList);
        }
        if (cacheGenomeItemList != null) {
            genomes.addAll(cacheGenomeItemList);
        }
        if (serverSideItemList != null) {
            genomes.addAll(serverSideItemList);
        }
        return genomes;
    }

    /**
     * Reconstructs the user-define genome property file.
     *
     * @param genomeItemList The list of user-define genome GenomeListItem
     * objects to store in the property file.
     *
     * @throws IOException
     */
    public void rebuildClientGenomeList(LinkedHashSet<GenomeListItem> genomeItemList)
            throws IOException {

        if ((genomeItemList == null)) {
            return;
        }

        File listFile = new File(GENOME_CACHE_DIRECTORY, USER_DEFINED_GENOME_LIST_FILE);

        if (!listFile.exists()) {
            listFile.createNewFile();
        }

        StringBuffer buffer = new StringBuffer();
        Properties listProperties = new Properties();
        for (GenomeListItem genomeListItem : genomeItemList) {

            buffer.append(genomeListItem.getDisplayableName());
            buffer.append("\t");
            buffer.append(genomeListItem.getLocation());
            buffer.append("\t");
            buffer.append(genomeListItem.getId());

            listProperties.setProperty(genomeListItem.getId(), buffer.toString());
            buffer.delete(0, buffer.length());
        }
        GenomeImporter.storeUserDefinedGenomeListToFile(listFile, listProperties);
    }

    /**
     * Create a genonem list record (same format as used by the server) for
     * genome files.
     *
     * @param genomeArchive The genome file from which to extract a record.
     * @param userDefined true if archive is a user-defined genome archive.
     *
     * @return A tab delimetered genome list record containing
     * ( name[tab]genome location[tab]genomeId ).
     *
     * @throws IOException
     */
    public String buildClientSideGenomeListRecord(File genomeArchive, boolean userDefined)
            throws IOException {

        GenomeDescriptor genomeDescriptor = createGenomeDescriptor(genomeArchive, userDefined);

        StringBuffer buffer = new StringBuffer();
        buffer.append(genomeDescriptor.getName());
        buffer.append("\t");
        buffer.append(genomeArchive.getAbsoluteFile());
        buffer.append("\t");
        buffer.append(genomeDescriptor.getId());

        return buffer.toString();
    }

    /**
     * Read the user-defined genome property file to find enough information to
     * display the genome in IGV.
     *
     * @param file A java properties file containing tab delimetered data
     * (display name [tab] genome file location [tab] genome id) about
     * the user-defined genome.
     *
     * @return A java Properties object contain the file's content.
     */
    public static Properties retrieveUserDefinedGenomeListFromFile(File file) {

        Properties properties = new Properties();

        if ((file != null) && file.exists()) {
            FileInputStream input = null;
            try {
                input = new FileInputStream(file);
                properties.load(input);
            } catch (FileNotFoundException e) {
                log.error("Property file for user-defined genomes was not " + "found!", e);
            } catch (IOException e) {
                log.error("Error readin property file for user-defined " + "genomes!", e);
            } finally {
                if (input != null) {
                    try {
                        input.close();
                    } catch (IOException e) {
                        log.error("Error closing property file for " + "user-defined genomes!",
                                e);
                    }
                }
            }
        }
        return properties;
    }

    /**
     * Create an IGV representation of a user-defined genome.
     *
     * @param genomeZipLocation A File path to a directory in which the .genome
     * output file will be written.
     * @param cytobandFileName A File path to a file that contains cytoband data.
     * @param refFlatFileName A File path to a gene file.
     * @param fastaFileName A File path to a FASTA file, a .gz file containing a
     * single FASTA file, or a directory containing ONLY FASTA files.
     * @param relativeSequenceLocation A relative path to the location
     * (relative to the .genome file to be created) where the sequence data for
     * the new genome will be written.
     * @param genomeDisplayName The unique user-readable name of the new genome.
     * @param genomeId The id to be assigned to the genome.
     * @param genomeFileName The file name (not path) of the .genome archive
     * file to be created.
     * @param monitor A ProgressMonitor used to track progress - null,
     * if no progress updating is required.
     * @param sequenceOutputLocationOverride
     *
     * @return GenomeListItem
     *
     * @throws FileNotFoundException
     */
    public GenomeListItem defineGenome(String genomeZipLocation, String cytobandFileName,
            String refFlatFileName, String fastaFileName,
            String relativeSequenceLocation, String genomeDisplayName,
            String genomeId, String genomeFileName,
            ProgressMonitor monitor,
            String sequenceOutputLocationOverride)
            throws FileNotFoundException {

        File archiveFile = null;
        File zipFileLocation = null;
        File fastaInputFile = null;
        File refFlatFile = null;
        File cytobandFile = null;
        File sequenceLocation = null;

        if ((genomeZipLocation != null) && (genomeZipLocation.trim().length() != 0)) {
            zipFileLocation = new File(genomeZipLocation);
            PreferenceManager.getInstance().setLastGenomeImportDirectory(zipFileLocation);
        }

        if ((cytobandFileName != null) && (cytobandFileName.trim().length() != 0)) {
            cytobandFile = new File(cytobandFileName);
            PreferenceManager.getInstance().setLastCytobandDirectory(cytobandFile.getParentFile());
        }

        if ((refFlatFileName != null) && (refFlatFileName.trim().length() != 0)) {
            refFlatFile = new File(refFlatFileName);
            PreferenceManager.getInstance().setLastRefFlatDirectory(refFlatFile.getParentFile());
        }

        if ((fastaFileName != null) && (fastaFileName.trim().length() != 0)) {
            fastaInputFile = new File(fastaFileName);

            PreferenceManager.getInstance().setLastFastaDirectory(fastaInputFile.getParentFile());

            // The sequence info only matters if we have FASTA
            if ((relativeSequenceLocation != null) && (relativeSequenceLocation.trim().length() != 0)) {
                sequenceLocation = new File(genomeZipLocation, relativeSequenceLocation);
                if (!sequenceLocation.exists()) {
                    sequenceLocation.mkdir();
                }
                PreferenceManager.getInstance().setLastSequenceDirectory(sequenceLocation);
            }
        }

        if (monitor != null) {
            monitor.fireProgressChange(25);
        }

        archiveFile = (new GenomeImporter()).createGenomeArchive(zipFileLocation,
                genomeFileName, genomeId, genomeDisplayName, relativeSequenceLocation,
                fastaInputFile, refFlatFile, cytobandFile, sequenceOutputLocationOverride, monitor);

        if (monitor != null) {
            monitor.fireProgressChange(75);
        }

        GenomeListItem genomeListItem =
                GenomeManager.getInstance().importGenome(archiveFile.getAbsolutePath(), false, true);

        return genomeListItem;
    }

    /**
     * This method takes any string and generates a unique key based on
     * that string. Other class typically call this method to make a genome id
     * from some string but it can be used to generate a key for anything.
     * TODO This method probably belongs in a Utilities class.
     *
     *
     * @param text
     *
     * @return A unique key based on the text.
     */
    public String generateGenomeKeyFromText(String text) {

        if (text == null) {
            return null;
        }

        text = text.trim();
        text = text.replace(" ", "_");
        text = text.replace("(", "");
        text = text.replace(")", "");
        text = text.replace("[", "");
        text = text.replace("]", "");
        text = text.replace("{", "");
        text = text.replace("}", "");
        return text;
    }

    /*
     *  REMOVED METHODS
     *
     * public void removeAllUserDefinedGenomes() {
     *
     *   File listFile = new File(GENOME_CACHE_DIRECTORY, USER_DEFINED_GENOME_LIST_FILE);
     *
     *   if (!listFile.exists())
     *   {
     *       return;
     *   }
     *   listFile.delete();
     * }
     *
     * public File createFileObject(File location, String genomeId) {
     *   return new File(location, (genomeId + "_Genome" + GENOME_FILE_EXTENSION));
     * }
     *
     * public void buildGenome(File cytobandFile, File refFlatFile, File fastaFile, Properties genomeProperties) throws Exception {
     *
     *   String sequenceUrl = genomeProperties.getProperty(GENOME_SEQUENCE_URL_KEY);
     *   String id = genomeProperties.getProperty(GENOME_ID_KEY);
     *
     *   if ((fastaFile != null) && (sequenceUrl != null))
     *   {
     *       throw new RuntimeException("Both FASTA file and sequenceURL " + "cannot be set!");
     *   }
     *
     *   if (fastaFile != null) {}
     *   else if (cytobandFile != null)
     *   {
     *
     *       if (refFlatFile != null)
     *       {
     *           genomeProperties.setProperty(GENOME_SEQUENCE_URL_KEY, null);
     *       }
     *   }
     * }
     *
     */
}
