/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/



/*
* Genome.java
*
* Created on November 9, 2007, 9:05 AM
*
* To change this template, choose Tools | Template Manager
* and open the template in the editor.
 */
package org.broad.igv.feature;

//~--- JDK imports ------------------------------------------------------------

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;

/**
 * Simple model of a genome, essesntially a collection of cytobands.
 */
public class Genome {

    private String name;
    private LinkedHashMap<String, Chromosome> chromosomeMap;
    List<String> chromosomeNames;
    long length = -1;

    /**
     * Creates a new instance of Genome
     *
     * @param name
     */
    public Genome(String name) {
        this.name = name;
        setChromosomeMap(new LinkedHashMap(24));
    }

    /**
     * Method description
     *
     *
     * @param chromosomeMap
     */
    public void setChromosomeMap(LinkedHashMap<String, Chromosome> chromosomeMap) {
        this.chromosomeMap = chromosomeMap;
    }

    /**
     * Method description
     *
     *
     * @param chrName
     *
     * @return
     */
    public Chromosome getChromosome(String chrName) {
        return chromosomeMap.get(chrName);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public List<String> getChromosomeNames() {
        if (chromosomeNames == null)
        {
            chromosomeNames = new LinkedList<String>(chromosomeMap.keySet());
            Collections.sort(chromosomeNames, new ChromosomeComparator());

        }
        return chromosomeNames;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Collection<Chromosome> getChromosomes() {
        return chromosomeMap.values();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public long getLength() {
        if (length < 0)
        {
            length = 0;
            for (Chromosome chr : chromosomeMap.values())
            {
                length += chr.getLength();
            }
        }
        return length;
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public long getCumulativeOffset(String chr) {
        long offset = 0;
        for (String c : getChromosomeNames())
        {
            if (chr.equals(c))
            {
                break;
            }
            offset += getChromosome(c).getLength();
        }
        return offset;
    }

    /**
     * Return the chromosome coordinate in BP in genome coordinates in KBP
     *
     * @param chr
     * @param locationBP
     *
     * @return
     */
    public int getGenomeCoordinate(String chr, int locationBP) {
        return (int) ((getCumulativeOffset(chr) + locationBP) / 1000);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getName() {
        return name;
    }


    /**
     * Comparator for chromosome names.  
     */
    static class ChromosomeComparator implements Comparator<String> {

        /**
         *
         *
         * @param chr1
         * @param chr2
         *
         * @return
         */
        public int compare(String chr1, String chr2) {
            try
            {
                if(chr1.equals("chrM") || chr1.equals("M")) {
                    return 1;
                }
                else if(chr2.equals("chrM") || chr2.equals("M")) {
                    return -1;
                }
                ChrParts p1 = new ChrParts(chr1);
                ChrParts p2 = new ChrParts(chr2);
                return p1.compareTo(p2);
            }
            catch (NumberFormatException numberFormatException)
            {
                return 0;
            }

        }



        class ChrParts implements Comparable<ChrParts> {
            String prefix;
            int number;

            ChrParts(String str) {
                String chr = str.toUpperCase().replace("chr", "");
                if (endsWithNumber(chr))
                {
                    char[] charArray = chr.toCharArray();
                    int idx = charArray.length - 1;
                    while (idx >= 0)
                    {
                        if (!Character.isDigit(charArray[idx]))
                        {
                            break;
                        }
                        idx--;
                    }
                    idx++;
                    prefix = chr.substring(0, idx);
                    number = Integer.parseInt(chr.substring(idx));

                }
                else
                {
                    prefix = chr;
                    number = 0;
                }
            }

            private boolean endsWithNumber(String string) {
                char lastChar = string.toCharArray()[string.length() - 1];
                return Character.isDigit(lastChar);
            }

            /**
             * Note: chrM (mitochondria) always sorts last
             * @param arg0
             *
             * @return
             */
            public int compareTo(ChromosomeComparator.ChrParts arg0) {
                int rtnValue = this.prefix.compareTo(arg0.prefix);
                if (rtnValue != 0)
                {
                    return rtnValue;
                }
                else
                {
                    return this.number - arg0.number;
                }
            }

        }

    }
}
