/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * GeneManager.java
 *
 * Created on June 12, 2007, 5:22 PM
 *
 */
package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedInputStream;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import java.io.InputStream;

import java.util.*;
import java.util.zip.GZIPInputStream;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public class GeneManager {

    private static Logger logger = Logger.getLogger(GeneManager.class);
    /**
     * The genome for this gene manager.  This should probably be refactored
     * so that the genome owns the gene manger, or manages the genes itself.
     */
    private Genome genome;
    Map<String, List<Feature>> chromosomeGeneMap;
    String geneTrackName;
    Map<String, Feature> geneMap;
    /** Map of chromosome -> lsst gene */
    Map<String, Integer> maxLocationMap;
    /** Map of chromosome -> longest gene */
    Map<String, Feature> longestGeneMap;
    /** Set of loaded gene variants.  Used to prevent loading of duplicates. */
    Set<String> loadedGeneKeys;

    public GeneManager(String genomeId) {
        this(genomeId, "Gene");
    }

    /**
     * Creates a new instance of GeneManager
     *
     * @param genomeId
     */
    public GeneManager(String genomeId, String geneTrackName) {
        genome = GenomeManager.getInstance().getGenome(genomeId);
        if (genome == null)
        {
            throw new RuntimeException("Unknown genome: " + genomeId);
        }
        chromosomeGeneMap = new HashMap<String, List<Feature>>();
        geneMap = new HashMap<String, Feature>();
        maxLocationMap = new HashMap<String, Integer>();
        longestGeneMap = new HashMap<String, Feature>();
        loadedGeneKeys = new HashSet();
        this.geneTrackName = geneTrackName;
    }

    public String getGeneTrackName() {
        return geneTrackName;
    }

    /**
     * Add a gene.  The gene is mapped by name and chromosome.
     *
     * @param gene
     */
    public void addGene(Feature gene) {

        String key = gene.getIdentifier();

        if (!loadedGeneKeys.contains(key))
        {
            loadedGeneKeys.add(key);

            // If there is a genome associated with this manager only include
            // genes whose chromosome is contained in the genome.

            if (genome != null)
            {
                if (genome.getChromosome(gene.getChromosome()) == null)
                {
                    return;
                }
            }

            // If there are multiple variant of a gene, use the longest
            Feature currentGene = geneMap.get(gene.getName());
            if (gene.getIdentifier() != null)
            {
                geneMap.put(gene.getIdentifier().trim().toUpperCase(), gene);
            }
            if (currentGene == null)
            {
                if (gene.getName() != null)
                {
                    geneMap.put(gene.getName().trim().toUpperCase(), gene);
                }
            } else
            {
                int w1 = currentGene.getEnd() - currentGene.getStart();
                int w2 = gene.getEnd() - gene.getStart();
                if (w2 > w1)
                {
                    if (gene.getName() != null)
                    {
                        geneMap.put(gene.getName().trim().toUpperCase(), gene);
                    }
                }
            }



            // Update "longest gene" for this chromosome
            String chr = gene.getChromosome();
            List<Feature> geneDataList = chromosomeGeneMap.get(chr);
            if (geneDataList == null)
            {
                geneDataList = new ArrayList<Feature>();
                chromosomeGeneMap.put(chr, geneDataList);
                maxLocationMap.put(chr, (int) gene.getEnd());
            }
            if (gene.getEnd() > maxLocationMap.get(chr))
            {
                maxLocationMap.put(chr, (int) gene.getEnd());
            }

            if (longestGeneMap.get(chr) == null)
            {
                longestGeneMap.put(chr, gene);
            } else
            {
                if (gene.getLength() > longestGeneMap.get(chr).getLength())
                {
                    longestGeneMap.put(chr, gene);
                }
            }

            geneDataList.add((Feature) gene);
        }

    }

    /**
     * Method description
     *
     *
     * @param geneName
     *
     * @return
     */
    public Feature getGene(String geneName) {
        return geneMap.get(geneName.toUpperCase());
    }

    /**
     * Return the longest gene length for the given chromosome.  Feature length is the
     * total length, including introns.  If there genes have not been loaded guess 1 MB.
     *
     * @param chr
     *
     * @return
     */
    public int getLongestGeneLength(String chr) {
        Feature longestGene = longestGeneMap.get(chr);
        return ((longestGene == null) ? 1000000 : longestGene.getLength());
    }

    /**
     *
     * @return
     */
    public Map<String, List<Feature>> getChromsomeGeneMap() {
        return chromosomeGeneMap;
    }

    /**
     * Return the list of genes for the specified chromosome.
     *
     * @param chromosome
     *
     * @return
     */
    public List<Feature> getGenesForChromosome(String chromosome) {
        List<Feature> genes = chromosomeGeneMap.get(chromosome);
        if (genes == null)
        {
            logger.info("No genes found for chromosome: " + chromosome);
        }
        return genes;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Collection<String> getChromosomes() {
        return chromosomeGeneMap.keySet();
    }

    /**
     * Return the list of genes that overlap the  specified region.
     *
     * @param chromosome
     * @param start
     * @param end
     *
     * @return
     */
    public List<Feature> getGenesForRegion(String chromosome, int start, int end) {
        List<Feature> geneList = new ArrayList<Feature>();
        for (Feature gene : getGenesForChromosome(chromosome))
        {
            if ((gene.getEnd() >= start) && (gene.getStart() <= end))
            {
                geneList.add(gene);
            }
        }
        return geneList;
    }

    /**
     * Sort genes by position
     */
    public void sortGeneLists() {
        Comparator c = new Comparator() {

            public int compare(Object o1, Object o2) {
                Feature g1 = (Feature) o1;
                Feature g2 = (Feature) o2;
                return (int) (g1.getStart() - g2.getStart());
            }
        };

        for (List<Feature> geneList : chromosomeGeneMap.values())
        {
            Collections.sort(geneList, c);
        }
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public int getMaximumLocation(String chr) {
        return maxLocationMap.get(chr);
    }
    static Map<String, GeneManager> geneManagerCache = new HashMap();

    /**
     * Method description
     *
     *
     * @param genomeId
     *
     * @return
     */
    public static synchronized GeneManager getGeneManager(String genomeId) {

        GeneManager geneManager = geneManagerCache.get(genomeId);
        if (geneManager == null)
        {
            GenomeDescriptor genomeDescriptor =
                GenomeManager.getInstance().getGenomeDescriptor(genomeId);
            if (genomeDescriptor != null)
            {
                AsciiLineReader reader = getGeneReader(genomeDescriptor);
                if (reader != null)
                {
                    try
                    {
                        geneManager = new GeneManager(genomeId, genomeDescriptor.getGeneTrackName());
                        String geneFilename = genomeDescriptor.getGeneFileName();
                        FeatureParser parser = AbstractFeatureParser.getInstanceFor(geneFilename);
                        List<Feature> genes = parser.loadFeatures(reader);
                        for (Feature gene : genes)
                        {
                            geneManager.addGene(gene);
                            FeatureDB.addFeature(gene);
                        }

                        geneManager.sortGeneLists();

                        geneManagerCache.put(genomeId, geneManager);
                    } finally
                    {

                        if (reader != null)
                        {
                            reader.close();
                        }

                    }
                }
            }
        }

        return geneManager;
    }

    /**
     * Method description
     *
     *
     *
     * @param genomeDescriptor
     *
     * @return
     */
    private static AsciiLineReader getGeneReader(GenomeDescriptor genomeDescriptor) {

        InputStream is = null;
        try
        {

            InputStream inputStream = genomeDescriptor.getGeneFileNameInputStream();
            if (inputStream == null)
            {
                return null;
            }

            AsciiLineReader reader = null;
            if (genomeDescriptor.isGeneFileGZipFormat())
            {
                is = new GZIPInputStream(inputStream);
                reader = new AsciiLineReader(is);
            } else
            {
                is = new BufferedInputStream(inputStream);
                reader = new AsciiLineReader(is);
            }

            return reader;
        } catch (IOException ex)
        {
            logger.warn("Error loading the genome!", ex);
            return null;
        }

    }

    /**
     * Validate gene file
     *
     * @param file
     * @return
     */
    public static boolean isValid(File file) {

        try
        {

            if ((file == null) || !file.exists() || (file.length() < 1))
            {
                return false;
            } else
            {
                AsciiLineReader reader = new AsciiLineReader(new FileInputStream(file));
                return isValid(reader, file.getName());
            }
        } catch (Exception e)
        {
            return false;
        }

    }

    /**
     * Validate gene file
     *
     * @param reader
     * @param geneFilename
     * @return
     */
    public static boolean isValid(AsciiLineReader reader, String geneFilename) {

        if (reader != null)
        {

            try
            {
                FeatureParser parser = AbstractFeatureParser.getInstanceFor(geneFilename);
                List<Feature> features = parser.loadFeatures(reader);

                if ((features != null) && !features.isEmpty())
                {
                    return true;
                }

            } catch (Exception e)
            {
                logger.error("Invalid Gene file data : file=" + geneFilename, e);
            }
        }
        return false;
    }
}
