/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * FeatureUtils.java
 *
 * Useful utilities for working with Features
 */
package org.broad.igv.feature;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 *
 * @author jrobinso
 */
public class FeatureUtils {

    public static Map<String, List<Feature>> divideByChromosome(List<Feature> features) {
        Map<String, List<Feature>> featureMap = new LinkedHashMap();
        for (Feature f : features) {
            List<Feature> flist = featureMap.get(f.getChromosome());
            if (flist == null) {
                flist = new ArrayList();
                featureMap.put(f.getChromosome(), flist);
            }
            flist.add(f);
        }
        return featureMap;
    }

    /**
     * Segregate a list of possibly overlapping features into a list of
     * non-overlapping lists of features.
     */
    public static List<List<Feature>> segreateFeatures(List<Feature> features, double scale) {

        // Create a list to hold the lists of non-overlapping features
        List<List<Feature>> segmentedLists = new ArrayList();

        // Make a working copy of the original list.
        List<Feature> workingList = new LinkedList(features);
        sortFeatureList(workingList);

        // Loop until all features have been allocated to non-overlapping lists
        while (workingList.size() > 0) {

            List<Feature> nonOverlappingFeatures = new LinkedList();
            List<Feature> overlappingFeatures = new LinkedList();

            // Prime the loop with the first feature, it can't overlap itself
            Feature f1 = workingList.remove(0);
            nonOverlappingFeatures.add(f1);
            while (workingList.size() > 0) {
                Feature f2 = workingList.remove(0);
                int scaledStart = (int) (f2.getStart() / scale);
                int scaledEnd = (int) (f1.getEnd() / scale);
                if (scaledStart > scaledEnd) {
                    nonOverlappingFeatures.add(f2);
                    f1 = f2;
                } else {
                    overlappingFeatures.add(f2);
                }
            }

            // Add the list of non-overlapping features and start again with whats left
            segmentedLists.add(nonOverlappingFeatures);
            workingList = overlappingFeatures;
        }
        return segmentedLists;
    }

    /**
     * Sort the feature list by ascending start value
     */
    public static void sortFeatureList(List<? extends LocusScore> features) {

        Collections.sort(features, new Comparator() {

            public int compare(Object o1, Object o2) {
                LocusScore f1 = (LocusScore) o1;
                LocusScore f2 = (LocusScore) o2;
                return (int) (f1.getStart() - f2.getStart());
            }
        });
    }

    public static LocusScore getFeatureAt(double position, double minWidth,
            List<? extends LocusScore> features) {

        int startIdx = 0;
        int endIdx = features.size();

        while (startIdx != endIdx) {
            int idx = (startIdx + endIdx) / 2;

            LocusScore feature = features.get(idx);

            // Correct for zero vs 1 based coordinates.
            // TODO -- document this.
            double effectiveStart = feature.getStart() + 1;
            if (position >= effectiveStart) {

                double effectiveWidth = Math.max(minWidth, feature.getEnd() - feature.getStart());

                if (position <= effectiveStart + effectiveWidth) {
                    return features.get(idx);
                } else {
                    if (idx == startIdx) {
                        return null;
                    } else {
                        startIdx = idx;
                    }
                }
            } else {
                endIdx = idx;
            }
        }

        return null;
    }

    /**
     * Get the index of the feature just to the right of the given position.
     * If there is no feature to the right return -1;
     * @param position
     * @param features
     * @return
     */
    public static LocusScore getFeatureAfter(double position, List<? extends LocusScore> features) {

        if (features.size() == 0 ||
                features.get(features.size() - 1).getStart() <= position) {
            return null;
        }

        int startIdx = 0;
        int endIdx = features.size();

        while (startIdx != endIdx) {
            int idx = (startIdx + endIdx) / 2;
            double distance = features.get(idx).getStart() - position;
            if (distance <= 0) {
                startIdx = idx;
            } else {
                endIdx = idx;
            }
            if (endIdx - startIdx < 10) {
                break;
            }
        }

        for (int idx = startIdx; idx < features.size() - 1; idx++) {
            if (features.get(idx).getStart() > position) {
                return features.get(idx);
            }
        }

        return null;

    }

    public static LocusScore getFeatureBefore(double position, List<? extends LocusScore> features) {

        if (features.size() == 0 ||
                features.get(features.size() - 1).getStart() <= position) {
            return null;
        }

        int startIdx = 0;
        int endIdx = features.size() - 1;

        while (startIdx != endIdx) {
            int idx = (startIdx + endIdx) / 2;
            double distance = features.get(idx).getStart() - position;
            if (distance <= 0) {
                startIdx = idx;
            } else {
                endIdx = idx;
            }
            if (endIdx - startIdx < 10) {
                break;
            }
        }

        if (features.get(endIdx).getStart() >= position) {
            for (int idx = endIdx; idx >= 0; idx--) {
                if (features.get(idx).getStart() < position) {
                    return features.get(idx);
                }
            }
        } else {
            for (int idx = endIdx + 1; idx < features.size(); idx++) {
                if (features.get(idx).getStart() >= position) {
                    return features.get(idx - 1);
                }

            }
        }
        return null;


    }
}
