/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/







package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------




/**
 * A sub region of a feature.  For example,  a Gene exon
 *
 * @author jrobinso
 */
public class Exon extends AbstractFeature {

    /**
     * The index of the exon relative to the start codon.  The exon with the start
     * codon is number "1".
     */
    private int number;
    private int readingFrame = -1;
    private int codingStart;
    private int codingEnd;
    private AminoAcidSequence aminoAcidSequence;
    boolean utr = false;
    
    /**
     * Constructs ...
     *
     *
     *
     * @param chr
     * @param start
     * @param end
     * @param strand
     */
    public Exon(String chr, int start, int end, Strand strand) {
        super(chr, start, end, strand);
        
        // By default the entre exon is a coding region
        this.codingStart = start;
        this.codingEnd = end;
    }
    
    /**
     * Flag indicating that the entire exon is the UTR.
     * @param utr
     */
    public void setUTR(boolean utr) {
        this.utr = utr;
        if(getStrand() == Strand.POSITIVE) {
            codingStart = codingEnd = getEnd();
        }
        else {
            codingStart = codingEnd = getStart();
        }
    }

    /**
     * Method description
     *
     *
     * @param codingStart
     */
    public void setCodingStart(int codingStart) {
        this.codingStart = Math.max(getStart(), codingStart);
    }

    /**
     * Method description
     *
     *
     * @param codingEnd
     */
    public void setCodingEnd(int codingEnd) {
        this.codingEnd = Math.min(getEnd(), codingEnd);
    }

    /**
     * Method description
     *
     *
     * @param offset
     */
    public void setReadingFrame(int offset) {
        this.readingFrame = offset;
    }

    /**
     * Method description
     *
     *
     * @param phase
     */
    public void setPhase(int phase) {
        if (getStrand() == Strand.POSITIVE)
        {
            readingFrame = phase;
        }
        else if (getStrand() == Strand.NEGATIVE)
        {
            int modLen = (getCodingLength() - phase) % 3;
            readingFrame = modLen;
        }
    }


    /**
     * Method description
     *
     *
     * @return
     */
    public int getCdStart() {
        return codingStart;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getCdEnd() {
        return this.codingEnd;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getCodingLength() {
        return utr ? 0 : Math.max(0, codingEnd - codingStart);
    }

    /**
     * This is exposed for unit tests.
     *
     * @return
     */
    int getReadingShift() {
        return readingFrame;
    }


    /**
     * Method description
     *
     *
     * @return
     */
    public AminoAcidSequence getAminoAcidSequence() {
        if (aminoAcidSequence == null)
        {
            computeAminoAcidSequence();
        }
        return aminoAcidSequence;
    }

    /**
     * Method description
     *
     *
     * @param aminoAcidSequence
     */
    public void setAminoAcidSequence(AminoAcidSequence aminoAcidSequence) {
        this.aminoAcidSequence = aminoAcidSequence;
    }

    private void computeAminoAcidSequence() {

        if(utr) {
            return;
        }
        int start = getStart();
        int end = getEnd();
        String chr = getChromosome();
        if (readingFrame >= 0)
        {
            int readStart = (codingStart > start) ? codingStart : start + readingFrame;
            int readEnd = Math.min(end, codingEnd);
            if (readEnd > readStart + 3)
            {
                String genome = IGVModel.getInstance().getViewContext().getGenomeId();
                aminoAcidSequence = AminoAcidManager.getAminoAcidSequence(genome, chr, readStart,
                        readEnd, getStrand());
            }
        }
    }

    public LocusScore copy() {
        Exon copy = new Exon(getChromosome(), getStart(), getEnd(), getStrand());
        copy.aminoAcidSequence = this.aminoAcidSequence;
        copy.codingEnd = this.codingEnd;
        copy.codingStart = this.codingStart;
        return copy;
    }

    public String getValueString(double position, WindowFunction windowFunction) {
        return number > 0 ? "Exon number: " + number : "";
    }

    public int getNumber() {
        return number;
    }

    public void setNumber(int number) {
        this.number = number;
    }

}
