/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */






package org.broad.igv.feature;

//~--- non-JDK imports --------------------------------------------------------

import org.apache.log4j.Logger;

import org.broad.igv.track.WindowFunction;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * A convenience class providing default implementation for many Feature
 *  methods.
 * @author jrobinso
 */
public class BasicFeature extends AbstractFeature {

    private static Logger log = Logger.getLogger(BasicFeature.class);
    protected List<Exon> exons;
    protected int level = 1;
    private String type;
    protected String name = "";
    protected float score = Float.NaN;
    protected float confidence;
    String identifier;

    /**
     * Constructs ...
     *
     */
    public BasicFeature() {}

    /**
     * Constructs ...
     *
     *
     * @param chr
     * @param start
     * @param end
     */
    public BasicFeature(String chr, int start, int end) {

        this(chr, start, end, Strand.NONE);
    }

    /**
     * Constructs ...
     *
     *
     * @param chr
     * @param start
     * @param end
     * @param strand
     */
    public BasicFeature(String chr, int start, int end, Strand strand) {
        super(chr, start, end, strand);
    }

    /**
     * Constructs ...
     *
     *
     * @param feature
     */
    public BasicFeature(BasicFeature feature) {
        super(feature.getChromosome(), feature.getStart(), feature.getEnd(), feature.getStrand());
        this.name = feature.name;
        this.confidence = feature.confidence;
        this.color = feature.color;
        this.description = feature.description;
        this.exons = feature.exons;
        this.level = feature.level;
        this.score = feature.score;
        this.identifier = feature.identifier;
        this.type = feature.type;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public BasicFeature copy() {
        return new BasicFeature(this);
    }


    /**
     *
     *
     * @param identifier
     */
    public void setIdentifier(String identifier) {
        this.identifier = identifier;
    }

    /**
     *
     *
     * @param name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     *
     *
     * @return
     */
    public String getName() {
        return name;
    }

    /**
     * Return a string for popup text, and related uses.  The default just
     * returns the feature name.  Its expected that this method will be
     * overriden in subclasses.
     *
     *
     *
     * @param ignored
     * @return
     */
    public String getValueString(double position, WindowFunction ignored) {
         StringBuffer valueString = new StringBuffer();
       
        // Get exon number, if over an exon
        if(this.exons != null) {
            for(Exon exon : exons) {
                if(position >= exon.getStart() && position <= exon.getEnd()) {
                    valueString.append(exon.getValueString(position, ignored) + "<br>");
                }
            }
        }
        
        if (name != null)
        {
            valueString.append(name);
        }
        if ((identifier != null) && ((name == null) ||!name.equals(identifier)))
        {
            valueString.append("<br>" + identifier);
        }
        if (description != null)
        {
            valueString.append("<br>" + description);
        }
        return valueString.toString();
    }

    /**
     * Method description
     *
     *
     * @param score
     */
    public void setScore(float score) {
        this.score = score;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public float getScore() {
        return score;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public boolean hasScore() {
        return Float.isNaN(score) ? false : true;
    }

    /**
     * Method description
     *
     *
     * @param level
     */
    public void setLevel(int level) {
        this.level = level;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getLevel() {
        return level;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public List<Exon> getExons() {
        return exons;
    }

    /**
     * Method description
     *
     */
    public void sortExons() {
        if (exons != null)
        {
            Collections.sort(exons, new Comparator<Feature>() {

                public int compare(Feature arg0, Feature arg1) {
                    return arg0.getStart() - arg1.getStart();
                }
            });
        }
    }

    /**
     * Method description
     *
     *
     * @param region
     */
    public void addExon(Exon region) {
        if (exons == null)
        {
            exons = new ArrayList();
        }
        setStart(Math.min(getStart(), region.getStart()));
        setEnd(Math.max(getEnd(), region.getEnd()));
        exons.add(region);
    }

    /**
     * Method description
     *
     *
     * @param o
     *
     * @return
     */
    public int compareTo(Object o) {
        Feature otherObj = (Feature) o;
        return (int) (getStart() - otherObj.getStart());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public String getIdentifier() {
        return identifier;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getExonCount() {
        return (exons == null) ? 0 : exons.size();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getType() {
        return type;
    }

    /**
     * Method description
     *
     *
     * @param type
     */
    public void setType(String type) {
        this.type = type;
    }
}
