/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.feature;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author jrobinso
 */
public class AminoAcidManager {

    static String filename = "codonTable.txt";

    static Map<String, AminoAcid> codonTable;

    /**
     * Get the amino acid sequence for an interval.   
     * Assumptions and conventions
     * 
     *    The start and end positions are on the positive strand (reference coordinate system)
     *    irrespective of the read direction.
     * 
     *    Reading will begin from the startPosition if strand == POSITIVE, endPosition if NEGATIVE
     * 
     * @param startPosition
     * @param endPosition
     * @param direction
     * @return
     */
    public static AminoAcidSequence getAminoAcidSequence(String genomeId, String chr,
            int startPosition, int endPosition, Strand strand) {

        byte[] seqBytes = SequenceManager.readSequence(genomeId, chr, startPosition, endPosition);
        if (seqBytes == null) {
            return null;
        }
        else {
            String nucSequence = new String(seqBytes);
            List<AminoAcid> acids = getAminoAcids(nucSequence, strand);

            // Set the start position of this amino acid.  
            int aminoStart = startPosition;
            return new AminoAcidSequence(strand, aminoStart, acids);
        }

    }

    /**
     * Return an amino acid sequence for the input sequence.  
     * @param sequence
     * @param direction
     * @return
     */
    public static List<AminoAcid> getAminoAcids(String sequence, Strand direction) {

        if (codonTable == null) {
            initTable();
        }

        // Sequence must be divisible by 3. It is the responsibilty of the 
        // calling program to send a sequence properly aligned.  
        int readLength = sequence.length() / 3;
        List<AminoAcid> acids = new ArrayList(readLength);

        for (int i = 0; i <= sequence.length() - 3; i += 3) {
            String codon = sequence.substring(i, i + 3);
            if (direction == Strand.NEGATIVE) {
                codon = getNucleotideComplement(codon);
            }
            AminoAcid aa = codonTable.get(codon);
            if(aa == null) {
                aa = AminoAcid.NULL_AMINO_ACID;
            }
            acids.add(codonTable.get(codon));

        }
        return acids;
    }

    private static String getNucleotideComplement(String sequence) {
        char[] complement = new char[sequence.length()];
        for (int i = 0; i < sequence.length(); i++) {
            int j = sequence.length() - i - 1;
            char c = sequence.charAt(i);
            switch (c) {
                case 'T':
                    complement[j] = 'A';
                    break;
                case 'A':
                    complement[j] = 'T';
                    break;
                case 'C':
                    complement[j] = 'G';
                    break;
                case 'G':
                    complement[j] = 'C';
                    break;
                default:
                    complement[j] = c;
            }
        }
        return new String(complement);
    }

    static synchronized void initTable() {

        if (codonTable == null) {
            codonTable = new HashMap();


            try {

                InputStream is = AminoAcidManager.class.getResourceAsStream("/resources/" + filename);
                BufferedReader reader = new BufferedReader(new InputStreamReader(is));

                String nextLine = null;
                while ((nextLine = reader.readLine()) != null) {
                    String[] tokens = nextLine.split("\t");
                    if (tokens.length == 4) {
                        String codon = tokens[0].trim().toUpperCase();
                        String fullName = tokens[1].trim();
                        String shortName = tokens[2].trim();
                        char symbol = tokens[3].trim().charAt(0);
                        codonTable.put(codon, new AminoAcid(fullName, shortName, symbol));
                    }

                }
            } catch (IOException ex) {
                Logger.getLogger(AminoAcidManager.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
    }
}
