/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.feature;

import org.apache.log4j.Logger;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.renderer.BasicFeatureRenderer;
import org.broad.igv.renderer.GeneTrackRenderer;
import org.broad.igv.track.FeatureTrack;
import org.broad.igv.track.TrackProperties;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.data.DatasetParserUtils;

import java.io.FileWriter;
import java.io.IOException;

import java.io.PrintWriter;

import java.util.ArrayList;
import java.util.List;
import org.broad.igv.track.TrackType;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractFeatureParser implements FeatureParser {

    private static Logger log = Logger.getLogger(IGVMainFrame.class);
    private String filePath;

    /* An object to collection track properties, if specified in the feature file. */
    private TrackProperties trackProperties = null;

    /**
     * Return an parser instance appropriate the the file type.  Currently the filename
     * is used to determine file type, this is fragile obviously but what is the alternative?
     *
     * @param filename
     * @return
     */
    public static FeatureParser getInstanceFor(String filename) {

        String tmp = getStrippedFilename(filename);

        if (tmp.endsWith("bed") || tmp.endsWith("map")) {
            return new BEDFileParser();
        }

        if (tmp.contains("refflat")) {
            return new UCSCGeneTableParser(UCSCGeneTableParser.Type.REFFLAT);
        } else if (tmp.contains("genepred") || tmp.contains("ensgene") || tmp.contains("refgene")) {
            return new UCSCGeneTableParser(UCSCGeneTableParser.Type.GENEPRED);
        } else if (tmp.contains("ucscgene")) {
            return new UCSCGeneTableParser(UCSCGeneTableParser.Type.UCSCGENE);
        } else if (tmp.endsWith("gff") || tmp.endsWith("gtf")) {
            return new GFFParser();
        } else if (tmp.endsWith("gff3")) {
            return new GFF3Parser();
        } else if (tmp.endsWith("embl")) {
            return new EmblFeatureTableParser();
        } else {
            return new BEDFileParser();
        }




    }

    /**
     * Return true if the file represented by the locator contains feature.  This method
     * returns true by default.  It can be overriden by subclasses representing ambiguous
     * file content.
     * 
     * @param locator
     * @return true if a feature file
     */
    public boolean isFeatureFile(ResourceLocator locator) {
        return true;

    }

    /**
     * Load all features in this file.
     *
     * @param locator
     *
     * @return
     */
    public List<FeatureTrack> loadTracks(ResourceLocator locator) {

        FeatureTrack track = new FeatureTrack(locator, locator.getDisplayName());

        List<Feature> features = loadFeatures(track, locator, -1);
        track.initFeatures(features);

        track.setRendererClass(BasicFeatureRenderer.class);
        track.setMinimumHeight(35);
        track.setHeight(45);
        track.setRendererClass(GeneTrackRenderer.class);
        if (trackProperties != null) {
            track.setTrackProperties(trackProperties);
        }

        List<FeatureTrack> tracks = new ArrayList();
        tracks.add(track);
        return tracks;
    }

    /**
     * Parse a limited number of lines in this file and return a list of features found.
     *
     *
     * @param locator
     * @param maxLines
     *
     * @return
     */
    protected List<Feature> loadFeatures(FeatureTrack track, ResourceLocator locator, int maxLines) {

        // File file = new File(locator.getPath());
        int nLines = 0;
        AsciiLineReader reader = null;
        List<Feature> features = new ArrayList<Feature>();
        try {
            reader = ParsingUtils.openAsciiReader(locator);
            return loadFeatures(track, reader, maxLines);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    protected List<Feature> loadFeatures(ResourceLocator locator, int maxLines) {
        return loadFeatures(null, locator, maxLines);
    }

    /**
     * Method description
     *
     *
     * @param reader
     *
     * @return
     */
    public List<Feature> loadFeatures(AsciiLineReader reader) {
        return loadFeatures(null, reader, -1);
    }

    /**
     * Load all features in this file.
     *
     *
     * @param reader
     * @param maxLines
     *
     * @return
     *
     */
    public List<Feature> loadFeatures(FeatureTrack track, AsciiLineReader reader, int maxLines) {

        List<Feature> features = new ArrayList<Feature>();

        try {
            int nLines = 0;
            String nextLine;
            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0)) {
                nextLine = nextLine.trim();
                nLines++;
                if ((maxLines > 0) && (nLines > maxLines)) {
                    break;
                }

                try {
                    if (track != null && nextLine.startsWith("#type")) {
                        String[] tokens = nextLine.split("=");
                        if (tokens.length > 1) {
                            try {
                                TrackType type = TrackType.valueOf(tokens[1]);
                                track.setTrackType(type);
                            } catch (Exception e) {
                                log.error("Error converting track type: " + tokens[1]);
                            }
                        }
                    //} else if (nextLine.startsWith("track")) {
                    // TODO -- parse track line
                    } else {
                        Feature feature = parseLine(nextLine);
                        if (feature != null) {
                            features.add(feature);
                        }
                    }

                } catch (NumberFormatException e) {

                    // Expected condition -- for example comments.  don't log as it slows down
                    // the parsing and is not useful information.
                }
            }

            parsingComplete(features);
            FeatureDB.addFeatures(features);
        } catch (java.io.EOFException e) {

            // This exception is due to a known bug with java zip library.  Not
            // in general a real error, and nothing we can do about it in any
            // event.
            return features;
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }

        return features;
    }

    protected void parsingComplete(List<Feature> features) {

        // Default -- do nothing.
    }

    abstract protected Feature parseLine(String nextLine);

    protected static String getStrippedFilename(String filename) {
        String tmp = filename.toLowerCase();

//      String off common, non-informative extensions .gz, .tab, .txt, and .csv
        if (filename.endsWith(".gz")) {
            tmp = tmp.substring(0, tmp.length() - 3);
        }
        if (tmp.endsWith(".tab") || tmp.endsWith(".txt") || tmp.endsWith(".csv")) {
            tmp = tmp.substring(0, tmp.length() - 4);
        }
        return tmp;
    }

    /**
     * Convenience method.  Write a list of features out as a BED file
     *
     * @param features
     * @param outputfile
     */
    public static void dumpFeatures(List<Feature> features, String outputfile) {

        PrintWriter pw = null;

        try {
            pw = new PrintWriter(new FileWriter(outputfile));
            pw.println("Header row");
            for (Feature gene : features) {
                pw.print(gene.getName() + "\t");
                pw.print(gene.getIdentifier() + "\t");
                pw.print(gene.getChromosome() + "\t");
                if (gene.getStrand() == Strand.POSITIVE) {
                    pw.print("+\t");
                } else if (gene.getStrand() == Strand.NEGATIVE) {
                    pw.print("-\t");
                } else {
                    pw.print(" \t");
                }
                pw.print(gene.getStart() + "\t");
                pw.print(gene.getEnd() + "\t");

                List<Exon> regions = gene.getExons();
                pw.print(regions.size() + "\t");
                for (Exon exon : regions) {
                    pw.print(exon.getStart() + ",");
                }
                pw.print("\t");
                for (Exon exon : regions) {
                    pw.print(exon.getEnd() + ",");
                }
                pw.println();

            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (pw != null) {
                pw.close();
            }
        }
    }

    protected void setTrackProperties(TrackProperties trackProperties) {
        this.trackProperties = trackProperties;
    }
}
