/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.data.rnai;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.feature.ParsingUtils;

//~--- JDK imports ------------------------------------------------------------

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public class RNAIHairpinParser {

    private static Logger log = Logger.getLogger(RNAIHairpinParser.class);
    private String filename;

    /**
     * Constructs ...
     *
     *
     * @param filename
     */
    public RNAIHairpinParser(String filename) {

        this.filename = filename;
    }

    /**
     * Method description
     *
     */
    public void parse() {


        AsciiLineReader reader = null;

        try
        {
            log.debug("Loading data for: " + filename);
            reader = new AsciiLineReader(new FileInputStream(filename));

            // Parse comments
            parseAttributes(reader);

            // Parse header
            parseHeaderRow(reader);

            // Parse data
            String nextLine = reader.readLine();

            while ((nextLine = reader.readLine()) != null)
            {
                if (!nextLine.startsWith("#"))
                {
                    String[] tokens = new String[15];
                    int nTokens = ParsingUtils.split(nextLine, tokens, '\t');
                    if (nTokens > 11)
                    {
                        try
                        {
                            String batchId = new String(tokens[0].trim());

                            String hairpinName = new String(tokens[4].trim().toUpperCase());

                            float scoreMean = Float.NaN;
                            try
                            {
                                scoreMean = Float.parseFloat(tokens[8]);

                            } catch (NumberFormatException numberFormatException)
                            {

                            // Nothing to do -- expected condition
                            }

                            float scoreSTD = 0;
                            try
                            {
                                scoreSTD = Float.parseFloat(tokens[9]);
                            } catch (NumberFormatException numberFormatException)
                            {

                            // Nothing to do -- expected condition
                            }

                            String gene = new String(tokens[13].trim().toUpperCase());

                            RNAIHairpinValue score = new RNAIHairpinValue(hairpinName, scoreMean,
                                scoreSTD);

                            // Make batch_conditon key
                            // Rules from Jessee -- ignore conditions starting with *.  None, standard,
                            // and blank are all equivalent.
                            String cond = tokens[2].trim();
                            if (!cond.startsWith("*"))
                            {
                                if (cond.equals("None") || cond.equals("Standard"))
                                {
                                    cond = "";
                                }

                                String batchCond = batchId + "_" + cond;

                                RNAIHairpinCache.getInstance().addHairpinScore(batchCond, gene,
                                    score);
                            }
                        } catch (Exception ex)
                        {
                            log.error("Skipping line: " + nextLine, ex);
                        }
                    }
                }
            }

        } catch (FileNotFoundException e)
        {
            log.error("RNAI file not found: " + filename);
            throw new RuntimeException("File not found: " + filename);

        } catch (IOException e)
        {
            log.error(filename, e);
            throw new RuntimeException("Error parsing file " + filename, e);
        } finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }
    }

    /**
     * Parse the attributes from the comment section and annotate the
     *
     *
     */
    private void parseAttributes(AsciiLineReader reader) throws IOException {

    // TODO -- parse comments
    }

    private void parseHeaderRow(AsciiLineReader reader) throws IOException {

    // Nothing to do here.  Column positions are fixed.
    }
}
