/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */




package org.broad.igv.data.rnai;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.feature.Feature;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.track.WindowFunction;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Class description
 *
 *
 * @version    Enter version here..., 08/10/06
 * @author     Enter your name here...
 */
public class RNAIGeneScore implements LocusScore {

    String batchCondition;
    Feature gene;
    float geneScore;
    int geneConfidence;
    boolean hasGeneConfidence;
    int numberOfHairpins;
    int start;
    int end;

    /**
     * Constructs ...
     *
     *
     *
     * @param batchId
     * @param gene
     * @param geneScore
     * @param numberOfHairpins
     */
    public RNAIGeneScore(String batchId, Feature gene, float geneScore,
                         int numberOfHairpins) {
        this.batchCondition = batchId;
        this.gene = gene;
        this.geneScore = geneScore;
        geneConfidence = 3;
        hasGeneConfidence = false;
        this.numberOfHairpins = numberOfHairpins;
        this.start = gene.getStart();
        this.end = gene.getEnd();
    }
    /**
     * Constructs ...
     *
     *
     *
     * @param batchId
     * @param gene
     * @param geneScore
     * @param confidence
     * @param numberOfHairpins
     */
    public RNAIGeneScore(String batchId, Feature gene, float geneScore, int confidence,
                         int numberOfHairpins) {
        this.batchCondition = batchId;
        this.gene = gene;
        this.geneScore = geneScore;
        this.geneConfidence = confidence;
        hasGeneConfidence = true;        
        this.numberOfHairpins = numberOfHairpins;
        this.start = gene.getStart();
        this.end = gene.getEnd();
    }

    /**
     * Copy constructor
     *
     * @param score
     */
    public RNAIGeneScore(RNAIGeneScore score) {
        this.batchCondition = score.batchCondition;
        this.gene = score.gene;
        this.geneScore = score.geneScore;
        this.geneConfidence = score.geneConfidence;
        this.numberOfHairpins = score.numberOfHairpins;
        this.start = score.getStart();
        this.end = score.getEnd();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public RNAIGeneScore copy() {
        return new RNAIGeneScore(this);
    }


    /**
     * Checks if there is a confidence value for the gene score
     *
     *
     * @return
     */
    public boolean hasGeneConfidence() {
        return hasGeneConfidence;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Feature getGene() {
        return gene;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getStart() {
        return start;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getEnd() {
        return end;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public float getScore() {
        return geneScore;
    }

    // This method required for interface

    /**
     * Method description
     *
     *
     * @return
     */
    public float getConfidence() {
        return geneConfidence;
    }

    /**
     * Method description
     *
     *
     * @param start
     */
    public void setStart(int start) {
        this.start = start;
    }

    /**
     * Method description
     *
     *
     * @param end
     */
    public void setEnd(int end) {
        this.end = end;
    }

    /**
     * Method description
     *
     *
     * @param confidence
     */
    public void setConfidence(float confidence) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Return the list of hairpin scores associated with this gene.  Scores are sorted alphabetically
     * by haipin name
     *
     * @return
     */
    List<RNAIHairpinValue> hairpinScores = null;

    /**
     * Method description
     *
     *
     * @return
     */
    public Collection<RNAIHairpinValue> getHairpinValues() {
        if (hairpinScores == null)
        {
            Collection<RNAIHairpinValue> tmp =
                RNAIHairpinCache.getInstance().getHairpinScores(batchCondition, gene.getName().toUpperCase());
            if (tmp != null)
            {
                hairpinScores = new ArrayList(tmp);
            }
        }
        
        return hairpinScores;
    }

    /**
     * Method description
     *
     *
     * @param ignored
     *
     * @return
     */
    public String getValueString(double position, WindowFunction ignored) {

        StringBuffer buf = new StringBuffer(100);
        buf.append("<b>Gene: " + gene.getName() + "</b><br>");
        buf.append("<b>Intensity: " + geneScore + "<br></b>");

        if(hasGeneConfidence)
            buf.append("Confidence: " + geneConfidence + "<br>");
        
        Collection<RNAIHairpinValue> hpins = getHairpinValues();
        if ((hpins == null) || hpins.isEmpty())
        {
            buf.append("# Hairpins: " + numberOfHairpins + "<br>");
        }
        else
        {
            buf.append("Hairpin scores:<br>");
            for (RNAIHairpinValue hpScore : hpins)
            {
                if(hpScore.hasScoreSTD())
                {
                    buf.append(hpScore.getName() + ": " + hpScore.getScoreMean() + "  ("
                           + hpScore.getScoreSTD() + ")<br>");
                }
                else
                {
                     buf.append(hpScore.getName() + ": " + hpScore.getScoreMean() + "<br>");   
                }
            }
        }

        return buf.toString();

    }
}
