/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */






package org.broad.igv.data.rnai;

//~--- non-JDK imports --------------------------------------------------------

import java.util.Collection;
import org.broad.igv.IGVConstants;

import org.broad.igv.data.*;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.Genome;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.track.TrackType;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.feature.FeatureUtils;

/**
 *
 * @author jrobinso
 */
public class RNAIDataSource implements DataSource {

    /**
     * The screen name (or batch id)
     */
    private String screen;

    /**
     * The screen "condition".  This value can be null.
     */
    private String condition;
    
    private String displayName;
    
    private boolean scoresAreSorted = false;

    /**
     * Map of chr -> sorted list of data points.  Data is sorted by increasing
     * start location.
     */
    Map<String, List<LocusScore>> dataMap;

    /**
     * Constructs ...
     *
     *
     *
     * @param screen
     * @param condition
     */
    public RNAIDataSource(String screen, String condition) {
        this.screen = screen;
        this.condition = condition;
        this.displayName = screen;
        if(condition != null && condition.length() > 0) {
            displayName += " (" + condition + " )";
        }
        dataMap = new HashMap();
    }

    /**
     * Method description
     *
     *
     * @param dpt
     */
    public void addGeneScore(RNAIGeneScore dpt) {
        String chr = dpt.getGene().getChromosome();
        List<LocusScore> chrDataPoints = dataMap.get(chr);
        if (chrDataPoints == null)
        {
            chrDataPoints = new ArrayList(500);
            dataMap.put(chr, chrDataPoints);
        }
        chrDataPoints.add(dpt);

        // Also add the score to the "All" chromosome.
        List<LocusScore> chrAllScores = dataMap.get(IGVConstants.CHR_ALL);
        if (chrAllScores == null)
        {
            chrAllScores = new ArrayList(500);
            dataMap.put(IGVConstants.CHR_ALL, chrAllScores);
        }
        Genome genome = IGVModel.getInstance().getViewContext().getGenome();
        RNAIGeneScore genomeScore = new RNAIGeneScore(dpt);
        int genomeStart = genome.getGenomeCoordinate(dpt.getGene().getChromosome(), dpt.getStart());
        int genomeEnd = genome.getGenomeCoordinate(dpt.getGene().getChromosome(), dpt.getEnd());
        genomeScore.setStart(genomeStart);
        genomeScore.setEnd(genomeEnd);
        chrAllScores.add(genomeScore);
    }
    
    /**
     * Sort gene scores by ascending start position.  This should be called after all gene scores
     * have been added.
     */
    private void sortScores() {       
        for(List<LocusScore> scores : dataMap.values()) {
            FeatureUtils.sortFeatureList(scores);
        }
        
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param windowFunction
     *
     * @return
     */
    public double getDataMax(String chr) {
        return 3;
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param windowFunction
     *
     * @return
     */
    public double getDataMin(String chr) {
        return -3;
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param startLocation
     * @param endLocation
     * @param zoom
     * @param windowFunction
     *
     * @return
     */
    public List<LocusScore> getSummaryScoresForRange(String chr, int startLocation,
            int endLocation, int zoom) {
        
        if(!scoresAreSorted) {
            sortScores();
        }
        return dataMap.get(chr);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getTrackType() {
        return TrackType.RNAI;
    }

    /**
     * Method description
     *
     *
     * @param statType
     */
    public void setWindowFunction(WindowFunction statType) {

        // Ignored
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isLogNormalized() {
        return true;
    }

    /**
     * Method description
     *
     *
     * @param timestamp
     */
    public void refreshData(long timestamp) {

        // Ignored
    }

    /**
     * The screen name (or batch id)
     *
     * @return
     */
    public String getScreen() {
        return screen;
    }


    /**
     * The screen "condition".  This value can be null.
     *
     * @return
     */
    public String getCondition() {
        return condition;
    }
    
    public String getDisplayName() {
        return displayName;
    }
    
    /** RNAi data is not windowed */
    public WindowFunction getWindowFunction() {
        return null;
    }

    static List<WindowFunction> emptyList = new ArrayList();
    public Collection<WindowFunction> getAvailableWindowFunctions() {
        return emptyList;
    }
}
