/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.data;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import org.apache.log4j.Logger;


/**
 * This class encapuslates data for a single chromosome from a segmented
 * dataset.   Its main purpose is to provide a single object that can
 * easily be serialized and passed from server to IGV clients.
 * 
 * @author jrobinso
 */
public class SegmentedChromosomeData {

    private static Logger log = Logger.getLogger(SegmentedChromosomeData.class);
    
    private String[] sampleNames;

    Map<String, int[]> startLocations;

    Map<String, int[]> endLocations;

    Map<String, float[]> valueMap;

    public SegmentedChromosomeData() {
    }

    public SegmentedChromosomeData(
            String[] sampleNames,
            Map<String, int[]> startLocations,
            Map<String, int[]> endLocations,
            Map<String, float[]> valueMap) {
        this.sampleNames = sampleNames;
        this.startLocations = startLocations;
        this.endLocations = endLocations;
        this.valueMap = valueMap;
    }

    /**
     * Serialize this instance to a stream.  
     * 
     * Format:
     * Sample names as a string : 
     *   sample1 tab sample2 tabnetc...  line-feed 
     * Followed by (for each sample) 
     *   numberOfPoints (int) 
     *   start locations (int array)
     *   end locations (int array)
     *   values (float array)
     * 
     * @param stream  stream to serialize to
     * 
     */
    public void serialize(OutputStream stream) {

        try {
            DataOutputStream os = new DataOutputStream(new BufferedOutputStream(stream));

            for (String sample : getSampleNames()) {
                os.writeChars(sample);
                os.writeChar('\t');
            }
            os.writeChar('\n');
            for (String sample : getSampleNames()) {
                int[] starts = startLocations.get(sample);
                int[] ends = endLocations.get(sample);
                float[] values = valueMap.get(sample);
                int nPts = starts.length;

                assert ends.length == nPts;
                assert values.length == nPts;

                os.writeInt(nPts);
                for (int i = 0; i < nPts; i++) {
                    os.writeInt(starts[i]);
                }
                for (int i = 0; i < nPts; i++) {
                    os.writeInt(ends[i]);
                }
                for (int i = 0; i < nPts; i++) {
                    os.writeFloat(values[i]);
                }
            }
            os.flush();

        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Initialize the object by deserializing the stream.    
     * @see #serialize
     * @param stream
     */
    public void deserialize(InputStream stream) {
        startLocations = new HashMap();
        endLocations = new HashMap();
        valueMap = new HashMap();
        try {

            DataInputStream is = new DataInputStream(new BufferedInputStream(stream));

            StringArrayList strings = new StringArrayList(100);
            char c = is.readChar();
            while (c != '\n') {
                CharArrayList chars = new CharArrayList(10);
                while (c != '\t') {
                    chars.add(c);
                    c = is.readChar();
                }
                String nm = new String(chars.toArray()).trim();
                strings.add(nm);
                c = is.readChar();
            }
            sampleNames = strings.toArray();

            for (String sample : getSampleNames()) {
                int nPts = is.readInt();
                int[] starts = new int[nPts];
                for (int i = 0; i < nPts; i++) {
                    starts[i] = is.readInt();
                }
                int[] ends = new int[nPts];
                for (int i = 0; i < nPts; i++) {
                    ends[i] = is.readInt();
                }
                float[] values = new float[nPts];
                for (int i = 0; i < nPts; i++) {
                    values[i] = is.readFloat();
                }
                startLocations.put(sample, starts);
                endLocations.put(sample, ends);
                valueMap.put(sample, values);
            }

        } 
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    public String[] getSampleNames() {
        return sampleNames;
    }

    public int[] getStartLocations(String sampleName) {
        return startLocations.get(sampleName);
    }

    public int[] getEndLocations(String sampleName) {
        return endLocations.get(sampleName);
    }

    public float[] getValues(String sampleName) {
        return valueMap.get(sampleName);
    }
    }
