/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.data;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;

/**
 *
 * @author jrobinso
 */
public class SegmentedBinaryLocalReader implements SegmentedBinaryReader {

    String filePath;

    List<String> chromosomes;

    List<String> sampleNames;

    Map<String, String> dataFileMap = new HashMap();

    Map<String, String> attrs;

    public SegmentedBinaryLocalReader(String filePath) {
        this.filePath = filePath;
    }

    /**
     * Return the set of sample names in this data set
     * @return
     */
    public List<String> getSampleNames() {
        if (sampleNames == null) {
            sampleNames = readAsStrings("data/samples.txt");
        }
        return sampleNames;

    }

    public String getStringAttribute(String key) {
        return getAttributes().get(key);
    }

    public Map<String, String> getAttributes() {
        if (attrs == null) {
            attrs = new HashMap();
            List<String> lines = readAsStrings("data/attributes.txt");
            if (lines != null) {
                for (String kv : lines) {
                    String[] tokens = kv.split("=");
                    attrs.put(tokens[0], tokens[1]);
                }
            }
        }
        return attrs;

    }

    public SegmentedChromosomeData getChromosomeData(String chr) {

        return readChromosomeData(chr);

    }

    /**
     * Return the segment start locations for the sample and chromosome
     * @param sample the sample name
     * @param chr the chromosome name
     * @return array of end locations
     */
    public int[] getStartLocations(String sample, String chr) {
        String entryName = "data/" + sample + "/" + chr + "/starts.bin";
        return readAsInts(entryName);
    }

    /**
     * Return the segment end locations for the sample and chromosome
     * @param sample the sample name
     * @param chr the chromosome name
     * @return array of end locations
     */
    public int[] getEndLocations(String sample, String chr) {
        String entryName = "data/" + sample + "/" + chr + "/ends.bin";
        return readAsInts(entryName);
    }

    /**
     * Return the data values for the given sample and chromosom
     * @param sample the sample name
     * @param chr the chromosome name
     * @return data values
     */
    public float[] getValues(String sample, String chr) {
        String entryName = "data/" + sample + "/" + chr + "/values.bin";
        return readAsFloats(entryName);
    }

    private List<String> readAsStrings(String entryName) {
        List<String> strings = new ArrayList();
        ZipFile zipFile = null;
        try {
            zipFile = new ZipFile(new File(filePath));

            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                return null;
            }

            BufferedReader br = new BufferedReader(new InputStreamReader(zipFile.getInputStream(entry)));
            String nextLine = null;
            while ((nextLine = br.readLine()) != null) {
                strings.add(nextLine.trim());
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            ZipFile.closeQuietly(zipFile);

        }
        return strings;

    }

    private SegmentedChromosomeData readChromosomeData(String chr) {
        ZipFile zipFile = null;
        BufferedInputStream is = null;
        try {
            String entryName = "data/" + chr + ".bin";
            zipFile = new ZipFile(new File(filePath));
            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                return null;
            }

            is = new BufferedInputStream(zipFile.getInputStream(entry));

            SegmentedChromosomeData cd = new SegmentedChromosomeData();
            cd.deserialize(is);
            return cd;

        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            ZipFile.closeQuietly(zipFile);
        }

    }

    /**
     * Read the contents of a zip entry as an array of ints.
     * 
     * @param entryName name of the entry
     * @return array of ints
     */
    private int[] readAsInts(String entryName) {

        ZipFile zipFile = null;
        try {
            zipFile = new ZipFile(new File(filePath));
            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                return null;
            }

            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(zipFile.getInputStream(entry)));

            int nInts = (int) (entry.getSize() / 4);
            int[] ints = new int[nInts];
            for (int i = 0; i <
                    nInts; i++) {
                ints[i] = is.readInt();
            }

            return ints;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            ZipFile.closeQuietly(zipFile);
        }

    }

    /**
     * Read the content of a zip entry as an array of floats.
     * @param entryName name of the entry
     * @return contents as an array of floats
     */
    private float[] readAsFloats(String entryName) {

        ZipFile zipFile = null;
        try {
            zipFile = new ZipFile(new File(filePath));
            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                return null;
            }

            DataInputStream is = new DataInputStream(
                    new BufferedInputStream(zipFile.getInputStream(entry)));

            int nInts = (int) (entry.getSize() / 4);
            float[] values = new float[nInts];
            for (int i = 0; i <
                    nInts; i++) {
                values[i] = is.readFloat();
            }

            return values;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            ZipFile.closeQuietly(zipFile);
        }


    }

}


