/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

package org.broad.igv.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.IGVConstants;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackType;
import org.broad.igv.ui.IGVModel;

/**
 *
 * @author jrobinso
 */
public class SegmentedBinaryDataSet implements SegmentedDataSet {

    SegmentedBinaryReader reader;

    private long lastRefreshTime = 0;

    /**
     * Assume data is non-log value until suggested otherwise by the precense
     * of negative numbers.  TODO This is a fragile assumption, the user should
     * input this information directly.
     */
    Boolean logNormalized = null;

    private List<String> sampleNames = null;

    /** Map of [heading ->  [chr -> [list of segmentsCache]]] */
    Map<String, Map<String, List<LocusScore>>> segmentsCache = new HashMap();

    Map<String, SegmentedChromosomeData> chrData = new HashMap();
    
    TrackType type = TrackType.COPY_NUMBER;

    public SegmentedBinaryDataSet(ResourceLocator locator) {
        reader = locator.isLocal() ? new SegmentedBinaryLocalReader(locator.getPath()) : new SegmentedBinaryRemoteReader(locator);

        try {
            type = TrackType.valueOf(reader.getStringAttribute("type"));
        }
        catch(Exception e) {
            //ignore;
        }
        
        try {
            String logNormalizedString = reader.getStringAttribute("logNormalized");
            if(logNormalizedString != null) {
                logNormalized = logNormalizedString.equals("true");
            }

        } catch (Exception exception) {
            // ignore
        }

        // TODO Temporary -- start loading whole genome scores ascynchonously
        if (!IGVModel.getInstance().getViewContext().getChrName().equals(IGVConstants.CHR_ALL)) {
            (new Thread(new Runnable() {

                public void run() {
                    synchronized (chrData) {
                        SegmentedChromosomeData cd = reader.getChromosomeData(IGVConstants.CHR_ALL);
                        chrData.put(IGVConstants.CHR_ALL, cd);
                    }
                }
                })).start();
        }

    }

    public synchronized void refreshData(long timestamp) {
        if (timestamp > lastRefreshTime) {
            logNormalized = false;
            segmentsCache =
                    new HashMap();
            lastRefreshTime =
                    timestamp;
        }

    }

    // TODO -- the synchronized keyword can be removed with the async whole genome load
    public List<LocusScore> getSegments(String heading, String chr) {

        Map<String, List<LocusScore>> chrSegments = segmentsCache.get(heading);
        if (chrSegments == null) {
            chrSegments = new HashMap();
            segmentsCache.put(heading, chrSegments);
        }

        List<LocusScore> segments = chrSegments.get(chr);

        if (segments == null) {

            SegmentedChromosomeData cd = chrData.get(chr);
            if (cd == null) {
                cd = reader.getChromosomeData(chr);
                chrData.put(chr, cd);
            }

            int[] startLocations = cd.getStartLocations(heading);
            int[] endLocations = cd.getEndLocations(heading);
            float[] values = cd.getValues(heading);
            
            if(startLocations == null || startLocations.length == 0) {
                return null;
            }

            if (logNormalized == null) {
                logNormalized = false;
                for (int i = 0; i <
                        values.length; i++) {
                    if (values[i] < 0) {
                        logNormalized = true;
                        break;
                    }

                }
            }
            assert (startLocations.length == endLocations.length);
            assert (endLocations.length == values.length);

            segments = new ArrayList(startLocations.length);
            for (int i = 0; i <
                    startLocations.length; i++) {
                segments.add(new Segment(startLocations[i], endLocations[i], values[i]));
                chrSegments.put(chr, segments);
            }

        }
        return segments;
    }

    public List<String> getSampleNames() {
        if (sampleNames == null) {
            sampleNames = reader.getSampleNames();
        }

        return sampleNames;
    }

    public TrackType getType() {
        return type;
    }

// TODO -- check for negative numbers
    public boolean isLogNormalized() {
        return logNormalized == null ? true : logNormalized;
    }

    public double getDataMax(String chr) {
        return 3;
    }

    public double getDataMin(String chr) {
        return -3;
    }

    public synchronized List<LocusScore> getWholeGenomeScores(String heading) {
        return getSegments(heading, IGVConstants.CHR_ALL);

    }

    public static class ChromosomeChunk {
    }

}
