/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * sample       chrom   loc.start       loc.end num.mark        num.informative seg.mean
TCGA-02-0001-01C-01D-0183-04    1       554268  74674720        6892    6721    0.2077
TCGA-02-0001-01C-01D-0183-04    1       74693652        75110251        37      37      -0.2659
 */
package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.feature.Genome;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackType;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 *
 * @author jrobinso
 */
public class SegmentedAsciiDataSet implements SegmentedDataSet {

    SegmentFileParser parser;
    TrackType trackType = TrackType.COPY_NUMBER;
    float dataMax = -Float.MAX_VALUE;
    float dataMin = Float.MAX_VALUE;
    /**
     * Assume data is non-log value until suggested otherwise by the precense
     * of negative numbers.  TODO This is a fragile assumption, the user should
     * input this information directly.
     */
    boolean logNormalized = false;
    /** Map of [heading ->  [chr -> [list of chrSegments]]] */
    Map<String, Map<String, List<LocusScore>>> segments = new HashMap();
    /** Set of chromosomes represented in this dataset */
    Set<String> chromosomes = new HashSet();
    List<String> headings = new ArrayList();
    private Map<String, List<LocusScore>> wholeGenomeScoresCache = new HashMap();
    private long lastRefreshTime = 0;

    /**
     * Constructs ...
     *
     *
     * @param locator
     */
    public SegmentedAsciiDataSet(ResourceLocator locator) {
        parser = new SegmentFileParser(locator);
        parser.loadSegmentTracks(this);

    }

    /**
     * Method description
     *
     *
     * @param timestamp
     */
    public synchronized void refreshData(long timestamp) {
        if (timestamp > lastRefreshTime) {
            dataMax = -Float.MAX_VALUE;
            dataMin = Float.MAX_VALUE;
            logNormalized = false;
            segments = new HashMap();
            headings = new ArrayList();
            wholeGenomeScoresCache.clear();
            parser.loadSegmentTracks(this);
            lastRefreshTime = timestamp;
        }

    }

    /**
     * Method description
     *
     *
     * @param heading
     * @param chr
     * @param start
     * @param end
     * @param value
     * @param snpCount
     */
    public void addSegment(String heading, String chr, int start, int end, float value,
            int snpCount) {

        Map<String, List<LocusScore>> chrSegments = segments.get(heading);
        if (chrSegments == null) {
            headings.add(heading);
            chrSegments = new HashMap();
            segments.put(heading, chrSegments);
        }

        List<LocusScore> segmentList = chrSegments.get(chr);
        if (segmentList == null) {
            segmentList = new ArrayList<LocusScore>();
            chrSegments.put(chr, segmentList);
        }
        segmentList.add(new Segment(start, start, end, end, value, snpCount));
        dataMax = Math.max(dataMax, value);
        dataMin = Math.min(dataMin, value);
        if (value < 0) {
            logNormalized = true;
        }

        chromosomes.add(chr);

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Set<String> getChromosomes() {
        return chromosomes;
    }

    /**
     * Method description
     *
     *
     * @param heading
     * @param chr
     *
     * @return
     */
    public List<LocusScore> getSegments(String heading, String chr) {
        Map<String, List<LocusScore>> chrSegments = segments.get(heading);
        return (chrSegments == null) ? null : chrSegments.get(chr);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public List<String> getDataHeadings() {
        return headings;
    }


    public List<String> getSampleNames() {
        return headings;
    }
   /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getType() {
        return trackType;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isLogNormalized() {
        return logNormalized;
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public double getDataMax(String chr) {
        return dataMax;
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public double getDataMin(String chr) {
        return dataMin;
    }

    /**
     * Method description
     *
     *
     * @param heading
     *
     * @return
     */
    public List<LocusScore> getWholeGenomeScores(String heading) {


        List<LocusScore> wholeGenomeScores = wholeGenomeScoresCache.get(heading);
        if ((wholeGenomeScores == null) || wholeGenomeScores.isEmpty()) {
            int locationUnit = 1000;
            Genome genome = IGVModel.getInstance().getViewContext().getGenome();

            // Compute the smallest concievable feature that could be viewed on the
            // largest screen.  Be conservative.   The smallest feature is one at
            // the screen resolution scale in <chr units> / <pixel>
            int maxScreenSize = 4000;
            double minFeatureSize = ((double) genome.getLength()) / (maxScreenSize * locationUnit);

            long offset = 0;
            wholeGenomeScores = new ArrayList(1000);
            for (String chr : genome.getChromosomeNames()) {
                List<LocusScore> chrSegments = getSegments(heading, chr);
                if (chrSegments != null) {
                    int lastgEnd = -1;
                    for (LocusScore score : chrSegments) {
                        Segment seg = (Segment) score;
                        int gStart = (int) ((offset + score.getStart()) / locationUnit);
                        int gEnd = (int) ((offset + score.getEnd()) / locationUnit);
                        if ((gEnd - gStart) > minFeatureSize) {
                            wholeGenomeScores.add(new Segment(gStart, gStart, gEnd,
                                    gEnd, seg.getScore(), seg.getSnpCount()));
                        }
                    }

                }
                offset += genome.getChromosome(chr).getLength();
            }
            wholeGenomeScoresCache.put(heading, wholeGenomeScores);
        }
        return wholeGenomeScores;

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getTrackType() {
        return trackType;
    }

    /**
     * Method description
     *
     *
     * @param trackType
     */
    public void setTrackType(TrackType trackType) {
        this.trackType = trackType;
    }

 
}
